import logging
import json
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


class MarketOrderServiceResource(sdk2.Resource):
    releasers = ["dmpolyakov"]
    releasable = True
    share = True  # (create rbtorrent) resource
    test_attr = sdk2.Attributes.Integer("Some test attribute")


class Price:
    def __init__(self, currency, value):
        self.currency = currency
        self.value = value


class ServiceSupplier:
    def __init__(self, id, name):
        self.id = id
        self.name = name


class ServiceOfferFilter:
    def __init__(self, hid, vendor, sku):
        self.hid = hid
        self.vendor = vendor
        self.sku = sku


class MarketAccessServiceModel:
    def __init__(self, service_id, title, description, price, service_supplier, filter, rid, ya_service_id):
        self.service_id = service_id
        self.title = title
        self.description = description
        self.price = price
        self.service_supplier = service_supplier
        self.filter = filter
        self.rid = rid
        self.ya_service_id = ya_service_id


class PartnerWithService:
    def __init__(self, partner_id, partner_title, service_id):
        self.partner_id = partner_id
        self.partner_title = partner_title
        self.service_id = service_id


class CatalogServiceItem:
    def __init__(self, id, title, description, price, currency, hid, vendor_id, category_id, sku_id, region_id, ya_service_id):
        self.id = id
        self.title = title
        self.description = description
        self.price = price
        self.currency = currency
        self.hid = hid
        self.vendor_id = vendor_id
        self.category_id = category_id
        self.sku_id = sku_id
        self.region_id = region_id
        self.ya_service_id = ya_service_id


class MarketAccessServicesList:
    def __init__(self, services, dsbs_shops_with_services, skus_by_service_id):
        self.services = services
        self.dsbs_shops_with_services = dsbs_shops_with_services
        self.skus_by_service_id = skus_by_service_id


class MarketPartnerItemServiceUpdateTask(sdk2.Task):

    class Parameters(sdk2.Parameters):
        production = sdk2.parameters.Bool("Use production tables and endpoins", default=False)

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('yandex-yt'),
            PipEnvironment('requests'),
        ]

    def on_execute(self):
        import yt.wrapper as yt
        production = self.Parameters.production
        token_yt = sdk2.yav.Secret("sec-01fe1bgqddwzj2rw5dtph1ms9w").data()["token_yt"]
        yt.config["proxy"]["url"] = "hahn"
        yt.config["token"] = token_yt

        catalog_service_items = self.get_service_catalog_from_yt(yt, production)
        partner_with_services_items = self.get_partner_with_services_from_yt(token_yt)
        dsbs_shops = self.get_dsbs_shops_ids_from_yt(yt, production)
        skus_by_services = self.get_skus_by_services(yt, production)
        text_for_market_access = self.get_text_for_market_access_file(catalog_service_items,
                                                                      partner_with_services_items,
                                                                      dsbs_shops,
                                                                      skus_by_services)
        rbtorrent = self.create_sandbox_text_file_resource(text_for_market_access)
        self.update_file_in_market_access(rbtorrent, production)

    def get_service_catalog_from_yt(self, yt_wrapper, production):
        if production:
            services_table = yt_wrapper.TablePath('//home/market/users/dmpolyakov/services/ExternalServices')
            filters_table = yt_wrapper.TablePath('//home/market/users/dmpolyakov/services/ServiceToSkuFilter')
        else:
            services_table = yt_wrapper.TablePath('//home/market/users/dmpolyakov/services/test/ExternalServices')
            filters_table = yt_wrapper.TablePath('//home/market/users/dmpolyakov/services/test/ServiceToSkuFilter')

        yt_wrapper.run_sort(services_table, sort_by=['service_id'])
        yt_wrapper.run_sort(filters_table, sort_by=['service_id'])

        services = yt_wrapper.read_table(services_table, format='json')
        filters = list(yt_wrapper.read_table(filters_table, format='json'))

        catalog_service_items = []
        filter_index = 0
        for service in services:
            logging.info("Filter index: " + str(filter_index))
            if filter_index < len(filters) and service['service_id'] == filters[filter_index]['service_id']:
                while filter_index < len(filters) and service['service_id'] == filters[filter_index]['service_id']:
                    item = CatalogServiceItem(
                        id=service["service_id"],
                        title=service["title"],
                        description=service["description"],
                        price=service["price"],
                        currency=service["currency"],
                        hid=filters[filter_index]['hid'],
                        vendor_id=filters[filter_index]['vendor_id'],
                        category_id=filters[filter_index]['category_id'],
                        sku_id=filters[filter_index]['sku_id'],
                        region_id=service['region_id'],
                        ya_service_id=service['ya_service_id']
                    )
                    catalog_service_items.append(item)
                    filter_index += 1
            else:
                item = CatalogServiceItem(
                    id=service["service_id"],
                    title=service["title"],
                    description=service["description"],
                    price=service["price"],
                    currency=service["currency"],
                    hid=None,
                    vendor_id=None,
                    category_id=None,
                    sku_id=None,
                    region_id=service['region_id'],
                    ya_service_id=service['ya_service_id']
                )
                catalog_service_items.append(item)
        logging.info("Success got " + str(len(catalog_service_items)) + " service catalog items")
        return catalog_service_items

    def get_skus_by_services(self, yt_wrapper, production):
        if production:
            skus_by_service_id = yt_wrapper.TablePath('//home/market/users/dmpolyakov/services/services_to_skus')
        else:
            skus_by_service_id = yt_wrapper.TablePath('//home/market/users/dmpolyakov/services/test/services_to_skus')

        return {i['service_id']: i['sku'] for i in yt_wrapper.read_table(skus_by_service_id, format='json')}

    def get_dsbs_shops_ids_from_yt(self, yt_wrapper, production):
        if production:
            dsbs_shops_table = yt_wrapper.TablePath('//home/market/users/dmpolyakov/services/dbs_suppliers')
        else:
            dsbs_shops_table = yt_wrapper.TablePath('//home/market/users/dmpolyakov/services/test/dbs_suppliers')

        dsbs_shops = yt_wrapper.read_table(dsbs_shops_table, format='json')
        dsbs_shops_ids = map(lambda shop: shop['supplier_id'], dsbs_shops)

        return dsbs_shops_ids

    def get_partner_with_services_from_yt(self, token_yt):
        res = []
        res.append(PartnerWithService(partner_id=1000, partner_title="JavaBusters", service_id=10100))
        return res

    def get_text_for_market_access_file(self, catalog_service_items, partner_with_services_items, dsbs_shops, skus_by_service_id):
        res = []  # List<MarketAccessServiceModel>
        for partner_with_service in partner_with_services_items:
            for catalog_service_item in catalog_service_items:
                market_access_item = self.create_market_access_service_model(partner_with_service, catalog_service_item)
                res.append(market_access_item)

        body = MarketAccessServicesList(
            services=res,
            dsbs_shops_with_services=dsbs_shops,
            skus_by_service_id=skus_by_service_id,
        )

        res_json_temp = json.dumps(body, default=self.obj_dict)
        res_dict_temp = json.loads(res_json_temp)
        res_dict = self.clear_none(res_dict_temp)
        res_json = json.dumps(res_dict)
        logging.info("MarketAccess data: " + res_json)
        return res_json

    def create_sandbox_text_file_resource(self, text):
        resource = MarketOrderServiceResource(self, "Market order services resource", "market_order_services.json")
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.write_bytes(text)
        resource_data.ready()
        logging.info(resource)
        rbtorrent = resource.skynet_id
        logging.info("Sandbox resource creating successful: " + rbtorrent)
        return rbtorrent

    def update_file_in_market_access(self, rbtorrent, production):
        from .MarketAccessClient import MarketAccessClient

        resource_name = "market_partner_order_services"

        secret_id = "sec-01fe1bgqddwzj2rw5dtph1ms9w"
        secret = sdk2.yav.Secret(secret_id).data()

        if production:
            self_tvm_id = int(secret["self_tvm_id"])
            self_tvm_secret = secret["self_tvm_secret"]
            dst_tvm_id = int(secret["dst_tvm_id"])
            access_client = MarketAccessClient('https://access.vs.market.yandex.net', logging)
            tvm_ticket = access_client.get_tvm_ticket(self_tvm_id, self_tvm_secret, dst_tvm_id)
        else:
            access_client = MarketAccessClient('https://access.tst.vs.market.yandex.net', logging)
            tvm_ticket = secret['tvm_ticket']

        access_client.create_version(resource_name, rbtorrent, tvm_ticket)
        logging.info("Update file in market access is successful")

    def create_market_access_service_model(self, partner_link, catalog_item):
        price = Price(
            currency=catalog_item.currency,
            value=catalog_item.price,
        )
        service_supplier = ServiceSupplier(
            id=partner_link.partner_id,
            name=partner_link.partner_title,
        )
        service_offer_filter = ServiceOfferFilter(
            hid=catalog_item.hid,
            vendor=catalog_item.vendor_id,
            sku=catalog_item.sku_id,
        )
        res = MarketAccessServiceModel(
            service_id=catalog_item.id,
            title=catalog_item.title,
            description=catalog_item.description,
            price=price,
            service_supplier=service_supplier,
            filter=service_offer_filter,
            rid=catalog_item.region_id,
            ya_service_id=catalog_item.ya_service_id
        )
        return res

    def obj_dict(self, obj):
        return obj.__dict__

    def clear_none(self, obj):
        if type(obj) is list:
            res = self._filter_none_in_list(obj)
        elif type(obj) is dict:
            res = self._filter_none_in_dict(obj)
        return res

    def _filter_none_in_list(self, input_list):
        res = []
        for value in input_list:
            if value is None:
                continue
            if type(value) is dict or type(value) is list:
                res.append(self.clear_none(value))
            else:
                res.append(value)
        return res

    def _filter_none_in_dict(self, input_dict):
        res = {}
        for key, value in input_dict.items():
            if value is None:
                continue
            if type(value) is dict or type(value) is list:
                res[key] = self.clear_none(value)
            else:
                res[key] = value
        return res
