import json
import logging
import os
import shutil
from os.path import expanduser

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.resource_types import OTHER_RESOURCE as OtherResource
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git

POSTAMAT_GRADLE_RESOURCE_NAME = 'postamat-gradle-cache-dir'

logger = logging.getLogger(__name__)


class PostamatGradleTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    repo_dir = 'postamat-repo'

    class Requirements(sdk2.Task.Requirements):
        environments = (
            SandboxJavaJdkEnvironment('11.0.2'),
        )
        client_tags = ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)
        with sdk2.parameters.Group('Git Settings') as git_block:
            git_url = sdk2.parameters.String("SSH Git repository", required=True)
            ssh_secret = sdk2.parameters.YavSecret("Secret with ssh-key", default="sec-01ftg0cp40y3yakykc8khwrxkz")
            branch = sdk2.parameters.String("Branch", required=False)
            commit = sdk2.parameters.String("Commit", required=False)
            filter_branches = sdk2.parameters.Bool("Limit to refs/heads, refs/tags", default=True)

        with sdk2.parameters.Group('Build Settings') as build_block:
            build_id = sdk2.parameters.String("Build ID", default="0")
            project_dir = sdk2.parameters.String("Project dir", default="backoffice")
            tasks = sdk2.parameters.List("Gradle tasks", required=True, default=['clean'])
            parallel_execution = sdk2.parameters.Bool("Allow Gradle parallel execution", default=False)
            excluded_tasks = sdk2.parameters.List("Excluded Gradle tasks", required=False)
            sandbox_resources = sdk2.parameters.Dict("Required sandbox resources", required=False)
            required_resources = sdk2.parameters.Dict("Required sandbox resources", required=False)
            result_resources = sdk2.parameters.List("Result sandbox resources", required=False,
                                                    value_type=sdk2.parameters.String)

    class Context(sdk2.Task.Context):
        execution_number = 0

    def on_execute(self):
        self.Context.execution_number += 1
        logger.info("Clone git repository {repository}".format(repository=self.Parameters.git_url))
        ssh_key = self.Parameters.ssh_secret.data()['private-key']
        if os.path.exists(self.repo_dir):
            shutil.rmtree(self.repo_dir)
        with sdk2.ssh.Key(self, private_part=ssh_key):
            git = Git(self.Parameters.git_url, filter_branches=self.Parameters.filter_branches)
            git.clone(target_dir=self.repo_dir, branch=self.Parameters.branch, commit=self.Parameters.commit)
            # git.clone(target_dir=self.repo_dir, branch=self.Parameters.branch)
        try:
            self._do_build()
            self._publish_resources()
        except Exception as e:
            logger.error(e)
            raise
        finally:
            self._collect_and_publish_reports()

    def _do_build(self):
        self._restore_gradle_cache()
        self._request_resources()

        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            cmd = ['./gradlew',
                   '-Dorg.gradle.jvmargs=-Xms3g -Xmx3g',
                   '-Pbuild.number={}'.format(self.Parameters.build_id),
                   '--stacktrace']

            if self.Parameters.parallel_execution:
                cmd += ['--parallel']

            for task in self.Parameters.tasks:
                cmd += [task]

            for excluded_task in self.Parameters.excluded_tasks:
                if len(excluded_task.strip()) > 0:
                    cmd += ['-x', excluded_task]

            cwd = str(os.path.join(self.repo_dir, self.Parameters.project_dir))
            sp.check_call(cmd, cwd=cwd, stdout=pl.stdout, stderr=pl.stderr)
            self.set_info('Modules have been built')

        self._save_gradle_cache()

    def _request_resources(self):
        logger.info('Requesting resources')
        required_resources = self.Parameters.sandbox_resources or self.Parameters.required_resources or []
        for local_path in required_resources:
            sandbox_id = str(required_resources[local_path])
            logger.info("Requesting resource: '{path}' => '{sandbox_id}'".format(path=local_path, sandbox_id=sandbox_id))
            if sandbox_id.startswith('sbr:'):
                binary_resource = sdk2.Resource.find(id=sandbox_id.replace('sbr:', '', 1)).first()
                if binary_resource:
                    data = sdk2.ResourceData(binary_resource)
                    resource_path = str(data.path)
                    sp.call(['ln', '-s', resource_path, local_path])
                else:
                    logger.warn("Resource not found")
            else:
                logger.warn("Resource id unsupported")
        logger.info("Resources requested")

    def _publish_resource(self, type, name, description, ttl, src_path, dst_path=None):
        logger.info("Publishing resource '{}'".format(src_path))
        if dst_path:
            shutil.copytree(src_path, dst_path)
            res_path = dst_path
        else:
            res_path = src_path
        res_class = sdk2.Resource[type]
        resource = res_class(self, description, res_path, resource_name=name)
        resource.ttl = ttl
        data = sdk2.ResourceData(resource)
        data.ready()
        logger.info("Resource published")

    def _publish_resources(self):
        logger.info("Publishing resources")
        resources = map(lambda v: json.loads(v), self.Parameters.result_resources) or [
            {"type": "MARKET_POSTAMAT_TMS",
             "src_path": "postamat-repo/gates/eventgate/build/dist/postamat-tms.tar.gz",
             "name": "postamat-tms",
             "description": "Postamat tms application",
             "ttl": 1}
        ]
        logger.info("@resources: " + str(resources))
        for resource in resources:
            self._publish_resource(
                resource['type'],
                resource['name'],
                resource['description'],
                resource['ttl'],
                resource['src_path'],
                resource['dst_path'] if 'dst_path' in resource else None)
        logger.info("Resources published")

    def _restore_gradle_cache(self):
        binary_resource = sdk2.Resource.find(
            type=OtherResource,
            attrs={"resource_name": POSTAMAT_GRADLE_RESOURCE_NAME},
            state=ctr.State.READY
        ).order(-sdk2.Resource.id).first()
        if binary_resource:
            data = sdk2.ResourceData(binary_resource)
            path = str(data.path)
            dst_path = str(os.path.join(expanduser("~"), '.gradle'))
            shutil.copytree(path, dst_path)
            sp.call(['chmod', 'a+w', '-R', dst_path])

    def _save_gradle_cache(self):
        src_path = str(os.path.join(expanduser("~"), '.gradle'))
        self._publish_resource(
            'OTHER_RESOURCE',
            POSTAMAT_GRADLE_RESOURCE_NAME,
            'Postamat gradle cache dir',
            1,
            src_path,
            'gradle-cache')

    def _collect_and_publish_reports(self):
        reports_path = "build_reports_{}".format(self.Context.execution_number)
        src_path = str(os.path.join(self.repo_dir, self.Parameters.project_dir, 'build', 'reports'))
        if os.path.exists(src_path):
            self._publish_resource('BUILD_OUTPUT', 'build-reports', 'Build reports', 7, src_path, reports_path)
        else:
            logger.warn("Resource '{}' not found".format(src_path))
