import json
import logging
import os
import shlex
import shutil

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git

logger = logging.getLogger(__name__)


class PostamatJava11Task(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            SandboxJavaJdkEnvironment('11.0.2'),
        )
        client_tags = ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Parameters):
        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Git Settings') as git_block:
            git_url = sdk2.parameters.String("SSH Git repository", required=True)
            git_secret = sdk2.parameters.YavSecret(
                "Secret with private ssh key (require 'private-key' secret key)",
                required=True)
            git_branch = sdk2.parameters.String("Branch", required=False)
            git_commit = sdk2.parameters.String("Commit", required=False)
            git_filter_branches = sdk2.parameters.Bool("Limit to refs/heads, refs/tags", default=True)
            git_repo_dir = sdk2.parameters.String("Repository directory", required=True)

        with sdk2.parameters.Group('Input resources') as input_resources_block:
            input_resources = sdk2.parameters.Dict("Input sandbox resources", required=False)

        with sdk2.parameters.Group('Command line') as command_block:
            command_secret = sdk2.parameters.YavSecret("Secret environment variables", required=False)
            command_working_dir = sdk2.parameters.String("Command working directory", required=True)
            command_args = sdk2.parameters.String("Command", required=True)

        with sdk2.parameters.Group('Output resources') as output_resources_block:
            output_resources = sdk2.parameters.List(
                "Output sandbox resources",
                required=False,
                value_type=sdk2.parameters.String)

        custom_params = sdk2.parameters.JSON("Custom parameters", required=False)

    class Context(sdk2.Task.Context):
        execution_number = 0

    def on_execute(self):
        self.Context.execution_number += 1
        self._clone_git(self.Parameters)
        self._download_resources(self.Parameters)
        self._execute_command(self.Parameters)
        self._publish_resources(self.Parameters)

    def _clone_git(self, params):
        logger.info("Cloning git repository {repo} to {dir}".format(repo=params.git_url, dir=params.git_repo_dir))
        ssh_key = params.git_secret.data()['private-key']
        if os.path.exists(params.git_repo_dir):
            shutil.rmtree(params.git_repo_dir)
        with sdk2.ssh.Key(self, private_part=ssh_key):
            git = Git(params.git_url, filter_branches=params.git_filter_branches)
            git.clone(target_dir=params.git_repo_dir, branch=params.git_branch, commit=params.git_commit)

    def _download_resources(self, params):
        logger.info('Downloading resources')
        resources = params.input_resources or []
        for local_path in resources:
            sandbox_id = str(resources[local_path])
            logger.info(
                "Downloading resource: '{path}' => '{sandbox_id}'".format(path=local_path, sandbox_id=sandbox_id))
            if sandbox_id.startswith('sbr:'):
                binary_resource = sdk2.Resource.find(id=sandbox_id.replace('sbr:', '', 1)).first()
                if binary_resource:
                    data = sdk2.ResourceData(binary_resource)
                    resource_path = str(data.path)
                    sp.call(['ln', '-s', resource_path, local_path])
                else:
                    logger.warn("Resource not found")
            else:
                logger.warn("Resource id unsupported")
        logger.info("Resources downloaded")

    def _execute_command(self, params):
        with sdk2.helpers.ProcessLog(self, logger='execute') as pl:
            secret = params.command_secret
            env_vars = os.environ.copy()
            if secret:
                data = secret.data()
                if data:
                    for key in data:
                        env_vars[key] = data[key]
            cmd = shlex.split(params.command_args)
            sp.check_call(cmd, cwd=params.command_working_dir, env=env_vars, stdout=pl.stdout, stderr=pl.stderr)
            self.set_info('Command executed')

    def _publish_resources(self, params):
        logger.info("Publishing resources")
        resources = map(lambda v: json.loads(v), params.output_resources or [])
        for resource in resources:
            self._publish_resource(
                resource['type'],
                resource['name'],
                resource['description'],
                resource['ttl'],
                resource['src_path'],
                resource['dst_path'] if 'dst_path' in resource else None)
        logger.info("Resources published")

    def _publish_resource(self, type, name, description, ttl, src_path, dst_path=None):
        logger.info("Publishing resource '{}'".format(src_path))
        if dst_path:
            shutil.copytree(src_path, dst_path)
            res_path = dst_path
        else:
            res_path = src_path
        res_class = sdk2.Resource[type]
        resource = res_class(self, description, res_path, resource_name=name)
        resource.ttl = ttl
        data = sdk2.ResourceData(resource)
        data.ready()
        logger.info("Resource published")
