# -*- coding: utf-8 -*-

import os
import logging
import distutils

from sandbox.projects.common.nanny import nanny
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox import sdk2
from sandbox.projects.market.recom.dj.resources import MARKET_DJ_FORMULAS_BUNDLE


class DonloaderFromYt:
    def __init__(self, yt_proxy, yt_token):
        import yt.wrapper
        self.ytc = yt.wrapper.YtClient(proxy=yt_proxy, token=yt_token)

    def download_directory(self, src, dst):
        logging.info('Download directory %s from YT to %s' % (src, dst))
        logging.info("Create dir %s" % dst)
        distutils.dir_util.mkpath(dst)

        count = self.ytc.get(src + '/@count')
        logging.info("Total files in directory %s: %s" % (src, count))

        for node in self.ytc.list(src):
            self.download_file(os.path.join(src, node), os.path.join(dst, node))

    def download_file(self, src, dst):
        logging.info('Download file %s from YT to %s' % (src, dst))
        with open(dst, 'wb') as f:
            for chunk in self.ytc.read_file(src).chunk_iter():
                f.write(chunk)


class MAKE_MARKET_DJ_FORMULAS_BUNDLE(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("YT parameters"):
            yt_proxy = sdk2.parameters.String("YT_PROXY", required=True, default="hahn")
            yt_token = sdk2.parameters.String("YT_TOKEN", required=True)
            yt_path = sdk2.parameters.String("YT_PATH", required=True)

        dir_name = sdk2.parameters.String("Destination directory name", required=True)

        use_last_binary = sdk2.parameters.Bool('Use last binary archive', default=True)
        with use_last_binary.value[True]:
            with sdk2.parameters.RadioGroup("Binary release type") as binary_release_type:
                binary_release_type.values.stable = binary_release_type.Value('stable', default=True)
                binary_release_type.values.test = binary_release_type.Value('test')

    def on_execute(self):
        MARKET_DJ_FORMULAS_BUNDLE(self, 'Market DJ formulas: {}'.format(self.Parameters.description), self.Parameters.dir_name)
        super(MAKE_MARKET_DJ_FORMULAS_BUNDLE, self).on_execute()

        yt_token = sdk2.Vault.data(self.Parameters.owner, self.Parameters.yt_token)
        downloader = DonloaderFromYt(self.Parameters.yt_proxy, yt_token)
        downloader.download_directory(self.Parameters.yt_path, self.Parameters.dir_name)

    def on_save(self):
        if self.Parameters.use_last_binary:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={'task_type': self.type.name, 'release': self.Parameters.binary_release_type or 'stable'}
            ).first().id
        else:
            self.Requirements.tasks_resource = None
