# coding=utf-8

import logging
import os
import tarfile
from sandbox.projects.common import context_managers
from sandbox import sdk2


class MarketSvnData(sdk2.Resource):
    """ A market report svn-data resource """


class CreateResourceFromArcadia(sdk2.Task):
    """Create sandbox resource from arcadia url
    """

    _KEY_OWNER = 'MARKET'

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.String(
            'Url on data in arcadia',
            description='Example: arcadia:/arc/trunk/arcadia/market/svn-data',
            required=True
        )
        revision = sdk2.parameters.String(
            'Revision',
            description='If empty then HEAD',
            required=False
        )
        resource_type = sdk2.parameters.String(
            'Resource type to be created',
            required=True,
            default_value='OTHER_RESOURCE'
        )
        ttl = sdk2.parameters.String(
            'Time to live',
            required=False,
            default_value=14
        )

        with sdk2.parameters.Output:
            resource = sdk2.parameters.Resource("Resource", required=True)
            skynet_id = sdk2.parameters.String("Skynet ID", required=True)

    def on_execute(self):
        logging.info("on_execute")

        data_path = str(self.path('data'))
        arcadia_url = str(self.Parameters.arcadia_url)
        tar_file = str(self.path('data.tar.gz'))

        logging.info('data_path: {}'.format(data_path))
        logging.info('arcadia_url: {}'.format(arcadia_url))
        logging.info('tar_file: {}'.format(tar_file))

        sdk2.svn.Arcadia.checkout(
            url=arcadia_url,
            path=data_path,
            revision=self.Parameters.revision)

        def filter(tarinfo):
            excluded = [
                '.svn'
            ]
            if os.path.basename(tarinfo.name) in excluded:
                return None
            return tarinfo

        with tarfile.open(tar_file, 'w:gz') as tar:
            os.chdir(data_path)
            for path in os.listdir(data_path):
                tar.add(path, filter=filter)
            os.chdir(str(self.path()))

        resource_description = "Resource from {}. Revision: {}".format(
            arcadia_url,
            self.Parameters.revision
        )
        attrs = {}
        if self.Parameters.ttl:
            attrs['ttl'] = self.Parameters.ttl
        resource = sdk2.Resource[self.Parameters.resource_type](
            self,
            resource_description,
            tar_file,
            arch=None,
            system_attributes=None,
            **attrs)
        sdk2.ResourceData(resource).ready()

        logging.info("Resource id: {}".format(resource.id))
        logging.info("Resource skynet_id: {}".format(resource.skynet_id))

        self.Parameters.resource = resource
        self.Parameters.skynet_id = resource.skynet_id
