# -*- coding: utf-8 -*-

import logging
import requests
import time

import sandbox.sdk2 as sdk2
from sandbox.projects.common.binary_task import deprecated as binary_task

from datetime import datetime


class FlagsJsonChangeEventSender(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        vault_owner = sdk2.parameters.String(
            'Owner of vault secrets',
            required=True,
        )

        vault_token = sdk2.parameters.String(
            'Name of ABT token secret',
            required=True,
        )

    @property
    def abt_headers(self):
        if not hasattr(self, '__headers'):
            abt_token = sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.vault_token)
            self.__headers = {
                'X-Ab-Strict-Auth': 'true',
                'Authorization': 'OAuth {}'.format(abt_token),
            }
        return self.__headers

    def _request_all_tests(self):
        return requests.request(
            'GET',
            'https://ab.yandex-team.ru/deploying/flags.json/production/content',
            headers=self.abt_headers,
        ).json()

    def _request_test(self, testid):
        return requests.request(
            'GET',
            'https://ab.yandex-team.ru/api/testid/{}'.format(testid),
            headers=self.abt_headers,
        ).json()

    def on_execute(self):
        from market.pylibrary.tsum_events.tsum_events import Event, EventSender, EventStatus, MicroEvent

        response = self._request_all_tests()
        sender = EventSender()
        for experiment in response['content']:
            for context in experiment['contexts']:
                flags = context['CONTEXT'].get('MAIN', {}).get('MARKET', {}).get('rearr-factors', [])
                flags_string = ','.join(flags)
                if 'market_' not in flags_string:
                    continue

                for testid in context['TESTID']:
                    testid_response = self._request_test(testid)
                    event_time = datetime.strptime(testid_response['time_modified'], "%Y-%m-%dT%H:%M:%S.%f")

                    event_seconds = int(time.mktime(event_time.timetuple()))
                    event_id = "{}_{}".format(testid, event_seconds)
                    event_title = '{}:{}'.format(testid_response['ticket'], flags_string)

                    tags = ['market', 'exp:flags_json', 'exp:test:{}'.format(testid)]
                    log_ts = datetime.fromtimestamp(event_seconds).strftime('%Y-%m-%d %H:%M:%S')
                    logging.info('Sending event with id: "{}" and tags: "{}", event date: "{}"'.format(event_id, tags, log_ts))
                    event = Event(
                        id=event_id,
                        startTimeSeconds=event_seconds,
                        endTimeSeconds=event_seconds,
                        title=event_title,
                        text=testid_response['title'],
                        type='exp',
                        project='market',
                        status=EventStatus.Value('INFO'),
                        tags=tags
                    )

                    sender.send_events(event, timeout=60.0)
