#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import datetime
import functools
import httplib
import itertools
import json
import logging
import os
import time

AB_API_SERVER = 'ab.yandex-team.ru'
AB_REQUEST_TIMEOUT_S = 60

MAIN_REPORT = 0
PARALLEL_REPORT = 1
BLUE_REPORT = 2
APPS_REPORT = 3
REPORT_TYPES = (MAIN_REPORT, PARALLEL_REPORT, BLUE_REPORT, APPS_REPORT)

BLUE_ENVS = ('MARKET_BLUE', 'DESKTOP_BLUE', 'TOUCH_BLUE')
APPS_ENVS = ('MARKETAPPS',)
WHITE_ENVS = ('DESKTOP', 'TOUCH', 'MARKET')
ALL_ENVS = set(BLUE_ENVS + APPS_ENVS + WHITE_ENVS)

REPORT_TYPE_BY_QUEUE_ID = {
    7: MAIN_REPORT,
    1: PARALLEL_REPORT
}

DAYS = 14

GRAPH_WIDTH = 12
GRAPH_HEIGHT = 15
ROW_WIDTH = 24

GRAFANA_UID = 'yiaBxTBmz'


def to_str(s):
    return s.encode('utf-8') if isinstance(s, unicode) else s


def retry(max_try_count=10, initial_delay=2, exp_backoff=2):

    def retry_decorator(function):

        @functools.wraps(function)
        def retry_wrapper(*args, **kwargs):
            try_count = 0
            retry_delay = initial_delay
            while True:
                try:
                    return function(*args, **kwargs)
                except Exception:
                    try_count += 1
                    if try_count == max_try_count:
                        raise
                    logging.exception('Execution failed %d times, retrying...', try_count)
                    time.sleep(retry_delay)
                    retry_delay *= exp_backoff
        return retry_wrapper

    return retry_decorator


@functools.total_ordering
class TestInfo(object):
    def __init__(self, test_id, title='', ticket='', ticket_title='', rearr_list=list()):
        self.test_id = test_id
        self.title = title
        self.ticket = ticket
        self.ticket_title = ticket_title
        self.rearr_list = rearr_list

    def __repr__(self):
        return '''TestInfo('{}', '{}', '{}', {})'''.format(
            self.test_id, self.title, self.ticket, self.rearr_list
        )

    def __eq__(self, other):
        return self.test_id == other.test_id

    def __lt__(self, other):
        return self.test_id < other.test_id

    def __hash__(self):
        return hash(self.test_id)


class MetricInfo(object):
    def __init__(self, expr, title, unit, report_type, smooth, panel_id):
        self.expr = expr
        self.title = title
        self.unit = unit
        self.report_type = report_type
        self.smooth = smooth
        self.panel_id = panel_id


class ABRequester(object):
    def __init__(self, auth_token):
        self.__headers = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'X-Ab-Strict-Auth': 'true',
            'Authorization': 'OAuth ' + auth_token,
        }

    @retry()
    def get(self, query):
        logging.info('%s', query)
        conn = httplib.HTTPSConnection(AB_API_SERVER, timeout=AB_REQUEST_TIMEOUT_S)
        conn.connect()
        try:
            conn.request('GET', query, headers=self.__headers)
            response = conn.getresponse()
            response_data = response.read()
            if response.status != 200:
                raise Exception('AB returned error: {}\n{}'.format(response.status, response_data))
        finally:
            conn.close()
        return json.loads(response_data)


RPS_METRICS = (
    MetricInfo(
        "{{'project'='market-report', 'cluster'='stable', 'service'='market-meta-report', 'test_bucket'=~'$exp', 'test_bucket'='{test_id}', 'sensor'='rps', 'period'='one_min'}}",
        'Main Report RPS',
        'rps',
        MAIN_REPORT,
        False,
        1,
    ),
    MetricInfo(
        "{{'project'='market-report', 'cluster'='stable', 'service'='market-meta-report', 'test_bucket'=~'$exp', 'test_bucket'='{test_id}', 'sensor'='rps', 'period'='one_min'}}",
        'Apps Report RPS',
        'rps',
        APPS_REPORT,
        False,
        2,
    ),
)

OTHER_METRICS = (
    MetricInfo(
        "{{'project'='market-report', 'cluster'='stable', 'service'='market-meta-report', 'test_bucket'=~'$exp', 'test_bucket'='{test_id}', 'sensor'='cpu_time_us', 'period'='one_min', 'quantile'='0.99'}}/1000",
        'Main CPU Time',
        'ms',
        MAIN_REPORT,
        False,
        3,
    ),
    MetricInfo(
        "{{'project'='market-report', 'cluster'='stable', 'service'='market-meta-report', 'test_bucket'=~'$exp', 'test_bucket'='{test_id}', 'sensor'='cpu_time_us', 'period'='one_min', 'quantile'='0.99'}}/1000",
        'Apps CPU Time',
        'ms',
        APPS_REPORT,
        False,
        4,
    ),
    MetricInfo(
        "{{'project'='market-report', 'cluster'='stable', 'service'='market-meta-report', 'test_bucket'=~'$exp', 'test_bucket'='{test_id}', 'sensor'='timing', 'period'='one_min', 'quantile'='0.99'}}",
        'Main Meta Report Timings',
        'ms',
        MAIN_REPORT,
        False,
        5,
    ),
    MetricInfo(
        "{{'project'='market-report', 'cluster'='stable', 'service'='market-meta-report', 'test_bucket'=~'$exp', 'test_bucket'='{test_id}', 'sensor'='timing', 'period'='one_min', 'quantile'='0.99'}}",
        'Apps Meta Report Timings',
        'ms',
        APPS_REPORT,
        False,
        6,
    ),
    MetricInfo(
        "{{'project'='market-report', 'cluster'='stable', 'service'='market-meta-report', 'test_bucket'=~'$exp', 'test_bucket'='{test_id}', 'sensor'='base_timing', 'period'='one_min', 'quantile'='0.99'}}",
        'Main Base Report Timings',
        'ms',
        MAIN_REPORT,
        False,
        7,
    ),
    MetricInfo(
        "{{'project'='market-report', 'cluster'='stable', 'service'='market-meta-report', 'test_bucket'=~'$exp', 'test_bucket'='{test_id}', 'sensor'='base_timing', 'period'='one_min', 'quantile'='0.99'}}",
        'Apps Base Report Timings',
        'ms',
        APPS_REPORT,
        False,
        8,
    ),
)


def get_active_test_ids(number_of_days, ab):
    end_date = datetime.date.today()
    start_date = end_date - datetime.timedelta(days=number_of_days)
    query = '/api/testid/activity?datestart={}&dateend={}'.format(
        start_date.strftime('%Y%m%d'),
        end_date.strftime('%Y%m%d')
    )
    ab_data = ab.get(query)
    test_ids = set()
    for testid_info in ab_data:
        test_ids.add(testid_info['testid'])
    logging.info('Found %d active test ids', len(test_ids))
    return test_ids


def find_env_entry_in_subtree(node):
    if isinstance(node, dict):
        for key, value in node.iteritems():
            if key in ALL_ENVS:
                if isinstance(value, dict):
                    rearr = value.get('rearr-factors') or value.get('rearr')
                    if rearr is not None:
                        return key, rearr
            else:
                env, rearr = find_env_entry_in_subtree(value)
                if env is not None:
                    return env, rearr
    return None, None


def get_report_and_rearr_by_context_queue(context, queue_id):
    env, rearr = find_env_entry_in_subtree(context)
    if env is not None and rearr is not None:
        if env in BLUE_ENVS:
            report_type = BLUE_REPORT
        elif env in APPS_ENVS:
            report_type = APPS_REPORT
        elif env in WHITE_ENVS:
            report_type = REPORT_TYPE_BY_QUEUE_ID.get(queue_id, PARALLEL_REPORT)
        else:
            report_type = None

        if report_type is not None:
            return report_type, rearr

    return None, None


def get_report_test_list_by_type(number_of_days, ab):
    test_list_by_type = {report: set() for report in REPORT_TYPES}
    test_buckets = get_active_test_ids(number_of_days, ab)
    MAX_IDS_TO_REQUEST = 100
    pos = 0
    while pos < len(test_buckets):
        test_buckets_part = itertools.islice(test_buckets, pos, pos + MAX_IDS_TO_REQUEST)
        test_id_query = '/api/testid/?form=full'
        for test_id in test_buckets_part:
            test_id_query += '&id={}'.format(test_id)
        pos += MAX_IDS_TO_REQUEST
        ab_data = ab.get(test_id_query)
        for ab_item in ab_data:
            ab_params = json.loads(ab_item['params'])
            if not isinstance(ab_params, list):
                continue
            queue_id = to_str(ab_item['queue_id'])
            for ab_param in ab_params:
                if 'CONTEXT' not in ab_param:
                    continue
                report_type, rearr = get_report_and_rearr_by_context_queue(ab_param['CONTEXT'], queue_id)
                if report_type is None:
                    continue
                testid = ab_item['testid']
                title = to_str(ab_item['title']).replace('"', '')
                ab_ticket_data = ab.get('/api/task/?testid={}&form=full'.format(testid))
                if ab_ticket_data:
                    ticket = to_str(ab_ticket_data[0]['ticket'])
                    ticket_title = to_str(ab_ticket_data[0]['title'])
                else:
                    ticket = ''
                    ticket_title = ''
                    logging.warn('Task not found for testid %s', testid)
                test_list_by_type[report_type].add(TestInfo(testid, title, ticket, ticket_title, rearr))

    # Ищем контрольные тестиды
    control_tests_by_type = {report: set() for report in REPORT_TYPES}
    for report_type in REPORT_TYPES:
        for test_info in test_list_by_type[report_type]:
            ab_data = ab.get('/api/task/?testid={}&form=full'.format(test_info.test_id))
            if not ab_data:
                logging.warn('Task not found for testid %s', test_info.test_id)
                continue
            for testid in ab_data[0]['testids']:
                if TestInfo(testid) not in test_list_by_type[report_type]:
                    if testid in test_buckets:
                        ticket = to_str(ab_data[0]['ticket'])
                        ticket_title = to_str(ab_data[0]['title'])
                        control_tests_by_type[report_type].add((testid, ticket, ticket_title))
                    else:
                        logging.warn('testid %s is not active', testid)

        # Получение описаний (title)
        test_info_list = list()
        if control_tests_by_type[report_type]:
            test_id_query = '/api/testid/?form=full'
            for test_id, _, _ in control_tests_by_type[report_type]:
                test_id_query += '&id={}'.format(test_id)
            ab_data = ab.get(test_id_query)
            title_by_test_id = dict()
            for ab_item in ab_data:
                test_id = ab_item['testid']
                title = to_str(ab_item['title']).replace('"', '')
                title_by_test_id[test_id] = title
            for test_id, ticket, ticket_title in control_tests_by_type[report_type]:
                test_info_list.append(TestInfo(test_id, title_by_test_id.get(test_id, ''), ticket, ticket_title))
        control_tests_by_type[report_type] = test_info_list

    for report_type in REPORT_TYPES:
        test_list_by_type[report_type] = list(test_list_by_type[report_type])

    REPORT_NAME_BY_TYPE = {
        MAIN_REPORT: 'main',
        PARALLEL_REPORT: 'parallel',
        BLUE_REPORT: 'blue',
        APPS_REPORT: 'apps'
    }
    for report_type, report_name in REPORT_NAME_BY_TYPE.iteritems():
        logging.info('Selected %d test ids for %s report', len(test_list_by_type[report_type]), report_name)
        for test_info in test_list_by_type[report_type]:
            logging.info('%s', repr(test_info))
        logging.info('Found %d control test ids for %s report', len(control_tests_by_type[report_type]), report_name)
        for test_info in control_tests_by_type[report_type]:
            logging.info('%s', repr(test_info))
    return test_list_by_type, control_tests_by_type


def make_graph(metric_info, test_list, x, y, w, h, st_links):
    if st_links:
        tickets = set((test_info.ticket, test_info.ticket_title) for test_info in test_list)
        links = [
            {
                'title': ticket + ' ' + title,
                'type': 'absolute',
                'url': 'https://st.yandex-team.ru/{ticket}'.format(ticket=ticket)
            }
            for ticket, title in sorted(tickets) if ticket
        ]
    else:
        test_ids = set((test_info.test_id, test_info.title) for test_info in test_list)
        links = [
            {
                'title': str(test_id) + ' ' + title,
                'type': 'absolute',
                'url': 'https://ab.yandex-team.ru/testid/{test_id}'.format(test_id=test_id)
            }
            for test_id, title in sorted(test_ids)
        ]

    def metric_target(metric_info, test_info):
        expr = 'group_by_time($__interval, \'avg\', drop_empty_lines({expr}))'.format(
            expr=metric_info.expr.format(test_id=test_info.test_id)
        )
        if metric_info.smooth:
            expr = 'movingAverage({}, "1h")'.format(expr)
        expr = 'alias({expr}, "{ticket} ({test_id}) {title}")'.format(
            expr=expr,
            ticket=test_info.ticket,
            test_id=test_info.test_id,
            title=test_info.title
        )
        return expr

    def gen_ref_id(index):
        ref_id = ''
        rang = ord('Z') - ord('A') + 1
        while True:
            ref_id = chr(ord('A') + index % rang) + ref_id
            index = index // rang
            if index == 0:
                break
        return ref_id

    graph = {
        'datasource': 'Solomon',
        'nullPointMode': 'connected',
        'fill': 0,
        'gridPos': {
            'h': h,
            'w': w,
            'x': x,
            'y': y
        },
        'id': metric_info.panel_id,
        'legend': {
            'show': True
        },
        'links': links,
        'targets': [
            {
                'refId': gen_ref_id(index),
                'target': metric_target(metric_info, test_info)
            }
            for index, test_info in enumerate(test_list)
        ],
        'title': metric_info.title,
        'tooltip': {
            'sort': 2
        },
        'type': 'graph',
        'yaxes': [
            {
                'format': metric_info.unit,
                'show': True
            }
            for _ in range(2)
        ],
    }
    return graph


def sort_by_ticket(test_info_list):
    return sorted(test_info_list, key=lambda test_info: (test_info.ticket, test_info.test_id))


def make_panels(test_list_by_report_type, control_testids_by_type):

    def advance_coords(x, y):
        x += GRAPH_WIDTH
        if x >= ROW_WIDTH:
            x = 0
            y += GRAPH_HEIGHT
        return x, y
    panels = list()
    x = 0
    y = 0
    for metric_info in RPS_METRICS:
        panels.append(make_graph(
            metric_info,
            sort_by_ticket(test_list_by_report_type[metric_info.report_type]),
            x,
            y,
            GRAPH_WIDTH,
            GRAPH_HEIGHT,
            True
        ))
        x, y = advance_coords(x, y)
    for metric_info in OTHER_METRICS:
        panels.append(make_graph(
            metric_info,
            sort_by_ticket(test_list_by_report_type[metric_info.report_type] + control_testids_by_type[metric_info.report_type]),
            x,
            y,
            GRAPH_WIDTH,
            GRAPH_HEIGHT,
            False
        ))
        x, y = advance_coords(x, y)
    return panels


def make_dashboard(number_of_days, auth_token):
    test_list_by_report_type, control_testids_by_type = get_report_test_list_by_type(number_of_days, ABRequester(auth_token))

    test_info_list = list()
    for i in test_list_by_report_type.values():
        test_info_list.extend(i)
    for i in control_testids_by_type.values():
        test_info_list.extend(i)
    test_id_filter_by_experiment = dict()
    for test_info in test_info_list:
        experiment_descr = test_info.ticket + ' ' + test_info.ticket_title
        if experiment_descr not in test_id_filter_by_experiment:
            test_id_filter_by_experiment[experiment_descr] = list()
        test_id_filter_by_experiment[experiment_descr].append(test_info.test_id)
    for experiment_descr in test_id_filter_by_experiment:
        test_id_filter_by_experiment[experiment_descr] = r'^({})$'.format('|'.join(
            [str(test_id) for test_id in test_id_filter_by_experiment[experiment_descr]]
        ))

    return {
        'annotations': {
            'list': [
                {
                    'datasource': 'market',
                    'enable': False,
                    'hide': False,
                    'iconColor': '#FADE2A',
                    'limit': 100,
                    'name': 'Flags.json',
                    'showIn': 0,
                    'tags': 'exp:flags_json',
                    'type': 'tags'
                },
                {
                    'datasource': 'market',
                    'enable': False,
                    'hide': False,
                    'iconColor': 'rgba(255, 96, 96, 1)',
                    'limit': 100,
                    'name': 'Стоп-кран',
                    'showIn': 0,
                    'tags': 'emergency_break',
                    'type': 'tags'
                }
            ]
        },
        'description': 'Активные эксперименты над Репортом',
        'graphTooltip': 1,
        'links': [
            {
                'icon': 'external link',
                'title': 'Sandbox Task Source Code',
                'type': 'link',
                'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/market/report/GenerateExperimentDashboard'
            },
            {
                'icon': 'external link',
                'title': 'Sandbox Scheduler',
                'type': 'link',
                'url': 'https://sandbox.yandex-team.ru/scheduler/23030'
            },
            {
                'icon': 'external link',
                'title': 'Solomon dashboard',
                'type': 'link',
                'url': 'https://monitoring.yandex-team.ru/projects/market-report/dashboards/monn295tt8gef5hcs2ge'
            },
        ],
        'panels': make_panels(test_list_by_report_type, control_testids_by_type),
        'refresh': '1m',
        'schemaVersion': 16,
        'tags': [
            'market_report',
            'market_report_global_metrics',
        ],
        'templating': {
            'list': [
                {
                    'allValue': '.*',
                    'current': {
                        'text': 'All',
                        'value': '$__all'
                    },
                    'includeAll': True,
                    'label': 'experiments',
                    'multi': True,
                    'name': 'exp',
                    'options': [
                        {
                            'selected': True,
                            'text': 'All',
                            'value': '$__all'
                        }
                    ] + [
                        {
                            'selected': False,
                            'text': experiment_descr,
                            'value': test_id_filter_by_experiment[experiment_descr]
                        }
                        for experiment_descr in sorted(test_id_filter_by_experiment.keys())
                    ],
                    'query': '.*,' + ','.join([test_id_filter_by_experiment[experiment_descr] for experiment_descr in sorted(test_id_filter_by_experiment.keys())]),
                    'type': 'custom'
                },
            ]
        },
        'time': {
            'from': 'now-{}d'.format(number_of_days + 1),
            'to': 'now'
        },
        'title': 'Report Experiments ({days} days) [autogenerated on {date}]'.format(
            days=number_of_days,
            date=datetime.datetime.now().strftime('%Y-%m-%d %H:%M')
        ),
        'uid': GRAFANA_UID
    }


@retry()
def publish_dashboard(dashboard, auth_token):
    headers = {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': 'OAuth ' + auth_token
    }
    body = {
        'dashboard': dashboard,
        'folderId': 0,
        'overwrite': True
    }
    conn = httplib.HTTPSConnection('grafana.yandex-team.ru', timeout=10)
    conn.connect()
    try:
        conn.request('DELETE', '/api/dashboards/uid/{uid}'.format(uid=GRAFANA_UID), headers=headers)
        conn.getresponse().read()
        conn.request('POST', '/api/dashboards/db', json.dumps(body), headers)
        response = conn.getresponse()
        response_data = response.read()
        if response.status != 200:
            raise Exception('Grafana API returned error: {}\n{}'.format(response.status, response_data))
    finally:
        conn.close()
    return response_data


def get_token(service_name, token_env_var):
    auth_token = os.environ.get(token_env_var)
    if auth_token:
        return auth_token
    token_file_path = os.path.expanduser('~/.robot-market-st.tokens')
    if not os.path.isfile(token_file_path):
        return None
    with open(token_file_path, 'r') as token_file:
        tokens = json.loads(token_file.read())
        return tokens.get(service_name)


def get_grafana_token():
    # yo can get it from https://oauth.yandex-team.ru/authorize?response_type=token&client_id=9590d99772e74c7386610688d4a5c1b3
    return get_token('grafana', 'GRAFANA_AUTH_TOKEN')


def get_ab_token():
    # you can get it from https://oauth.yandex-team.ru/authorize?response_type=token&client_id=b4e27037711d4c53a8f182cb7697e655
    return get_token('ab', 'AB_AUTH_TOKEN')


def main():
    arg_parser = argparse.ArgumentParser()
    arg_parser.add_argument('--publish', action='store_true')
    arg_parser.add_argument('--days', type=int, default=DAYS)
    args = arg_parser.parse_args()
    if args.publish:
        auth_token = get_grafana_token()
        if not auth_token:
            print 'No token'

    log_formatter = logging.Formatter('%(asctime)s [%(levelname)-5.5s]  %(message)s')
    root_logger = logging.getLogger()
    root_logger.setLevel(logging.INFO)

    console_handler = logging.StreamHandler()
    console_handler.setFormatter(log_formatter)
    root_logger.addHandler(console_handler)

    dashboard = make_dashboard(args.days, get_ab_token())

    if args.publish:
        print publish_dashboard(dashboard, auth_token)
    else:
        print json.dumps(dashboard, indent=2)


if __name__ == '__main__':
    main()
