#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import collections
import datetime
import functools
import httplib
import logging
import json
import os
import re
import time
from copy import deepcopy

ROW_WIDTH = 24
ROW_HEIGHT = 1
GRAPH_HEIGHT = 14
SHADOW_GRAPH_HEIGHT = 12
SMALL_HEIGHT = 10

UNLIMITED_TIME_VALUE = '100500'

REPORT_TYPES = {
    'market': {
        'title': 'Main Base',
        'report_table': 'main-report',
        'nginx_table': 'main-report-nginx',
        'error_table': 'main-report-common',
        'host_type': 'report',
        'sub_role': 'market',
        'DEFAULT_MAX_TIME': 1960,
        'MAX_TIME_VALUES': (195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': ('accessories', 'actual_delivery', 'also_viewed', 'alt_same_shop', 'attractive_models', 'best_model_reviews', 'bestdeals', 'better_price', 'book_now_incut', 'brand_products', 'brandbestdeals', 'categories_and_models_by_history', 'compare_products', 'deals', 'defaultoffer', 'delivery_status', 'geo', 'mainreport', 'model_modifications', 'modelinfo', 'multi_category', 'offerinfo', 'omm_electro', 'omm_findings', 'outlets', 'parallel', 'personal_categories', 'personal_offers', 'personal_recommendations', 'personalcategorymodels', 'popular_products', 'prime', 'print_doc', 'product_accessories', 'product_analogs', 'productoffers', 'products_by_history', 'products_by_history_ex', 'promo', 'promoted_categories', 'recipe_by_glfilters', 'recipes_by_doc', 'recipes_contain_glfilters', 'redoffers', 'shop_info', 'sku_search', 'top_categories', 'user_feed', 'vendor_offers_models', 'visualanalog', 'yellow_micro', 'yellow_msku', 'yellow_promo'),
        'GRAFANA_UID': 'L6I0nWLiz',
        'PER_HOST_GRAFANA_UID': 'F-lfg_Emz',
    },
    'meta-market': {
        'title': 'Main Meta',
        'report_table': 'meta-report',
        'nginx_table': 'meta-report-nginx',
        'error_table': 'meta-report-common',
        'host_type': 'meta',
        'sub_role': 'market',
        'location': 'ALL',
        'DEFAULT_MAX_TIME': 1960,
        'MAX_TIME_VALUES': (195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': ('accessories', 'actual_delivery', 'also_viewed', 'alt_same_shop', 'attractive_models', 'best_model_reviews', 'bestdeals', 'better_price', 'book_now_incut', 'brand_products', 'brandbestdeals', 'categories_and_models_by_history', 'compare_products', 'deals', 'defaultoffer', 'delivery_status', 'geo', 'mainreport', 'model_modifications', 'modelinfo', 'multi_category', 'offerinfo', 'omm_electro', 'omm_findings', 'outlets', 'parallel', 'personal_categories', 'personal_offers', 'personal_recommendations', 'personalcategorymodels', 'popular_products', 'prime', 'print_doc', 'product_accessories', 'product_analogs', 'productoffers', 'products_by_history', 'products_by_history_ex', 'promo', 'promoted_categories', 'recipe_by_glfilters', 'recipes_by_doc', 'recipes_contain_glfilters', 'redoffers', 'shop_info', 'sku_search', 'top_categories', 'user_feed', 'vendor_offers_models', 'visualanalog', 'yellow_micro', 'yellow_msku', 'yellow_promo', 'bids_recommender', 'model_bids_recommender', 'partner_model_offers', 'partner_offer_counts', 'vendor_incut'),
        'GRAFANA_UID': 'gzgBAxS7z',
        'PER_HOST_GRAFANA_UID': '1ZgurZz7z',
    },
    'parallel': {
        'title': 'Parallel',
        'report_table': 'parallel-report',
        'nginx_table': 'parallel-report-nginx',
        'error_table': 'parallel-report-common',
        'host_type': 'report',
        'sub_role': 'parallel',
        'DEFAULT_MAX_TIME': 170,
        'MAX_TIME_VALUES': (100, 150, 170, 200, 220, 250, 300, 400, 500),
        'PLACES': ('images', 'omm_parallel', 'parallel'),
        'GRAFANA_UID': 'C-B7TcPik',
        'PER_HOST_GRAFANA_UID': '7wPLglPik',
    },
    'meta-parallel': {
        'title': 'Parallel Meta',
        'report_table': 'parallel-meta-report',
        'nginx_table': 'parallel-meta-report-nginx',
        'error_table': 'parallel-meta-report-common',
        'host_type': 'meta',
        'sub_role': 'parallel',
        'location': 'ALL',
        'DEFAULT_MAX_TIME': 170,
        'MAX_TIME_VALUES': (100, 150, 170, 200, 220, 250, 300, 400, 500),
        'PLACES': ('images', 'omm_parallel', 'parallel', 'parallel_data', 'parallel_slider', 'modelinfo', 'zen_feed'),
        'GRAFANA_UID': 'U2mL0bInk',
        'PER_HOST_GRAFANA_UID': '39_u9Wk7k',
    },
    'api': {
        'title': 'Api Base',
        'report_table': 'main-report-api',
        'nginx_table': 'main-report-api-nginx',
        'error_table': 'api-report-common',
        'host_type': 'report',
        'sub_role': 'api',
        'DEFAULT_MAX_TIME': 780,
        'MAX_TIME_VALUES': (195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': ('accessories', 'actual_delivery', 'also_viewed', 'attractive_models', 'bestdeals', 'compare_products', 'defaultoffer', 'delivery_status', 'formalize_gl', 'geo', 'model_modifications', 'modelinfo', 'offerinfo', 'omm_market', 'omm_parallel', 'outlets', 'personal_categories', 'personalcategorymodels', 'popular_products', 'prime', 'product_accessories', 'product_analogs', 'productoffers', 'products_by_history', 'promo', 'shop_info', 'top_categories', 'user_feed'),
        'GRAFANA_UID': 'oXR4o5Pik',
        'PER_HOST_GRAFANA_UID': 'XqnPglEiz',
    },
    'meta-api': {
        'title': 'Api Meta',
        'report_table': 'api-meta-report',
        'nginx_table': 'api-meta-report-nginx',
        'error_table': 'api-meta-report-common',
        'host_type': 'meta',
        'sub_role': 'api',
        'DEFAULT_MAX_TIME': UNLIMITED_TIME_VALUE,
        'MAX_TIME_VALUES': (195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': ('accessories', 'actual_delivery', 'also_viewed', 'attractive_models', 'bestdeals', 'compare_products', 'defaultoffer', 'delivery_status', 'formalize_gl', 'geo', 'model_modifications', 'modelinfo', 'offerinfo', 'omm_market', 'omm_parallel', 'outlets', 'personal_categories', 'personalcategorymodels', 'popular_products', 'prime', 'product_accessories', 'product_analogs', 'productoffers', 'products_by_history', 'promo', 'shop_info', 'top_categories', 'user_feed'),
        'GRAFANA_UID': 'Id_-1fO7k',
        'PER_HOST_GRAFANA_UID': 'iSxf1BO7z',
    },
    'int': {
        'title': 'Int Base',
        'report_table': 'int-base-report',
        'nginx_table': 'int-base-report-nginx',
        'error_table': 'int-base-report-common',
        'host_type': 'report',
        'sub_role': 'int',
        'DEFAULT_MAX_TIME': UNLIMITED_TIME_VALUE,
        'MAX_TIME_VALUES': (200, 250, 300, 400, 500, 750, 1000, 1500, 2000),
        'PLACES': ('api_offerinfo', 'bids_recommender', 'demand_forecast', 'formalize_gl', 'miprime', 'model_bids_recommender', 'modelinfo', 'msku_info', 'offerinfo', 'partner_model_offers', 'partner_offer_counts', 'price_recommender', 'pricedrops_report', 'prime', 'shopoffers', 'sku_offers'),
        'GRAFANA_UID': 'w-mUdKS7z',
        'PER_HOST_GRAFANA_UID': 'dCywOFInk',
    },
    'meta-int': {
        'title': 'Int Meta',
        'report_table': 'main-report-int',
        'nginx_table': 'main-report-int-nginx',
        'error_table': 'int-main-report-common',
        'host_type': 'meta',
        'sub_role': 'int',
        'DEFAULT_MAX_TIME': UNLIMITED_TIME_VALUE,
        'MAX_TIME_VALUES': (200, 250, 300, 400, 500, 750, 1000, 1500, 2000),
        'PLACES': ('api_offerinfo', 'bids_recommender', 'demand_forecast', 'formalize_gl', 'miprime', 'model_bids_recommender', 'modelinfo', 'msku_info', 'offerinfo', 'partner_model_offers', 'partner_offer_counts', 'price_recommender', 'pricedrops_report', 'prime', 'shopoffers', 'sku_offers'),
        'GRAFANA_UID': '5BTVocPmk',
        'PER_HOST_GRAFANA_UID': '0NQPRlEmz',
    },
    'bk': {
        'title': 'Bk',
        'report_table': 'bk-report',
        'nginx_table': 'bk-report-nginx',
        'error_table': 'bk-report-common',
        'host_type': 'report',
        'sub_role': 'parallel',
        'DEFAULT_MAX_TIME': 1560,
        'MAX_TIME_VALUES': (100, 150, 170, 195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': ('accessories', 'also_viewed', 'attractive_models', 'best_model_reviews', 'bestdeals', 'better_price', 'blue_attractive_models', 'book_now_incut', 'brandbestdeals', 'categories_and_models_by_history', 'check_delivery_available', 'compare_products', 'deals', 'defaultoffer', 'geo', 'model_modifications', 'modelinfo', 'multi_category', 'offerinfo', 'omm_findings', 'parallel', 'personal_categories', 'personal_offers', 'personal_recommendations', 'personalcategorymodels', 'popular_products', 'prime', 'product_accessories', 'product_analogs', 'productoffers', 'products_by_history', 'products_by_history_ex', 'promo', 'promoted_categories', 'recipe_by_glfilters', 'recipes_by_doc', 'recipes_contain_glfilters', 'shop_info', 'shop_vendor_promo_clicks_stats', 'sku_offers', 'top_categories', 'vendor_offers_models'),
        'GRAFANA_UID': '6x6VocEik',
        'PER_HOST_GRAFANA_UID': 'LaOyRlPik',
    },
    'shadow': {
        'title': 'Shadow',
        'source_report_table': 'meta-report',
        'source_nginx_table': 'meta-report-nginx',
        'source_error_table': 'meta-report-common',
        'source_host_type': 'report',
        'source_env': 'prestable',
        'source_sub_role': 'market',
        'source_loc': 'sas',
        'source_cluster': '0',
        'shadow_report_table': 'shadow-report',
        'shadow_nginx_table': 'shadow-report-nginx',
        'shadow_error_table': 'shadow-report-common',
        'shadow_host_type': 'report',
        'shadow_env': '{production,priemka}',
        'shadow_sub_role': 'shadow',
        'shadow_loc': ('sas', 'vla'),
        'shadow_cluster': '0',
        'DEFAULT_MAX_TIME': 1960,
        'MAX_TIME_VALUES': (195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': ('accessories', 'actual_delivery', 'also_viewed', 'alt_same_shop', 'attractive_models', 'best_model_reviews', 'bestdeals', 'better_price', 'book_now_incut', 'brand_products', 'brandbestdeals', 'categories_and_models_by_history', 'compare_products', 'deals', 'defaultoffer', 'delivery_status', 'geo', 'mainreport', 'model_modifications', 'modelinfo', 'multi_category', 'offerinfo', 'omm_electro', 'omm_findings', 'outlets', 'parallel', 'personal_categories', 'personal_offers', 'personal_recommendations', 'personalcategorymodels', 'popular_products', 'prime', 'print_doc', 'product_accessories', 'product_analogs', 'productoffers', 'products_by_history', 'products_by_history_ex', 'promo', 'promoted_categories', 'recipe_by_glfilters', 'recipes_by_doc', 'recipes_contain_glfilters', 'redoffers', 'shop_info', 'sku_search', 'top_categories', 'user_feed', 'vendor_offers_models', 'visualanalog'),
        'GRAFANA_UID': 'iDHNT5Pik',
        'PER_HOST_GRAFANA_UID': 'CqRvv_Eik',
    },
    'market-kraken': {
        'title': 'Kraken Base',
        'report_table': 'kraken-base-report',
        'nginx_table': 'kraken-base-report-nginx',
        'error_table': 'kraken-base-report-common',
        'host_type': 'report',
        'sub_role': 'market-kraken',
        'location': 'vla',
        'DEFAULT_MAX_TIME': 1960,
        'MAX_TIME_VALUES': (195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': ('accessories', 'actual_delivery', 'also_viewed', 'blue_attractive_models', 'check_delivery_available', 'commonly_purchased', 'deals', 'defaultoffer', 'delivery_status', 'formalize_gl', 'geo', 'modelinfo', 'nids_info', 'offerinfo', 'outlets', 'popular_products', 'prime', 'product_accessories', 'product_accessories_blue', 'productoffers', 'products_by_history', 'promo_bundle', 'recipe_by_glfilters', 'recipes_contain_glfilters', 'shop_info', 'sku_offers'),
        'GRAFANA_UID': 'Briq9GHnz',
        'PER_HOST_GRAFANA_UID': 'W5HC9MHnz',
    },
    'meta-market-kraken': {
        'title': 'Kraken Meta',
        'report_table': 'marketkraken16-report',
        'nginx_table': 'marketkraken16-report-nginx',
        'error_table': 'marketkraken16-report-common',
        'host_type': 'meta',
        'sub_role': 'market-kraken',
        'location': 'vla',
        'DEFAULT_MAX_TIME': 1960,
        'MAX_TIME_VALUES': (195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': ('accessories', 'actual_delivery', 'also_viewed', 'blue_attractive_models', 'check_delivery_available', 'commonly_purchased', 'deals', 'defaultoffer', 'delivery_status', 'formalize_gl', 'geo', 'modelinfo', 'nids_info', 'offerinfo', 'outlets', 'popular_products', 'prime', 'product_accessories', 'product_accessories_blue', 'productoffers', 'products_by_history', 'promo_bundle', 'recipe_by_glfilters', 'recipes_contain_glfilters', 'shop_info', 'sku_offers'),
        'GRAFANA_UID': 'SDYNTcEiz',
        'PER_HOST_GRAFANA_UID': 'aHfCO2PGk',
    },
    'goods-warehouse': {
        'title': 'Goods Warehouse',
        'report_table': 'goods-warehouse-report',
        'nginx_table': 'goods-warehouse-report-nginx',
        'error_table': 'goods-warehouse-report-common',
        'host_type': 'report',
        'sub_role': 'goods-warehouse',
        'DEFAULT_MAX_TIME': 1960,
        'MAX_TIME_VALUES': (195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': (
            'accessories', 'actual_delivery', 'also_viewed', 'alt_same_shop', 'attractive_models', 'best_model_reviews',
            'bestdeals', 'better_price', 'book_now_incut', 'brand_products', 'brandbestdeals',
            'categories_and_models_by_history', 'compare_products', 'deals', 'defaultoffer', 'delivery_status', 'geo',
            'mainreport', 'model_modifications', 'modelinfo', 'multi_category', 'offerinfo', 'omm_electro', 'omm_findings',
            'outlets', 'parallel', 'personal_categories', 'personal_offers', 'personal_recommendations',
            'personalcategorymodels', 'popular_products', 'prime', 'print_doc', 'product_accessories', 'product_analogs',
            'productoffers', 'products_by_history', 'products_by_history_ex', 'promo', 'promoted_categories',
            'recipe_by_glfilters', 'recipes_by_doc', 'recipes_contain_glfilters', 'redoffers', 'shop_info', 'sku_search',
            'top_categories', 'user_feed', 'vendor_offers_models', 'visualanalog', 'yellow_micro', 'yellow_msku',
            'yellow_promo'
        ),
        'GRAFANA_UID': '1HRljrI7z',
        'PER_HOST_GRAFANA_UID': '0aPEj9Snz',
    },
    'goods-parallel': {
        'title': 'Goods Parallel',
        'report_table': 'goods-parallel-report',
        'nginx_table': 'goods-parallel-report-nginx',
        'error_table': 'goods-parallel-report-common',
        'host_type': 'report',
        'sub_role': 'goods-parallel',
        'DEFAULT_MAX_TIME': 170,
        'MAX_TIME_VALUES': (100, 150, 170, 200, 220, 250, 300, 400, 500),
        'PLACES': ('images', 'omm_parallel', 'parallel'),
        'GRAFANA_UID': 'QQGjC9Snz',
        'PER_HOST_GRAFANA_UID': 'MkCXjrS7k',
    },
    'fresh-base': {
        'title': 'Fresh Base',
        'error_table': 'fresh-base-report-common',
        'sub_role': 'fresh-base',
        'host_type': 'report',
        'location': 'ALL',
        'DEFAULT_MAX_TIME': 780,
        'MAX_TIME_VALUES': (195, 490, 780, 980, 1170, 1560, 1960, 2450, 2940, 3920, 4900),
        'PLACES': ('accessories', 'actual_delivery', 'also_viewed', 'attractive_models', 'bestdeals', 'compare_products', 'defaultoffer', 'delivery_status', 'formalize_gl', 'geo', 'model_modifications', 'modelinfo', 'offerinfo', 'omm_market', 'omm_parallel', 'outlets', 'personal_categories', 'personalcategorymodels', 'popular_products', 'prime', 'product_accessories', 'product_analogs', 'productoffers', 'products_by_history', 'promo', 'shop_info', 'top_categories', 'user_feed'),
        'GRAFANA_UID': 'P1ZLZ7cnz',
        'PER_HOST_GRAFANA_UID': 'XPYpwztnz',
    },
}


def retry(max_try_count=5, initial_delay=2, exp_backoff=2):

    def retry_decorator(function):

        @functools.wraps(function)
        def retry_wrapper(*args, **kwargs):
            try_count = 0
            retry_delay = initial_delay
            while True:
                try:
                    return function(*args, **kwargs)
                except Exception:
                    try_count += 1
                    if try_count == max_try_count:
                        raise
                    logging.exception('Execution failed %d times, retrying...', try_count)
                    time.sleep(retry_delay)
                    retry_delay *= exp_backoff
        return retry_wrapper

    return retry_decorator


class Metrics(object):
    MetricInfo = collections.namedtuple('MetricInfo', ['expression', 'hidden'])

    def __init__(self):
        self.ref_id = 0
        self.metrics = dict()

    def add(self, expression, hidden=True):
        ref_id = chr(self.ref_id + ord('A'))
        self.ref_id += 1
        self.metrics[ref_id] = self.MetricInfo(expression, hidden)
        return ref_id

    def _expand_metric(self, metric_expr):
        while True:
            ref_pos = metric_expr.find('#')
            if ref_pos == -1:
                break
            ref_id = metric_expr[ref_pos + 1]
            metric_expr = metric_expr.replace(metric_expr[ref_pos: ref_pos + 2], self._expand_metric(self.metrics[ref_id].expression))
        return metric_expr

    def make_graph_targets(self, report_type):
        graph_targets = list()
        for ref_id in range(self.ref_id):
            ref_id = chr(ref_id + ord('A'))
            metric_info = self.metrics[ref_id]
            graph_targets.append({
                'hide': metric_info.hidden,
                'refId': ref_id,
                'target': subst_tmpl(report_type, metric_info.expression),
                'targetFull': subst_tmpl(report_type, self._expand_metric(metric_info.expression)),
                'textEditor': True,
            })
        return graph_targets


def gen_metric_groups(report_type, per_host):

    def gen_host_metric_name(metric_name, metric_type='max'):
        if per_host:
            result = 'one_min.HOST.$host.{metric_name}'.format(metric_name=metric_name)
        else:
            result = 'one_min.report_metrics.{metric_type}.$host_type.env.$env.sub_role.$sub_role.location.$loc.cluster.$cluster.metric.{metric_name}'.format(
                metric_type=metric_type, metric_name=metric_name.replace('.', '_')
            )
            if metric_type != 'sum':
                result += '.${}_p'.format(metric_type)
        return result

    def gen_metric_name(table_name, metric_name):
        if per_host:
            return 'one_min.${table_name}.{metric_name}.host.$host'.format(table_name=table_name, metric_name=metric_name)
        else:
            return 'one_min.${table_name}.{metric_name}.env.$env.loc.$loc.cluster.$cluster'.format(table_name=table_name, metric_name=metric_name)

    def gen_metric_name_with_error_code(table_name, metric_name):
        if per_host:
            return gen_metric_name(table_name, metric_name)
        else:
            return 'one_min.${table_name}.{metric_name}.env.$env.loc.$loc.cluster.$cluster.code.$error_code'.format(table_name=table_name, metric_name=metric_name)

    def gen_metric_name_with_place(table_name, metric_name):
        if per_host:
            return gen_metric_name(table_name, metric_name)
        else:
            return 'one_min.${table_name}.{metric_name}.env.$env.loc.$loc.cluster.$cluster.place.$place'.format(table_name=table_name, metric_name=metric_name)

    def gen_metric_name_with_place_and_quant(table_name, metric_name):
        if per_host:
            return '$i.${table_name}.{metric_name}.host.$host.$t'.format(table_name=table_name, metric_name=metric_name)
        else:
            return '$i.${table_name}.{metric_name}.env.$env.loc.$loc.cluster.$cluster.place.$place.$t'.format(table_name=table_name, metric_name=metric_name)

    def gen_metric_name_with_quant(table_name, metric_name):
        if per_host:
            return gen_metric_name_with_place_and_quant(table_name, metric_name)
        else:
            return '$i.${table_name}.{metric_name}.env.$env.loc.$loc.cluster.$cluster.$t'.format(table_name=table_name, metric_name=metric_name)

    def gen_host_metric(name, title, units, panel_id, metric_type='max'):
        return {
            'panel_id': panel_id,
            'metric': gen_host_metric_name(name, metric_type=metric_type),
            'width': ROW_WIDTH / 3,
            'height': graph_height,
            'title': title,
            'units': units,
            'agg': metric_type,
        }

    def gen_publisher_metric(name, title, panel_id):
        return {
            'panel_id': panel_id,
            'metric': gen_host_metric_name(name),
            'width': ROW_WIDTH / 3,
            'height': graph_height,
            'title': title,
            'hide_axis': True,
            'null_point_mode': 'connected',
            'smoothing': False,
        }

    def gen_backctld_metric_name(metric_name):
        if per_host:
            result = 'one_min.backctld_perf.general.max.host.$host.metric.{metric_name}'.format(metric_name=metric_name)
        else:
            result = 'one_min.backctld_perf.general.max.environment.$env.report_subrole.$sub_role.location.$loc.cluster_index.$cluster.metric.{metric_name}'.format(metric_name=metric_name)
        return result

    def gen_backctld_metric(name, title, units, panel_id):
        return {
            'panel_id': panel_id,
            'metric': gen_backctld_metric_name(name),
            'width': ROW_WIDTH / 3,
            'height': graph_height,
            'points': True,
            'title': title,
            'units': units,
        }

    def add_description(metric, description):
        metric['description'] = description
        return metric

    def get_error_percentage_metrics(logarithmic, base_panel_id):
        return [
            {
                'panel_id': base_panel_id + 0,
                'metric': gen_metric_name_with_place('report_table', 'error-percentage'),
                'width': ROW_WIDTH / 3,
                'height': graph_height,
                'title': 'Error Percentage [access.log place=$place]',
                'units': 'percent',
                'log_axis': logarithmic,
            },
            {
                'panel_id': base_panel_id + 1,
                'metric': gen_metric_name_with_place('report_table', 'partial-answer-percentage'),
                'width': ROW_WIDTH / 3,
                'height': graph_height,
                'title': 'Partial Response Percentage [access.log place=$place]',
                'units': 'percent',
                'log_axis': logarithmic,
            },
            {
                'panel_id': base_panel_id + 2,
                'metric': gen_metric_name('nginx_table', '4xx_error_percentage'),
                'width': ROW_WIDTH / 3,
                'height': graph_height,
                'title': 'Nginx 4xx Error Percentage',
                'units': 'percent',
                'log_axis': logarithmic,
            },
            {
                'panel_id': base_panel_id + 3,
                'metric': gen_metric_name('nginx_table', '5xx_error_percentage'),
                'width': ROW_WIDTH / 3,
                'height': graph_height,
                'title': 'Nginx 5xx Error Percentage',
                'units': 'percent',
                'log_axis': logarithmic,
            },
            {
                'panel_id': base_panel_id + 4,
                'metric': gen_metric_name('nginx_table', 'slow_response_percentage'),
                'width': ROW_WIDTH / 3,
                'height': graph_height,
                'title': 'Slow Response Percentage [place=parallel upstream_resp_time>180ms]',
                'units': 'percent',
                'visible': report_type == 'parallel',
                'log_axis': logarithmic,
            },
            {
                'panel_id': base_panel_id + 5,
                'metric': 'one_min.$nginx_table.status_codes.env.$env.loc.$loc.cluster.$cluster.status.$status',
                'width': ROW_WIDTH / 3,
                'height': graph_height,
                'title': 'Nginx Status Codes [status=$status]',
                'units': 'short',
                'log_axis': logarithmic,
                'visible': not per_host,
            },
        ]

    graph_height = SHADOW_GRAPH_HEIGHT if is_shadow_report(report_type) and not per_host else GRAPH_HEIGHT

    errors_group = {
        'name': 'Errors',
        'metrics': [
           {
               'panel_id': 15,
               'metric': gen_metric_name_with_error_code('error_table', 'errors'),
               'width': ROW_WIDTH / 3,
               'height': graph_height,
               'title': 'Errors [error.log code=$error_code]',
               'units': 'short',
               'links': [
                   {
                       'title': 'report (3xxx,9xxx)',
                       'type': 'absolute',
                       'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/market/report/library/error_code/error_code.h'
                   },
                   {
                       'title': 'libreport (1xxx)',
                       'type': 'absolute',
                       'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/market/library/error_code/error_code_base.h'
                   },
                   {
                       'title': 'libmarket (0xxx)',
                       'type': 'absolute',
                       'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/market/library/error_code/error_code.h'
                   },
                   {
                       'title': 'recom (2xxx)',
                       'type': 'absolute',
                       'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/market/library/recom/formula/ErrorCode.h'
                   }
               ],
           },
       ] + get_error_percentage_metrics(logarithmic=False, base_panel_id=200)
    }

    cpu_usage_group = {
        'name': 'CPU Usage',
        'metrics': [
            gen_host_metric('report_process_stats.utime', 'User CPU time', 'percentunit', panel_id=74),
            gen_host_metric('report_process_stats.stime', 'System CPU time', 'percentunit', panel_id=75),
            {
                'panel_id': 105,
                'metric': gen_metric_name_with_place_and_quant('report_table', 'cpu_time_us'),
                'width': ROW_WIDTH / 3,
                'height': graph_height,
                'title': 'CPU Time per Request [place=$place quant=$i,$t]',
                'units': 'µs'
            },
            {
                'panel_id': 110,
                'metric': gen_metric_name_with_place_and_quant('report_table', 'wait_time_us'),
                'width': ROW_WIDTH / 3,
                'height': graph_height,
                'title': 'Wait Time per Request [place=$place quant=$i,$t]',
                'units': 'µs'
            },
            {
                'panel_id': 111,
                'metric': gen_metric_name_with_place_and_quant('report_table', 'major_faults'),
                'width': ROW_WIDTH / 3,
                'height': graph_height,
                'title': 'Major Faults per Request [place=$place quant=$i,$t]',
                'units': 'short'
            },
        ]
    }

    metric_groups = [
        {
            'name': 'RPS',
            'metrics': [
                {
                    'panel_id': 1,
                    'metric': gen_metric_name_with_place('nginx_table', 'rps'),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Nginx RPS [place=$place]',
                    'units': 'reqps',
                    'visible': report_type != 'parallel',
                },
                {
                    'panel_id': 2,
                    'metric': gen_metric_name_with_place('report_table', 'rps'),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Meta RPS [place=$place]',
                    'units': 'reqps'
                },
            ]
        },
        {
            'name': 'Timings',
            'metrics': [
                {
                    'panel_id': 3,
                    'metric': 'removeBelowValue(transformNull(removeAboveValue(transformNull({}, -1), $m), $m), 0)'.format(gen_metric_name_with_place_and_quant('nginx_table', 'upstream_resp_time')),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Nginx Timings [place=$place timings=$i,$t limit=$m]',
                    'units': 'ms',
                    'visible': report_type != 'parallel',
                },
                {
                    'panel_id': 5,
                    'metric': 'removeBelowValue(transformNull(removeAboveValue(transformNull({}, -1), $m), $m), 0)'.format(gen_metric_name_with_place_and_quant('report_table', 'timing')),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Meta Timings [place=$place timings=$i,$t limit=$m]',
                    'units': 'ms'
                },
                {
                    'panel_id': 6,
                    'metric': 'removeBelowValue(transformNull(removeAboveValue(transformNull({}, -1), $m), $m), 0)'.format(gen_metric_name_with_place_and_quant('report_table', 'base_timing')),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Base Timings [place=$place timings=$i,$t limit=$m]',
                    'units': 'ms'
                },
                {
                    'panel_id': 109,
                    'metric': 'removeBelowValue(transformNull(removeAboveValue(transformNull({}, -1), $m), $m), 0)'.format(gen_metric_name_with_place_and_quant('report_table', 'external_snippet_stall_time_ms')),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'External Snippet Timings [place=$place timings=$i,$t limit=$m]',
                    'units': 'ms'
                },
                {
                    'panel_id': 10,
                    'metric': 'removeBelowValue(transformNull(removeAboveValue(transformNull({}, -1), $m), $m), 0)'.format(gen_metric_name_with_quant('report_table', 'product_request_timing')),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Product Response Timings [place=parallel wizards timings=$i,$t limit=$m]',
                    'units': 'ms',
                    'visible': report_type == 'parallel',
                },
                {
                    'panel_id': 4,
                    'metric': 'removeBelowValue(transformNull(removeAboveValue(transformNull({}, -1), $m), $m), 0)'.format(gen_metric_name_with_place_and_quant('nginx_table', 'req_time')),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Req Timings [place=$place timings=$i,$t limit=$m]',
                    'units': 'ms',
                    'visible': report_type != 'parallel',
                },
            ]
        },
        {
            'name': 'Documents',
            'metrics': [
                {
                    'panel_id': 11,
                    'metric': gen_metric_name_with_place_and_quant('report_table', 'total_documents_processed'),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Processed Documents [place=$place quant=$i,$t]',
                    'units': 'short'
                },
                {
                    'panel_id': 12,
                    'metric': gen_metric_name_with_place_and_quant('report_table', 'total_documents_accepted'),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Accepted Documents [place=$place quant=$i,$t]',
                    'units': 'short'
                },
                {
                    'panel_id': 13,
                    'metric': gen_metric_name('report_table', 'wizard_percentage.all'),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Product Response Percentage [place=parallel wizards]',
                    'units': 'percent',
                    'visible': report_type == 'parallel',
                },
                {
                    'panel_id': 14,
                    'metric': gen_metric_name('report_table', 'wizard_percentage.slow'),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Slow Product Response Percentage [place=parallel wizards full_elapsed>180ms]',
                    'units': 'percent',
                    'visible': report_type == 'parallel',
                },
            ]
        },
        errors_group,
        {
            'name': 'Errors [logarithmic scale]',
            'metrics': get_error_percentage_metrics(logarithmic=True, base_panel_id=300)
        },
        {
            'name': 'Core Dumps & OOM Kills',
            'metrics': [
                {
                    'panel_id': 16,
                    'metric': gen_host_metric_name('core_dump.report_core_count', 'sum'),
                    'width': ROW_WIDTH / 3,
                    'height': SMALL_HEIGHT,
                    'title': 'Report Core Dumps',
                    'units': 'short',
                    'bars': True,
                    'smoothing': False,
                },
                {
                    'panel_id': 17,
                    'metric': gen_host_metric_name('core_dump.count', 'sum'),
                    'width': ROW_WIDTH / 3,
                    'height': SMALL_HEIGHT,
                    'title': 'All Core Dumps',
                    'units': 'short',
                    'bars': True,
                    'smoothing': False,
                },
                {
                    'panel_id': 18,
                    'metric': gen_host_metric_name('porto_stats.oom_kills', 'sum'),
                    'width': ROW_WIDTH / 3,
                    'height': SMALL_HEIGHT,
                    'title': 'OOM Kills',
                    'units': 'short',
                    'bars': True,
                    'smoothing': False,
                },
            ]
        },
        {
            'name': 'Report Memory',
            'metrics': [
                gen_host_metric('memory_stats.report_rss_anon', 'Anonymous Memory Usage', 'bytes', panel_id=19),
                gen_host_metric('report_internals.anonymous_memory_consumption_after_global', 'Global Memory Usage', 'bytes', panel_id=21),
                gen_host_metric('report_internals.balloc_cur_size', 'Balloc Total Memory', 'bytes', panel_id=22),
                gen_host_metric('report_internals.balloc_gc_size', 'Balloc Cache Size', 'bytes', panel_id=23),
                gen_host_metric('report_internals.balloc_in_use', 'Balloc Memory Usage', 'bytes', panel_id=24),
                gen_host_metric('report_internals.lf_total_size', 'LF Total Memory', 'bytes', panel_id=112),
                gen_host_metric('report_internals.lf_cache_size', 'LF Cache Size', 'bytes', panel_id=113),
                gen_host_metric('report_internals.lf_alloc_size', 'LF Memory Usage', 'bytes', panel_id=25),
                gen_host_metric('report_internals.index_memory_lock_quota_usage', 'Index Size', 'bytes', panel_id=114),
                gen_host_metric('report_internals.rty_index_memory_lock_quota_usage', 'RTY Index Size', 'bytes', panel_id=115),
                gen_host_metric('report_process_stats.index_cache_usage', 'Index Cache Usage', 'bytes', metric_type='min', panel_id=116),
                {
                    'panel_id': 104,
                    'metric': gen_metric_name_with_place_and_quant('report_table', 'estimated_max_memory_usage'),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Memory Usage per Request [place=$place quant=$i,$t]',
                    'units': 'bytes'
                },
            ]
        },
        {
            'name': 'System Memory',
            'metrics': [
                gen_host_metric('porto_stats.memory_left', 'Memory Left', 'bytes', metric_type='min', panel_id=26),
                gen_host_metric('porto_stats.memory_free', 'Free Memory', 'bytes', metric_type='min', panel_id=118),
                gen_host_metric('porto_stats.cache_usage', 'Cache Usage', 'bytes', metric_type='min', panel_id=119),
                gen_host_metric('porto_stats.mlock_usage', 'Locked Memory', 'bytes', metric_type='max', panel_id=120),
            ]
        },
        cpu_usage_group,
        {
            'name': 'Disk Usage',
            'metrics': [
                gen_host_metric('disk_usage.report_data_size', 'Report Data', 'bytes', panel_id=35),
                gen_host_metric('disk_usage.dynamic_models_size', 'Dynamic Models', 'bytes', panel_id=36),
                gen_host_metric('disk_usage.formulas_size', 'Formulas', 'bytes', panel_id=37),
                gen_host_metric('disk_usage.generations_size', 'Full Generations', 'bytes', panel_id=38),
                gen_host_metric('disk_usage.packages_size', 'Packages', 'bytes', panel_id=40),
                gen_host_metric('disk_usage.pdata_free_space', 'Pdata Free Space', 'bytes', metric_type='min', panel_id=41),
                gen_host_metric('disk_usage.index_free_space', 'Index Free Space', 'bytes', metric_type='min', panel_id=121),
            ]
        },
        {
            'name': 'Network',
            'metrics': [
                {
                    'panel_id': 43,
                    'metric': gen_metric_name_with_place('nginx_table', 'request_length'),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Total Nginx In [place=$place]',
                    'units': 'Bps',
                    'visible': report_type != 'parallel',
                },
                {
                    'panel_id': 44,
                    'metric': gen_metric_name_with_place('nginx_table', 'bytes_sent'),
                    'width': ROW_WIDTH / 3,
                    'height': graph_height,
                    'title': 'Total Nginx Out [place=$place]',
                    'units': 'Bps',
                    'visible': report_type != 'parallel',
                },
                gen_host_metric('net_stats.bytes_received', 'Bytes In', 'Bps', panel_id=45),
                gen_host_metric('net_stats.bytes_sent', 'Bytes Out', 'Bps', panel_id=46),
                gen_host_metric('net_stats.curr_estab', 'Number of Connections', 'short', panel_id=47),
                gen_host_metric('net_stats.active_opens', 'Active Opens', 'ops', panel_id=48),
                gen_host_metric('net_stats.passive_opens', 'Passive Opens', 'ops', panel_id=49),
                gen_host_metric('net_stats.retrans_segs', 'Retransmitted Segments', 'ops', panel_id=55),
            ]
        },
        {
            'name': 'Publisher',
            'metrics': [
                gen_publisher_metric('report_versions.consistent_report_version', 'Report Version', panel_id=65),
                gen_publisher_metric('report_versions.consistent_index_generation', 'Index Generation', panel_id=66),
                gen_publisher_metric('report_versions.indexer_version', 'Indexer Version', panel_id=68),
                gen_publisher_metric('report_versions.dssm_version', 'DSSM Version', panel_id=69),
                gen_publisher_metric('report_versions.formulas_version', 'Formulas Version', panel_id=70),
                add_description(gen_publisher_metric('report_versions.report_status', 'Report Status', panel_id=106), '''0 = OPENED_CONSISTENT

1 = CLOSED_CONSISTENT_MANUAL_OPEN

2 = CLOSED_INCONSISTENT_MANUAL_OPEN

3 = CLOSED_INCONSISTENT_AUTO_OPEN

4 = connection problem'''),
            ]
        },
        {
            'name': 'Startup Time',
            'metrics': [
                gen_host_metric('report_internals.startup_time', 'Startup Time', 'ms', panel_id=71),
                gen_host_metric('report_internals.rty_startup_time', 'RTY Startup Time', 'ms', panel_id=72),
                gen_host_metric('report_internals.access_startup_time', 'Market Access Startup Time', 'ms', panel_id=97),
                gen_host_metric('report_internals.global_startup_time', 'Global Startup Time', 'ms', panel_id=73),
            ]
        },
        {
            'name': 'Reload',
            'metrics': [
                gen_backctld_metric('full_only_reload', 'Reload Time', 'ms', panel_id=95),
                gen_backctld_metric('copy_index', 'Unpack Index Time', 'ms', panel_id=96),
            ]
        },
        {
            'name': 'Process Stats',
            'metrics': [
                gen_host_metric('report_process_stats.majflt', 'Major Page Faults', 'ops', panel_id=76),
                gen_host_metric('report_process_stats.minflt', 'Minor Page Faults', 'ops', panel_id=77),
                gen_host_metric('report_process_stats.nonvoluntary_ctxt_switches', 'Non-voluntary Context Switches', 'ops', panel_id=78),
                gen_host_metric('report_process_stats.voluntary_ctxt_switches', 'Voluntary Context Switches', 'ops', panel_id=79),
            ]
        },
        {
            'name': 'Process I/O',
            'metrics': [
                gen_host_metric('report_process_stats.read_bytes', 'Bytes Read', 'Bps', panel_id=86),
                gen_host_metric('report_process_stats.write_bytes', 'Bytes Written', 'Bps', panel_id=87),
            ]
        },
        {
            'name': 'Process Stats 2',
            'metrics': [
                gen_host_metric('report_internals.meta_queue_length', 'Meta Queue Length', 'short', panel_id=90),
                gen_host_metric('report_internals.max_base_queue_length', 'Max Basesearch Queue Length', 'short', panel_id=91),
                gen_host_metric('report_internals.meta_active_thread_count', 'Meta Active Thread Count', 'short', panel_id=92),
                gen_host_metric('report_internals.base_active_thread_count', 'Basesearch Active Thread Count', 'short', panel_id=93),
                gen_host_metric('report_internals.relevance_calcer_active_thread_count', 'Relevance Pool Active Thread Count', 'short', panel_id=94),
            ]
        },
    ]

    def replace_metric_variables(metric, report_type):
        for key in REPORT_TYPES[report_type]:
            metric = metric.replace('$' + key, str(REPORT_TYPES[report_type][key]))
        return metric

    def change_group_host_to_report(group):
        base_report_type = report_type.replace('meta-', '')
        for metric_conf in group:
            if isinstance(metric_conf['metric'], list):
                metrics = []
                for metric in metric_conf['metric']:
                    metrics.append(replace_metric_variables(metric, base_report_type))
                metric_conf['metric'] = metrics
            else:
                metric_conf['metric'] = replace_metric_variables(metric_conf['metric'], base_report_type)

    base_copy_panel_id = 1000

    for group in [cpu_usage_group, errors_group]:
        if 'meta' in report_type:
            base_group = deepcopy(group)
            change_group_host_to_report(base_group['metrics'])

            base_group['name'] = 'Base ' + group['name']

            for metric in base_group['metrics']:
                metric['panel_id'] = base_copy_panel_id
                base_copy_panel_id += 1

            modified_groups = []
            for old_group in metric_groups:
                modified_groups.append(old_group)
                if old_group == group:
                    modified_groups.append(base_group)

            metric_groups = modified_groups

    return metric_groups


def subst_tmpl(report_type, text, prefix=None):
    for name, value in REPORT_TYPES[report_type].iteritems():
        if isinstance(value, str):
            if prefix:
                if name.startswith(prefix):
                    text = text.replace('${}'.format(name.split('_', 1)[1]), value)
            else:
                text = text.replace('${}'.format(name), value)
    return text


def make_graph(metric, report_type, per_host, x, y, w, h):
    overrides = list()
    if 'right_axis' in metric:
        overrides.extend([
            {
                'alias': name,
                'yaxis': 2
            }
            for name in metric['right_axis'].split(',')
        ])
    overrides.extend(metric.get('display_overrides', list()))

    if isinstance(metric['metric'], Metrics):
        targets = metric['metric'].make_graph_targets(report_type)
    else:
        SHADOW_GRAPH_TYPES = ('shadow',) if per_host else ('source', 'shadow')
        metric_list = list()
        if is_shadow_report(report_type):
            for metric_type in SHADOW_GRAPH_TYPES:
                metric_list.append(subst_tmpl(report_type, metric['metric'], metric_type))
        else:
            metric_list.append(subst_tmpl(report_type, metric['metric']))

        targets = list()
        for index, metric_expr in enumerate(metric_list):
            if per_host:
                label_regex = r'.+\.([a-z]{3}\d-\d{4}).+-(\d+)_gencfg.*'
                label_format = r'\1:\2'

                if metric.get('smoothing', True):
                    metric_expr = metric_expr + '$smooth'
                metric_expr = 'aliasSub(' + metric_expr + ', "{label_regex}", "{label_format}")'.format(
                    label_regex=label_regex, label_format=label_format
                )
            else:
                if 'report_metrics.' in metric_expr:
                    label_regex = r'.+\.env\.([^.]+)\.sub_role\.[^.]+\.location\.([^.]+)\.cluster\.([^.]+)\.metric\.'
                    if 'report_metrics.sum.' in metric_expr:
                        label_regex += r'(\.*)[^,)]+.*'
                    else:
                        label_regex += r'[^.]+\.([\d_]+).*'
                elif 'backctld_perf.' in metric_expr:
                    label_regex = r'.+\.environment\.([^.]+)\.report_subrole\.[^.]+\.location\.([^.]+)\.cluster_index\.([^.]+)\.(\.*).*'
                else:
                    label_regex = r'.+\.env\.([^.]+)\.loc\.([^.]+)\.cluster\.(\d+)'
                    if '.$error_code' in metric_expr:
                        label_regex += r'\.code\.([^,)]+).*'
                    if '.status.' in metric_expr:
                        label_regex += r'\.status\.([^,)]+).*'
                    elif '.place.' in metric_expr and '.$t' in metric_expr:
                        label_regex += r'\.place\.([^.]+\.[\d_]+).*'
                    elif '.place.' in metric_expr:
                        label_regex += r'\.place\.([^,)]+).*'
                    elif '.$t' in metric_expr:
                        label_regex += r'\.([^,)]+).*'
                    else:
                        label_regex += r'([^,]*).*'

                if is_shadow_report(report_type):
                    label_format = SHADOW_GRAPH_TYPES[index] + r'.\4'
                else:
                    label_format = '${label:csv}'

                metric_expr = 'removeEmptySeries(consolidateBy(' + metric_expr + ', "{}"))'.format(metric.get('agg', 'max'))
                if metric.get('smoothing', True):
                    metric_expr = metric_expr + '$smooth'
                metric_expr = 'sortByName(aliasSub(' + metric_expr + ', "{label_regex}", "{label_format}"), natural=True)'.format(
                    label_regex=label_regex, label_format=label_format
                )
            targets.append(
                {
                    'refId': chr(ord('A') + index),
                    'target': metric_expr
                }
            )

    graph = {
        'bars': metric.get('bars', False),
        'datasource': 'market',
        'description': metric.get('description'),
        'fill': metric.get('fill', 0),
        'gridPos': {
            'h': h,
            'w': w,
            'x': x,
            'y': y
        },
        'id': metric['panel_id'],
        'legend': {
            'show': True
        },
        'lines': not metric.get('bars', False) and not metric.get('points', False),
        'links': metric.get('links', list()),
        'maxDataPoints': '2000',
        'nullPointMode': metric.get('null_point_mode', 'null'),
        'points': metric.get('points', False),
        'pointradius': 4,
        'seriesOverrides': overrides,
        'stack': metric.get('stack', False),
        'targets': targets,
        'title': metric['title'],
        'tooltip': {
            'sort': 2
        },
        'type': 'graph',
        'yaxes': [
            {
                'format': metric.get('units', 'none'),
                'logBase': 2 if metric.get('log_axis') else 1,
                'max': metric.get('y_max'),
                'show': not metric.get('hide_axis', False)
            }
            for _ in range(2)
        ],
    }
    return graph


def get_standard_percentiles(report_type):
    if report_type == 'parallel':
        percentiles = ('0_5', '0_6', '0_7', '0_80', '0_90', '0_95', '0_97', '0_99', '0_995', '0_997', '0_998', '0_999', '1')
        labels = ('50%', '60%', '70%', '80%', '90%', '95%', '97%', '99%', '99.5%', '99.7%', '99.8%', '99.9%', '100%')
    else:
        percentiles = ('0_5', '0_6', '0_7', '0_80', '0_90', '0_95', '0_97', '0_99', '0_995', '0_997', '0_999', '1')
        labels = ('50%', '60%', '70%', '80%', '90%', '95%', '97%', '99%', '99.5%', '99.7%', '99.9%', '100%')
    return zip(percentiles, labels)


def get_host_metric_percentiles(is_max=True):
    if is_max:
        percentiles = ('0', '0_5', '0_90', '0_99', '1')
        labels = ('min', 'median', '90%', '99%', 'max')
    else:
        percentiles = ('0', '0_01', '0_1', '0_5', '1')
        labels = ('min', '1%', '10%', 'median', 'max')
    return zip(percentiles, labels)


def make_templating(report_type, per_host, nanny_auth_token):
    MAX_TIME_FORMAT = '{}ms'
    DEFAULT_MAX_TIME = str(REPORT_TYPES[report_type]['DEFAULT_MAX_TIME'])
    UNLIMITED_TIME_NAME = 'Unlimited'
    MAX_TIME_VALUES = [UNLIMITED_TIME_VALUE] + map(str, REPORT_TYPES[report_type]['MAX_TIME_VALUES'])
    ALL_PLACES = 'ALL'
    PLACES = [ALL_PLACES] + list(REPORT_TYPES[report_type]['PLACES'])
    DEFAULT_PERCENTILE = '0_99'
    DEFAULT_PERCENTILE_LABEL = [label for percentile, label in get_standard_percentiles(report_type) if percentile == DEFAULT_PERCENTILE][0]
    subst_prefix = 'shadow' if is_shadow_report(report_type) else None
    DEFAULT_HOST_METRIC_MAX_PERCENTILES = ['1']
    DEFAULT_HOST_METRIC_MAX_LABELS = ' + '.join([label for percentile, label in get_host_metric_percentiles(is_max=True) if percentile in DEFAULT_HOST_METRIC_MAX_PERCENTILES])
    DEFAULT_HOST_METRIC_MIN_PERCENTILES = ['0']
    DEFAULT_HOST_METRIC_MIN_LABELS = ' + '.join([label for percentile, label in get_host_metric_percentiles(is_max=False) if percentile in DEFAULT_HOST_METRIC_MIN_PERCENTILES])
    SMOOTH_LABELS = ('No', '15 minutes', '30 minutes', '1 hour', '2 hours', '4 hours', '8 hours', '1 day')
    SMOOTH_VALUES = ['|movingAverage(\'{}\')'.format(v) if v else '' for v in ('', '15min', '30min', '1h', '2h', '4h', '8h', '1d')]
    result = [
        {
            'current': {
                'text': 'one_min',
                'value': 'one_min'
            },
            'options': [
                {
                    'selected': True,
                    'text': 'one_min',
                    'value': 'one_min'
                },
                {
                    'selected': False,
                    'text': 'five_min',
                    'value': 'five_min'
                }
            ],
            'label': 'timings',
            'name': 'i',
            'query': 'one_min,five_min',
            'type': 'custom'
        },
        {
            'current': {
                'text': DEFAULT_PERCENTILE_LABEL,
                'value': DEFAULT_PERCENTILE,
            },
            'options': [
                {
                    'selected': percentile == DEFAULT_PERCENTILE,
                    'text': label,
                    'value': percentile
                }
                for percentile, label in get_standard_percentiles(report_type)
            ],
            'hide': 1,
            'includeAll': True,
            'multi': True,
            'name': 't',
            'query': ','.join(percentile for percentile, _ in get_standard_percentiles(report_type)),
            'type': 'custom'
        },
        {
            'current': {
                'text': UNLIMITED_TIME_NAME if DEFAULT_MAX_TIME == UNLIMITED_TIME_VALUE else MAX_TIME_FORMAT.format(DEFAULT_MAX_TIME),
                'value': DEFAULT_MAX_TIME
            },
            'options': [
                {
                    'selected': value == DEFAULT_MAX_TIME,
                    'text': UNLIMITED_TIME_NAME if value == UNLIMITED_TIME_VALUE else MAX_TIME_FORMAT.format(value),
                    'value': value
                }
                for value in MAX_TIME_VALUES
            ],
            'label': 'max time',
            'name': 'm',
            'query': ','.join(MAX_TIME_VALUES),
            'type': 'custom'
        },
    ]
    if per_host:
        cluster_list = get_cluster_list(report_type, nanny_auth_token)

        def gen_host_selector(host_list):
            return '{' + ','.join([host_name.replace('.', '_') for host_name in host_list]) + '}'

        def gen_short_host_list(cluster_id, host_list):
            return cluster_id + ' [' + host_list[0][:5] + ','.join([host_name[5:9] for host_name in host_list]) + ']'

        if len(cluster_list) > 0:
            result += [
                {
                    'current': {
                        'text': gen_short_host_list(cluster_list[0][0], cluster_list[0][1]),
                        'value': gen_host_selector(cluster_list[0][1])
                    },
                    'options': [
                        {
                            'selected': index == 0,
                            'text': gen_short_host_list(cluster_id, host_list),
                            'value': gen_host_selector(host_list)
                        }
                        for index, (cluster_id, host_list) in enumerate(cluster_list)
                    ],
                    'label': 'cluster',
                    'name': 'host',
                    'query': ','.join([gen_host_selector(host_list) for _, host_list in cluster_list]),
                    'type': 'custom'
                },
                {
                    'current': {
                        'text': 'ALL',
                        'value': 'ALL'
                    },
                    'hide': 2,
                    'name': 'place',
                    'options': [
                        {
                            'selected': True,
                            'text': 'ALL',
                            'value': 'ALL'
                        }
                    ],
                    'query': 'ALL',
                    'type': 'constant'
                }
            ]
    else:
        result += [
            {
                'allValue': '[!A]*',
                'current': {
                    'text': ALL_PLACES,
                    'value': ALL_PLACES
                },
                'options': [
                    {
                        'selected': False,
                        'text': 'All',
                        'value': '$__all'
                    }
                ] + [
                    {
                        'selected': place == ALL_PLACES,
                        'text': place,
                        'value': place
                    }
                    for place in PLACES
                ],
                'includeAll': True,
                'multi': True,
                'name': 'place',
                'query': ','.join(PLACES),
                'type': 'custom'
            },
        ]
        if is_shadow_report(report_type):
            SHADOW_LOCATIONS = REPORT_TYPES[report_type]['shadow_loc']
            DEFAULT_SHADOW_LOCATION = SHADOW_LOCATIONS[0]
            result += [
                {
                    'current': {
                        'text': DEFAULT_SHADOW_LOCATION,
                        'value': DEFAULT_SHADOW_LOCATION,
                    },
                    'options': [
                        {
                            'selected': shadow_location == DEFAULT_SHADOW_LOCATION,
                            'text': shadow_location,
                            'value': shadow_location
                        }
                        for shadow_location in SHADOW_LOCATIONS
                    ],
                    'label': 'shadow dc',
                    'name': 'loc',
                    'query': ','.join(SHADOW_LOCATIONS),
                    'type': 'custom'
                },
            ]
        else:
            result += [
                {
                    'current': {
                        'text': 'All',
                        'value': '$__all'
                    },
                    'datasource': 'market',
                    'includeAll': True,
                    'label': 'environment',
                    'multi': True,
                    'name': 'env',
                    'query': subst_tmpl(report_type, '$i.$report_table.rps.env.*'),
                    'refresh': 1,
                    'sort': 0,
                    'type': 'query'
                },
                {
                    'allValue': '[!A]*',
                    'current': {
                        'text': 'All',
                        'value': '$__all'
                    },
                    'datasource': 'market',
                    'includeAll': True,
                    'label': 'dc',
                    'multi': True,
                    'name': 'loc',
                    'query': subst_tmpl(report_type, '$i.$report_table.rps.env.$env.loc.*'),
                    'refresh': 1,
                    'sort': 0,
                    'type': 'query'
                },
                {
                    'allValue': '[0-9]*',
                    'current': {
                        'text': 'All',
                        'value': '$__all'
                    },
                    'datasource': 'market',
                    'includeAll': True,
                    'multi': True,
                    'name': 'cluster',
                    'query': subst_tmpl(report_type, '$i.$report_table.rps.env.$env.loc.$loc.cluster.*'),
                    'refresh': 1,
                    'sort': 3,
                    'type': 'query'
                },
            ]
        result += [
            {
                'allValue': '[0-9]*',
                'current': {
                    'text': 'ALL',
                    'value': [
                        'ALL'
                    ]
                },
                'datasource': 'market',
                'includeAll': True,
                'label': 'error codes',
                'multi': True,
                'name': 'error_code',
                'query': subst_tmpl(report_type, 'one_min.$error_table.errors.env.production.loc.ALL.cluster.ALL.code.*', subst_prefix),
                'refresh': 1,
                'sort': 3,
                'type': 'query'
            },
        ]
        if not per_host:
            result += [
                {
                    'allValue': '[!2]*',
                    'current': {
                        'text': '499',
                        'value': [
                            '499'
                        ]
                    },
                    'datasource': 'market',
                    'includeAll': True,
                    'label': 'status codes',
                    'multi': True,
                    'name': 'status',
                    'query': subst_tmpl(report_type, 'one_min.$nginx_table.status_codes.env.production.loc.ALL.cluster.ALL.status.*', subst_prefix),
                    'refresh': 1,
                    'sort': 3,
                    'type': 'query'
                },
            ]
        if not is_shadow_report(report_type):
            LABEL_FORMATS = (
                (
                    'environment',
                    r'\1'
                ),
                (
                    'location',
                    r'\2'
                ),
                (
                    'cluster',
                    r'\3'
                ),
                (
                    'place/percentile',
                    r'\4'
                ),
            )
            DEFAULT_LABEL_INDICES = (0, 1, 2)
            result += [
                {
                    'current': {
                        'text': ' + '.join(LABEL_FORMATS[index][0] for index in DEFAULT_LABEL_INDICES),
                        'value': [LABEL_FORMATS[index][1] for index in DEFAULT_LABEL_INDICES]
                    },
                    'label': 'label',
                    'multi': True,
                    'name': 'label',
                    'options': [
                        {
                            'selected': index in DEFAULT_LABEL_INDICES,
                            'text': label_format[0],
                            'value': label_format[1]
                        }
                        for index, label_format in enumerate(LABEL_FORMATS)
                    ],
                    'query': ','.join([label_format[1] for label_format in LABEL_FORMATS]),
                    'type': 'custom'
                },
            ]
        result += [
            {
                'current': {
                    'text': DEFAULT_HOST_METRIC_MIN_LABELS,
                    'value': DEFAULT_HOST_METRIC_MIN_PERCENTILES,
                },
                'options': [
                    {
                        'selected': percentile in DEFAULT_HOST_METRIC_MIN_PERCENTILES,
                        'text': label,
                        'value': percentile
                    }
                    for percentile, label in get_host_metric_percentiles(is_max=False)
                ],
                'includeAll': True,
                'label': 'min',
                'multi': True,
                'name': 'min_p',
                'query': ','.join(percentile for percentile, _ in get_host_metric_percentiles(is_max=False)),
                'type': 'custom'
            },
            {
                'current': {
                    'text': DEFAULT_HOST_METRIC_MAX_LABELS,
                    'value': DEFAULT_HOST_METRIC_MAX_PERCENTILES,
                },
                'options': [
                    {
                        'selected': percentile in DEFAULT_HOST_METRIC_MAX_PERCENTILES,
                        'text': label,
                        'value': percentile
                    }
                    for percentile, label in get_host_metric_percentiles(is_max=True)
                ],
                'includeAll': True,
                'label': 'max',
                'multi': True,
                'name': 'max_p',
                'query': ','.join(percentile for percentile, _ in get_host_metric_percentiles(is_max=True)),
                'type': 'custom'
            },
        ]
    result += [
        {
            'current': {
                'text': SMOOTH_LABELS[0],
                'value': SMOOTH_VALUES[0]
            },
            'name': 'smooth',
            'options': [
                {
                    'selected': label == SMOOTH_LABELS[0],
                    'text': label,
                    'value': value
                }
                for label, value in zip(SMOOTH_LABELS, SMOOTH_VALUES)
            ],
            'query': ','.join(SMOOTH_VALUES),
            'type': 'custom'
        },
    ]
    return {
        'list': result
    }


def make_row(name, y, collapsed, panel_id):
    return {
        'collapsed': collapsed,
        'gridPos': {
            'h': ROW_HEIGHT,
            'w': ROW_WIDTH,
            'x': 0,
            'y': y
        },
        'id': panel_id,
        'title': name,
        'type': 'row'
    }


def is_shadow_report(report_type):
    return 'shadow' in report_type


def make_attention(text, y, panel_id):
    return {
        "fieldConfig": {
            "defaults": {
                "custom": {}
            },
            "overrides": []
        },
        "gridPos": {
            "h": ROW_HEIGHT * 3,
            "w": ROW_WIDTH,
            "x": 0,
            "y": y
        },
        'id': panel_id,
        "options": {
            "content": text,
            "mode": "markdown"
        },
        "title": "Attention",
        "type": "text"
    }


def make_panels(report_type, per_host):
    rows = list()
    x = 0
    y = 0
    row_height = 0
    collapsed = True
    existing_panel_ids = set()
    for metric_group in gen_metric_groups(report_type, per_host):
        for metric in metric_group['metrics']:
            panel_id = metric['panel_id']
            if panel_id in existing_panel_ids:
                raise Exception('Duplicate panel id: {}'.format(panel_id))
            existing_panel_ids.add(panel_id)
    unique_panel_id = max(existing_panel_ids) + 1

    if 'attention' in REPORT_TYPES[report_type]:
        rows.append(make_attention(REPORT_TYPES[report_type].get('attention'), y, unique_panel_id))
        y += ROW_HEIGHT * 3
        unique_panel_id += 1

    for metric_group in gen_metric_groups(report_type, per_host):
        visible_metrics = [metric for metric in metric_group['metrics'] if 'visible' not in metric or metric['visible']]
        if not visible_metrics:
            continue
        row = make_row(metric_group['name'], y, collapsed, unique_panel_id)
        unique_panel_id += 1
        y += ROW_HEIGHT
        panels = list()
        for metric in visible_metrics:
            graph_width = metric['width']
            graph_height = metric['height']
            if x + graph_width > ROW_WIDTH:
                x = 0
                y += row_height
                row_height = 0
            panels.append(make_graph(metric, report_type, per_host, x, y, graph_width, graph_height))
            x += graph_width
            row_height = max(row_height, graph_height)
        if x != 0:
            x = 0
            y += graph_height
        if collapsed:
            row['panels'] = panels
            rows.append(row)
        else:
            rows.append(row)
            rows.extend(panels)
    return rows


def make_annotations(report_type):
    annotations = [
        {
            'builtIn': 1,
            'datasource': '-- Grafana --',
            'enable': False,
            'hide': True,
            'iconColor': '#70dbed',
            'name': 'Notes',
            'type': 'dashboard'
        },
    ]
    annotations += [
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#447ebc',
            'name': 'Front',
            'tags': 'nanny-resource:MARKET_DESKTOP_FRONT',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#ba43a9',
            'name': 'Touch',
            'tags': 'nanny-resource:MARKET_TOUCH_FRONT',
            'type': 'tags'
        },
    ]
    annotations += [
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#64b0c8',
            'name': 'Report',
            'tags': 'package:yandex-market-report',
            'type': 'tags'
        },
    ]
    annotations += [
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#629e51',
            'name': 'Exp',
            'tags': 'Type: exp',
            'type': 'tags'
        },
    ]
    annotations += [
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#e5ac0e',
            'name': 'Стоп-Кран',
            'tags': 'emergency_break',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#b877d9',
            'name': 'flags.json',
            'tags': 'exp:flags_json',
            'type': 'tags'
        },
    ]
    annotations += [
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#ff9830',
            'name': 'Push',
            'tags': 'type:mcrm-push',
            'type': 'tags'
        },
    ]
    return annotations


def make_links(report_type, per_host):
    links = [
        {
            'asDropdown': True,
            'icon': 'external link',
            'keepTime': True,
            'tags': [
                'market_report_global_metrics'
            ],
            'title': 'Global',
            'type': 'dashboards'
        },
        {
            'asDropdown': True,
            'icon': 'external link',
            'keepTime': True,
            'tags': [
                'market_report_cluster_metrics'
            ],
            'title': 'Cluster',
            'type': 'dashboards'
        },
        {
            'asDropdown': True,
            'icon': 'external link',
            'keepTime': True,
            'tags': [
                'market_report_host_metrics'
            ],
            'title': 'Host',
            'type': 'dashboards'
        },
    ]
    links += [
        {
            'icon': 'dashboard',
            'keepTime': True,
            'title': 'Front',
            'type': 'link',
            'url': 'https://graf.yandex-team.ru/dashboard/db/marketfront-pulse'
        },
        {
            'icon': 'dashboard',
            'keepTime': True,
            'title': 'Touch',
            'type': 'link',
            'url': 'https://grafana.yandex-team.ru/dashboard/db/market-front-touch-pulse'
        },
    ]
    links += [
        {
            'icon': 'dashboard',
            'keepTime': True,
            'title': 'AppHost',
            'type': 'link',
            'url': 'https://yasm.yandex-team.ru/panel/_LG58QP'
        },
    ]
    links += [
        {
            'icon': 'dashboard',
            'keepTime': True,
            'title': 'ES',
            'type': 'link',
            'url': 'https://yasm.yandex-team.ru/panel/es_report_{}_avg_time'.format(report_type.replace('-', '_'))
        },
    ]
    links += [
        {
            'icon': 'external link',
            'tags': [],
            'title': 'Generator',
            'type': 'link',
            'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/market/report/GeneratePerHostDashboards/cluster_metrics_dashboard.py'
        }
    ]
    if per_host:
        links += [
            {
                'icon': 'external link',
                'title': 'Sandbox Scheduler',
                'type': 'link',
                'url': 'https://sandbox.yandex-team.ru/scheduler/12325'
            }
        ]
    return links


def make_dashboard(report_type, per_host, nanny_auth_token, publish):
    return {
        'annotations': {'list': make_annotations(report_type)},
        'graphTooltip': 1,
        'links': make_links(report_type, per_host),
        'panels': make_panels(report_type, per_host),
        'refresh': '1m',
        'schemaVersion': 16,
        'tags': [
            'market_report',
            'market_report_host_metrics' if per_host else 'market_report_cluster_metrics',
        ] if publish else [],
        'time': {
            'from': 'now-2d',
            'to': 'now'
        },
        'templating': make_templating(report_type, per_host, nanny_auth_token),
        'title': '{title} Report [{sub_title}] [autogenerated on {date}]'.format(
            title=REPORT_TYPES[report_type]['title'],
            sub_title='Hosts' if per_host else 'Clusters',
            date=datetime.datetime.now().strftime('%Y-%m-%d %H:%M'),
        ),
        'uid': REPORT_TYPES[report_type]['PER_HOST_GRAFANA_UID' if per_host else 'GRAFANA_UID']
    }


@retry()
def publish_dashboard(dashboard, report_type, per_host, grafana_auth_token):
    headers = {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': 'OAuth ' + grafana_auth_token
    }
    body = {
        'dashboard': dashboard,
        'folderId': 0,
        'overwrite': True
    }
    conn = httplib.HTTPSConnection('grafana.yandex-team.ru', timeout=10)
    conn.connect()
    try:
        # conn.request('DELETE', '/api/dashboards/uid/{uid}'.format(
        #     uid=REPORT_TYPES[report_type]['PER_HOST_GRAFANA_UID' if per_host else 'GRAFANA_UID']
        # ), headers=headers)
        # conn.getresponse().read()
        conn.request('POST', '/api/dashboards/db', json.dumps(body), headers)
        response = conn.getresponse()
        response_data = response.read()
        if response.status != 200:
            raise Exception('Grafana API returned error: {}\n{}'.format(response.status, response_data))
    finally:
        conn.close()
    return response_data


def get_grafana_token():
    auth_token = os.environ.get('GRAFANA_AUTH_TOKEN')
    if auth_token:
        return auth_token
    token_file_path = os.path.expanduser('~/.robot-market-st.tokens')
    if not os.path.isfile(token_file_path):
        return None
    with open(token_file_path, 'r') as token_file:
        tokens = json.loads(token_file.read())
        return tokens.get('grafana')


def get_nanny_group_list(auth_token):
    group_list = list()
    nanny_data = send_nanny_api_request('/v2/services/?exclude_runtime_attrs=1&category=/market/report', auth_token)
    for group_info in nanny_data['result']:
        group_name = group_info['_id']
        group_list.append(group_name)
    logging.info('Found %d Nanny groups', len(group_list))
    return group_list


@retry()
def send_nanny_api_request(query, auth_token=None):
    headers = dict()
    headers['Accept'] = 'application/json'
    if auth_token:
        headers['Authorization'] = 'OAuth {0}'.format(auth_token)
    conn = httplib.HTTPConnection('nanny.yandex-team.ru')
    conn.request('GET', query, headers=headers)
    try:
        response = conn.getresponse()
        if response.status != 200:
            if response.status == 404:
                return None
            raise Exception('Nanny API returned error: {} {}\n{}'.format(response.status, response.reason, query))
        json_str = response.read()
    finally:
        conn.close()
    return json.loads(json_str)


def get_nanny_token():
    auth_token = os.environ.get('NANNY_AUTH_TOKEN')
    if auth_token:
        return auth_token
    token_file_path = os.path.expanduser('~/.robot-market-st.tokens')
    if not os.path.isfile(token_file_path):
        return None
    with open(token_file_path, 'r') as token_file:
        tokens = json.loads(token_file.read())
        return tokens.get('nanny')


def parse_nanny_group_name(group_name):
    parts = group_name.split('_')
    if len(parts) < 4 or parts[1] != 'report':
        return None
    is_exp = parts[-2] == 'exp1'
    env_type = 'exp' if is_exp else parts[0]
    dc = parts[-1]
    is_snippet = parts[-2] == 'snippet'
    report_type = '_'.join(parts[2:-2 if is_snippet or is_exp else -1])
    return (env_type, report_type, dc, is_snippet)


def get_host_list(group_name):
    logging.info('Processing Nanny group %s', group_name)
    nanny_data = send_nanny_api_request('/v2/services/{0}/current_state/instances/'.format(group_name))
    if nanny_data is None:
        return list()
    shard_matcher = re.compile(r'^a_shard_(\d+)$')
    replica_matcher = re.compile(r'^itag_replica_(\d+)$')
    host_list = list()
    for host_info in nanny_data['result']:
        cluster_index = None
        host_index = None
        for itag in host_info['itags']:
            shard_match = shard_matcher.match(itag)
            if shard_match:
                cluster_index = int(shard_match.group(1))
            replica_match = replica_matcher.match(itag)
            if replica_match:
                host_index = int(replica_match.group(1))
        if cluster_index is None or host_index is None:
            continue
        host_name = host_info['container_hostname']
        host_list.append((cluster_index, host_index, host_name))
    logging.info('Found %d hosts in group %s', len(host_list), group_name)
    return host_list


def gen_cluster_id(env_type, report_type, dc, cluster_index):
    return '{env_type}@{report_type}@{dc}@{cluster_index:02}'.format(
        env_type=env_type, report_type=report_type, dc=dc, cluster_index=cluster_index
    )


def get_cluster_list(report_type, nanny_auth_token):
    nanny_report_type = report_type.replace('-', '_')
    if nanny_report_type == 'bk':
        nanny_report_type = 'parallel'
    nanny_group_list = get_nanny_group_list(nanny_auth_token)
    host_list = list()
    for group_name in nanny_group_list:
        if not parse_nanny_group_name(group_name):
            continue
        env_type, group_report_type, dc, _ = parse_nanny_group_name(group_name)
        if group_report_type == nanny_report_type:
            for cluster_index, host_index, host_name in get_host_list(group_name):
                host_list.append(((env_type, group_report_type, dc, cluster_index), host_index, host_name))
    clusters = list()
    prev_cluster_info = None
    for cluster_info, _, host_name in sorted(host_list):
        if cluster_info != prev_cluster_info:
            clusters.append((gen_cluster_id(*cluster_info), list()))
            prev_cluster_info = cluster_info
        clusters[-1][1].append(host_name)
    for cluster in clusters:
        logging.info('Cluster %s hosts: %s', cluster[0], ','.join(cluster[1]))
    return clusters


def main():
    arg_parser = argparse.ArgumentParser(description='Generate report per-cluster or per-host metric dashboard for Grafana.')
    arg_parser.add_argument('--report-type', choices=REPORT_TYPES.keys())
    arg_parser.add_argument('--per-host', action='store_true', help='Create per-host dashboard')
    arg_parser.add_argument('--publish', action='store_true', help='Publish dashboard via Grafana API')
    args = arg_parser.parse_args()
    nanny_auth_token = None
    if args.per_host:
        nanny_auth_token = get_nanny_token()
        if not nanny_auth_token:
            print '''Obtain Nanny auth token here: https://nanny.yandex-team.ru/ui/#/oauth/
Set environment variable NANNY_AUTH_TOKEN=<token>
Another way is to get .robot-market-st.tokens from your colleague with 'nanny' token in it.'''
            return
    if args.publish:
        grafana_auth_token = get_grafana_token()
        if not grafana_auth_token:
            print '''Obtain Grafana auth token here: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=9590d99772e74c7386610688d4a5c1b3
Examine Response Headers->Location in browser network debugger, it is going to look like this: https://grafana.yandex-team.ru#access_token=<token>&token_type=bearer
Set environment variable GRAFANA_AUTH_TOKEN=<token>
Another way is to get .robot-market-st.tokens from your colleague with 'grafana' token in it.'''
            return
    if args.report_type:
        report_types = (args.report_type,)
    else:
        report_types = REPORT_TYPES.keys()
    for report_type in report_types:
        dashboard = make_dashboard(report_type, args.per_host, nanny_auth_token, args.publish)
        if args.publish:
            print publish_dashboard(dashboard, report_type, args.per_host, grafana_auth_token)
        else:
            print json.dumps(dashboard, indent=2)


if __name__ == '__main__':
    main()
