#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import datetime
import functools
import httplib
import json
import logging
import os
import re
import time

GRAPH_WIDTH = 6
GRAPH_HEIGHT = 8
ROW_HEIGHT = 1
ROW_WIDTH = 24

CONDUCTOR_GROUPS = ['market_search-full-perf-testing']

SMOOTH_LEVELS = [
    ('No', ''),
    ('15 minutes', '|movingAverage("15min")'),
    ('30 minutes', '|movingAverage("30min")'),
    ('1 hour', '|movingAverage("1h")'),
    ('4 hours', '|movingAverage("4h")'),
    ('1 day', '|movingAverage("1d")'),
]

KNOWN_METRIC_GROUPS = [
    {
        'name': 'System Memory',
        'metrics': [
            ('porto_stats.memory_left', 'min', 'bytes', 1),
            ('porto_stats.memory_limit_total', 'max', 'bytes', 2),
            ('porto_stats.cache_usage', 'min', 'bytes', 7),
            ('porto_stats.memory_free', 'min', 'bytes', 155),
            ('porto_stats.mlock_usage', 'max', 'bytes', 156),
            ('memory_stats.projected_memory_reserve', 'min', 'bytes', 159),
        ]
    },
    {
        'name': 'Report Memory',
        'metrics': [
            ('memory_stats.report_rss_anon', 'max', 'bytes', 10),
            ('memory_stats.report_rss_file', 'max', 'bytes', 11),
            ('memory_stats.report_rss_shmem', 'max', 'bytes', 12),
            ('report_internals.balloc_cur_size', 'max', 'bytes', 13),
            ('report_internals.balloc_gc_size', 'max', 'bytes', 14),
            ('report_internals.balloc_in_use', 'max', 'bytes', 15),
            ('report_internals.anonymous_memory_consumption_after_global', 'max', 'bytes', 16),
            ('report_internals.lf_total_size', 'max', 'bytes', 152),
            ('report_internals.lf_cache_size', 'max', 'bytes', 153),
            ('report_internals.lf_alloc_size', 'max', 'bytes', 19),
            ('report_internals.index_memory_lock_quota_usage', 'max', 'bytes', 148),
            ('report_internals.rty_index_memory_lock_quota_usage', 'max', 'bytes', 147),
            ('report_process_stats.anonymous_mapping_count', 'max', 'short', 149),
            ('report_process_stats.index_cache_usage', 'min', 'bytes', 151),
        ]
    },
    {
        'name': 'Disk Usage',
        'metrics': [
            ('disk_usage.report_data_size', 'max', 'bytes', 27),
            ('disk_usage.dynamic_models_size', 'max', 'bytes', 28),
            ('disk_usage.formulas_size', 'max', 'bytes', 29),
            ('disk_usage.generations_size', 'max', 'bytes', 30),
            ('disk_usage.packages_size', 'max', 'bytes', 32),
            ('disk_usage.pdata_free_space', 'min', 'bytes', 33),
            ('disk_usage.index_free_space', 'min', 'bytes', 157),
        ]
    },
    {
        'name': 'Block Device Stats',
        'metrics': [
            ('block_dev_stats.pdata.read_bytes', 'max', 'Bps', 99),
            ('block_dev_stats.pdata.write_bytes', 'max', 'Bps', 100),
            ('block_dev_stats.pdata.average_read_wait', 'max', 's', 101),
            ('block_dev_stats.pdata.average_write_wait', 'max', 's', 102),
            ('block_dev_stats.ssd.read_bytes', 'max', 'Bps', 104),
            ('block_dev_stats.ssd.write_bytes', 'max', 'Bps', 105),
            ('block_dev_stats.ssd.average_read_wait', 'max', 's', 106),
            ('block_dev_stats.ssd.average_write_wait', 'max', 's', 107),
            ('block_dev_stats.pdata.io_usage', 'max', 'percentunit', 103),
            ('block_dev_stats.ssd.io_usage', 'max', 'percentunit', 108),
        ]
    },
    {
        'name': 'Network Throughput',
        'metrics': [
            ('net_stats.bytes_received', 'max', 'Bps', 35),
            ('net_stats.bytes_sent', 'max', 'Bps', 36),
            ('net_stats.packets_received', 'max', 'pps', 37),
            ('net_stats.packets_sent', 'max', 'pps', 38)
        ]
    },
    {
        'name': 'System Network Stats',
        'metrics': [
            ('net_stats.retrans_segs', 'max', 'ops', 39),
            ('net_stats.curr_estab', 'max', 'short', 40),
            ('net_stats.active_opens', 'max', 'ops', 41),
            ('net_stats.passive_opens', 'max', 'ops', 42),
            ('net_stats.attempt_fails', 'max', 'ops', 43),
            ('net_stats.estab_resets', 'max', 'ops', 44),
            ('net_stats.in_errs', 'max', 'ops', 45),
            ('net_stats.out_rsts', 'max', 'ops', 46),
        ]
    },
    {
        'name': 'Report Network Stats',
        'metrics': [
            ('report_internals.neh_http_output_connection_count', 'max', 'short', 113),
            ('report_internals.net_connect_count', 'max', 'ops', 128),
            ('report_internals.net_accept_count', 'max', 'ops', 129),
            ('report_internals.net_cluster_peer_recv_bytes', 'max', 'Bps', 130),
            ('report_internals.net_cluster_peer_send_bytes', 'max', 'Bps', 131),
            ('report_internals.net_log_broker_recv_bytes', 'max', 'Bps', 132),
            ('report_internals.net_log_broker_send_bytes', 'max', 'Bps', 133),
            ('report_internals.net_other_incoming_recv_bytes', 'max', 'Bps', 134),
            ('report_internals.net_other_incoming_send_bytes', 'max', 'Bps', 135),
            ('report_internals.net_other_outgoing_recv_bytes', 'max', 'Bps', 136),
            ('report_internals.net_other_outgoing_send_bytes', 'max', 'Bps', 137),
        ],
    },
    {
        'name': 'Startup Time',
        'metrics': [
            ('report_internals.startup_time', 'max', 'ms', 51),
            ('report_internals.rty_startup_time', 'max', 'ms', 52),
            ('report_internals.access_startup_time', 'max', 'ms', 154),
            ('report_internals.global_startup_time', 'max', 'ms', 53),
        ]
    },
    {
        'name': 'Report Process Stats',
        'metrics': [
            ('report_process_stats.utime', 'max', 'percentunit', 54),
            ('report_process_stats.stime', 'max', 'percentunit', 55),
            ('report_process_stats.majflt', 'max', 'ops', 56),
            ('report_process_stats.minflt', 'max', 'ops', 57),
            ('report_process_stats.runqueue_wait_time', 'max', 'percentunit', 143),
            ('report_process_stats.nonvoluntary_ctxt_switches', 'max', 'ops', 58),
            ('report_process_stats.voluntary_ctxt_switches', 'max', 'ops', 59),
            ('report_process_stats.num_threads', 'max', 'short', 64),
        ]
    },
    {
        'name': 'Report Process I/O',
        'metrics': [
            ('report_process_stats.read_bytes', 'max', 'Bps', 60),
            ('report_process_stats.write_bytes', 'max', 'Bps', 61),
        ]
    },
    {
        'name': 'Report Process Stats 2',
        'metrics': [
            ('report_internals.meta_queue_length', 'max', 'short', 65),
            ('report_internals.max_base_queue_length', 'max', 'short', 66),
            ('report_internals.meta_active_thread_count', 'max', 'short', 67),
            ('report_internals.base_active_thread_count', 'max', 'short', 68),
            ('report_internals.relevance_calcer_active_thread_count', 'max', 'short', 69),
            ('report_internals.concurrent_pool_thread_count', 'max', 'short', 115),
        ]
    },
    {
        'name': 'System Stats',
        'metrics': [
            ('cpu_stats.context_switches', 'max', 'ops', 70),
            ('cpu_stats.processes_created', 'max', 'ops', 71),
            ('cpu_stats.processes_running', 'max', 'short', 72),
            ('cpu_stats.processes_blocked', 'max', 'short', 73),
            ('cpu_stats.interrupts', 'max', 'ops', 74),
            ('cpu_stats.soft_irqs', 'max', 'ops', 75),
            ('system_stats.kernel_version', 'max', 'none', 76),
            ('cpu_stats.uptime', 'max', 's', 77),
        ]
    },
    {
        'name': 'CPU Stats',
        'metrics': [
            ('cpu_stats.max_freq', 'max', 'hertz', 78),
            ('cpu_stats.avg_freq', 'max', 'hertz', 79),
            ('cpu_stats.cpu_usage', 'max', 'percentunit', 80),
            ('cpu_stats.user_time', 'max', 'percentunit', 81),
            ('cpu_stats.system_time', 'max', 'percentunit', 82),
            ('cpu_stats.nice_time', 'max', 'percentunit', 109),
        ]
    },
    {
        'name': 'Version & Index',
        'metrics': [
            ('report_versions.consistent_index_generation', 'max', 'none', 83),
            ('report_versions.consistent_report_version', 'max', 'none', 84),
            ('report_versions.indexer_version', 'max', 'none', 88),
            ('report_versions.dssm_version', 'max', 'none', 89),
            ('report_versions.formulas_version', 'max', 'none', 90),
            ('report_versions.index_age', 'max', 's', 158),
        ]
    },
    {
        'name': 'Documents',
        'metrics': [
            ('report_versions.offer_count', 'max', 'locale', 141),
            ('report_versions.blue_offer_count', 'max', 'locale', 150),
            ('rty_server.searchable_docs', 'max', 'locale', 142),
        ]
    },
    {
        'name': 'Report Status',
        'metrics': [
            ('report_versions.report_status', 'max', 'none', 138),
        ]
    },
    {
        'name': 'Core Dumps & OOM Kills',
        'metrics': [
            ('core_dump.report_core_count', 'sum', 'short', 91),
            ('core_dump.count', 'sum', 'short', 92),
            ('porto_stats.oom_kills', 'sum', 'short', 93)
        ]
    },
    {
        'name': 'External Requests Cache',
        'metrics': [
            ('report_internals.external_request_history_cache_hit_rate', 'max', 'percentunit', 111),
            ('report_internals.external_request_history_remote_cache_hit_rate', 'max', 'percentunit', 112),
        ]
    },
    {
        'name': 'Index Files',
        'metrics': [
            ('report_files.index_part_1.*', 'max', 'bytes', 94),
            ('report_files.index_part_model.*', 'max', 'bytes', 96),
            ('report_files.index_part_book.*', 'max', 'bytes', 116),
            ('report_files.index_part_cards.*', 'max', 'bytes', 117),
            ('report_files.mmap.*', 'max', 'bytes', 97),
            ('report_files.report-data.*', 'max', 'bytes', 98)
        ],
        'graph_width': ROW_WIDTH / 3,
        'graph_height': 12
    },
]


MEM_DASHBOARD_METRIC_GROUPS = [
    {
        'name': 'Offers and Memory',
        'metrics': [
            ('porto_stats.memory_limit_total', 'max', 'bytes', 1),
            ('porto_stats.memory_left', 'min', 'bytes', 2),
            ('report_versions.offer_count', 'max', 'locale', 3),
            ('report_files.index_part_1.total', 'max', 'bytes', 4),
            ('report_files.index_part_blue.total', 'max', 'bytes', 5),
            {
                'title': 'offfer count / memory used',
                'metrics': {
                    'memory_limit': ('porto_stats.memory_limit_total', 'max'),
                    'memory_left': ('porto_stats.memory_left', 'min'),
                    'offer_count': ('report_versions.offer_count', 'max'),
                },
                'expression': 'scale(divideSeries({{offer_count}}, diffSeries({{memory_limit}}, movingWindow({{memory_left}}, "2min", "min"))), {factor})'.format(factor=2**32),
                'units': 'locale',
                'label': 'offers per Gb',
                'panel_id': 6
            },
            {
                'title': 'offfer count / offer index size',
                'metrics': {
                    'index_size': ('report_files.index_part_1.total', 'max'),
                    'offer_count': ('report_versions.offer_count', 'max'),
                },
                'expression': 'scale(divideSeries({{offer_count}}, {{index_size}}), {factor})'.format(factor=2**32),
                'units': 'locale',
                'label': 'offers per Gb',
                'panel_id': 7
            },
            {
                'title': 'blue offfer count / blue offer index size',
                'metrics': {
                    'blue_index_size': ('report_files.index_part_blue.total', 'max'),
                    'blue_offer_count': ('report_versions.blue_offer_count', 'max'),
                },
                'expression': 'scale(divideSeries({{blue_offer_count}}, {{blue_index_size}}), {factor})'.format(factor=2**32),
                'units': 'locale',
                'label': 'offers per Gb',
                'panel_id': 8
            },
        ]
    },
]


def make_row(name, y, panel_id):
    return {
        'collapsed': True,
        'gridPos': {
            'h': ROW_HEIGHT,
            'w': ROW_WIDTH,
            'x': 0,
            'y': y
        },
        'id': panel_id,
        'title': name,
        'type': 'row'
    }


def make_metric_expression(metric, agg, single_host):
    if single_host:
        metric_sub_exp = 'one_min.HOST.$host.' + metric
        metric_exp = metric_sub_exp + r'[[smooth]]|aliasSub(".+\.HOST\.(.{9}).+", "\1")'
    else:
        metric_id = metric.replace('.', '_')
        metric_exp = 'one_min.report_metrics.{agg}.$host_type.env.$env.sub_role.$sub_role.location.$location.cluster.$cluster.metric.{metric_id}'
        if agg != 'sum':
            metric_exp += '.1'
        metric_exp = metric_exp.format(agg=agg, metric_id=metric_id)
        return metric_exp


def make_graph_for_compound_expression(metric_dict, x, y, w, h, single_host):
    exp_dict = dict()
    for key, entry in metric_dict['metrics'].iteritems():
        metric, agg = entry
        exp_dict[key] = make_metric_expression(metric, agg, single_host)
    metric_exp = metric_dict['expression'].format(**exp_dict)
    metric_exp += '|removeEmptySeries()[[smooth]]|alias("{label}")'.format(label=metric_dict['label'])
    graph = {
        'datasource': 'market',
        'fill': 0,
        'gridPos': {
            'h': h,
            'w': w,
            'x': x,
            'y': y
        },
        'id': metric_dict['panel_id'],
        'maxDataPoints': '2000',
        'targets': [
            {
                'refId': 'A',
                'target': metric_exp
            }
        ],
        'title': metric_dict['title'],
        'tooltip': {
            'sort': 2
        },
        'type': 'graph',
        'yaxes': [
            {
                'format': metric_dict['units'],
                'show': True
            },
            {
                'format': metric_dict['units'],
                'show': True
            }
        ],
    }
    return graph


def make_graph(metric, x, y, w, h, agg, units, single_host, panel_id):
    multi_graph = '*' in metric
    if single_host:
        metric_sub_exp = 'one_min.HOST.$host.' + metric
        if multi_graph:
            metric_exp = metric_sub_exp + r'|removeEmptySeries()[[smooth]]|aliasSub("^.+{}(.+)$", "\1")|sortByName(True)'.format(metric.rstrip('*'))
        else:
            metric_exp = metric_sub_exp + r'[[smooth]]|aliasSub(".+\.HOST\.(.{9}).+", "\1")'
    else:
        metric_id = metric.replace('.', '_')
        metric_exp = 'one_min.report_metrics.{agg}.$host_type.env.$env.sub_role.$sub_role.location.$location.cluster.$cluster.metric.{metric_id}'
        if multi_graph:
            label = r'.*\.([^.]+)\.env\.([^.]+)\.sub_role\.([^.]+)\.location\.([^.]+)\.cluster\.([^.]+)\.metric\.{metric}([^.]*).*'.format(metric=metric_id.rstrip('*'))
            if agg != 'sum':
                metric_exp += '.1'
        elif agg == 'sum':
            label = r'.*\.([^.]+)\.env\.([^.]+)\.sub_role\.([^.]+)\.location\.([^.]+)\.cluster\.([^.]+)\.metric\.([^,]*).*'
        else:
            label = r'.*\.([^.]+)\.env\.([^.]+)\.sub_role\.([^.]+)\.location\.([^.]+)\.cluster\.([^.]+)\.metric\.[^.]+\.([^,]+).*'
            metric_exp += '.$p'
        metric_exp += '|removeEmptySeries()[[smooth]]|aliasSub("{label}", "${{label:csv}}")|sortByName(True)'
        metric_exp = metric_exp.format(agg=agg, metric_id=metric_id, label=label)
    graph = {
        'datasource': 'market',
        'fill': 0,
        'gridPos': {
            'h': h,
            'w': w,
            'x': x,
            'y': y
        },
        'id': panel_id,
        'maxDataPoints': '2000',
        'targets': [
            {
                'refId': 'A',
                'target': metric_exp
            }
        ],
        'title': metric,
        'tooltip': {
            'sort': 2
        },
        'type': 'graph',
        'yaxes': [
            {
                'format': units,
                'show': True
            },
            {
                'format': units,
                'show': True
            }
        ],
    }
    if multi_graph:
        graph['seriesOverrides'] = [
            {
                'alias': 'total',
                'yaxis': 2
            }
        ]
    return graph


def make_templating(single_host, nanny_auth_token):
    var_list = list()
    if single_host:
        def gen_host_value(host_info):
            return host_info[5].replace('.', '_')

        def gen_host_text(host_info):
            env_type, report_type, dc, cluster_index, host_index, host_name = host_info
            return '{env_type}@{report_type}@{dc}@{cluster_index:02}@{host_index} {host_name}'.format(
                env_type=env_type, report_type=report_type, dc=dc, cluster_index=cluster_index, host_index=host_index, host_name=host_name[:9]
            )

        host_list = get_host_list(nanny_auth_token) + get_hp_host_list()

        var_list.append(
            {
                'current': {
                    'text': gen_host_text(host_list[0]),
                    'value': gen_host_value(host_list[0])
                },
                'options': [
                    {
                        'selected': index == 0,
                        'text': gen_host_text(host_info),
                        'value': gen_host_value(host_info)
                    }
                    for index, host_info in enumerate(host_list)
                ],
                'multi': True,
                'name': 'host',
                'query': ','.join([host_info[5] for host_info in host_list]),
                'type': 'custom'
            }
        )
    else:
        var_list.extend(
            [
                {
                    'current': {
                        'text': 'report',
                        'value': 'report'
                    },
                    'datasource': 'market',
                    'includeAll': True,
                    'label': 'host type',
                    'multi': True,
                    'name': 'host_type',
                    'query': 'one_min.report_metrics.max.*',
                    'refresh': 1,
                    'sort': 0,
                    'type': 'query'
                },
                {
                    'current': {
                        'text': 'production',
                        'value': 'production'
                    },
                    'datasource': 'market',
                    'includeAll': True,
                    'label': 'environment',
                    'multi': True,
                    'name': 'env',
                    'query': 'one_min.report_metrics.max.$host_type.env.*',
                    'refresh': 1,
                    'sort': 0,
                    'type': 'query'
                },
                {
                    'current': {
                        'text': 'market',
                        'value': 'market'
                    },
                    'datasource': 'market',
                    'includeAll': True,
                    'label': 'report type',
                    'multi': True,
                    'name': 'sub_role',
                    'query': 'one_min.report_metrics.max.$host_type.env.$env.sub_role.*',
                    'refresh': 1,
                    'sort': 0,
                    'type': 'query'
                },
                {
                    'current': {
                        'text': 'ALL',
                        'value': 'ALL'
                    },
                    'datasource': 'market',
                    'includeAll': True,
                    'multi': True,
                    'name': 'location',
                    'query': 'one_min.report_metrics.max.$host_type.env.$env.sub_role.$sub_role.location.*',
                    'refresh': 1,
                    'sort': 0,
                    'type': 'query'
                },
                {
                    'current': {
                        'text': 'ALL',
                        'value': 'ALL'
                    },
                    'datasource': 'market',
                    'includeAll': True,
                    'multi': True,
                    'name': 'cluster',
                    'query': 'one_min.report_metrics.max.$host_type.env.$env.sub_role.$sub_role.location.$location.cluster.*',
                    'refresh': 1,
                    'sort': 3,
                    'type': 'query'
                }
            ]
        )
    var_list.append(
        {
            'current': {
                'text': 'All',
                'value': '$__all'
            },
            'datasource': 'market',
            'includeAll': True,
            'label': 'percentiles',
            'multi': True,
            'name': 'p',
            'query': 'one_min.report_metrics.{min,max}.report.env.ALL.sub_role.ALL.location.ALL.cluster.ALL.metric.porto_stats_{memory_left,memory_usage}.*',
            'refresh': 1,
            'sort': 0,
            'type': 'query'
        }
    )
    if not single_host:
        LABEL_FORMATS = (
            (
                'host type',
                r'\1'
            ),
            (
                'environment',
                r'\2'
            ),
            (
                'report type',
                r'\3'
            ),
            (
                'location',
                r'\4'
            ),
            (
                'cluster',
                r'\5'
            ),
            (
                'percentile',
                r'\6'
            ),
        )
        SELECTED_LABEL_INDEX = 5
        var_list.append(
            {
                'current': {
                    'text': LABEL_FORMATS[SELECTED_LABEL_INDEX][0],
                    'value': LABEL_FORMATS[SELECTED_LABEL_INDEX][1]
                },
                'label': 'label format',
                'multi': True,
                'name': 'label',
                'options': [
                    {
                        'selected': index == SELECTED_LABEL_INDEX,
                        'text': label_format[0],
                        'value': label_format[1]
                    }
                    for index, label_format in enumerate(LABEL_FORMATS)
                ],
                'query': ','.join([label_format[1] for label_format in LABEL_FORMATS]),
                'type': 'custom'
            }
        )
    var_list.append(
        {
            'current': {
                'text': SMOOTH_LEVELS[0][0],
                'value': SMOOTH_LEVELS[0][1]
            },
            'name': 'smooth',
            'options': [
                {
                    'selected': name == SMOOTH_LEVELS[0][0],
                    'text': name,
                    'value': value
                }
                for name, value in SMOOTH_LEVELS
            ],
            'query': ','.join(value for _, value in SMOOTH_LEVELS),
            'type': 'custom'
        }
    )
    return {
        'list': var_list
    }


def make_panels(single_host, metric_groups):
    rows = list()
    x = 0
    y = 0
    existing_panel_ids = set()
    for metric_group in metric_groups:
        for entry in metric_group['metrics']:
            if isinstance(entry, dict):
                panel_id = entry['panel_id']
            else:
                _, _, _, panel_id = entry
            if panel_id in existing_panel_ids:
                raise Exception('Duplicate panel id: {}'.format(panel_id))
            existing_panel_ids.add(panel_id)
    unique_panel_id = max(existing_panel_ids) + 1

    for metric_group in metric_groups:
        row = make_row(metric_group['name'], y, unique_panel_id)
        unique_panel_id += 1
        y += ROW_HEIGHT
        graph_width = metric_group.get('graph_width', GRAPH_WIDTH)
        graph_height = metric_group.get('graph_height', GRAPH_HEIGHT)
        panels = list()
        for entry in metric_group['metrics']:
            if isinstance(entry, dict):
                panel = make_graph_for_compound_expression(entry, x, y, graph_width, graph_height, single_host)
            else:
                metric, agg, units, panel_id = entry
                panel = make_graph(metric, x, y, graph_width, graph_height, agg, units, single_host, panel_id)
            panels.append(panel)
            x += graph_width
            if x >= ROW_WIDTH:
                x = 0
                y += graph_height
        if x != 0:
            x = 0
            y += graph_height
        row['panels'] = panels
        rows.append(row)
    return rows


def make_annotations():
    return [
        {
            'builtIn': 1,
            'datasource': '-- Grafana --',
            'enable': False,
            'hide': True,
            'iconColor': '#70dbed',
            'name': 'Notes',
            'type': 'dashboard'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#447ebc',
            'name': 'Front',
            'tags': 'nanny-resource:MARKET_DESKTOP_FRONT',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#ba43a9',
            'name': 'Touch',
            'tags': 'nanny-resource:MARKET_TOUCH_FRONT',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#64b0c8',
            'name': 'Report',
            'tags': 'package:yandex-market-report',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#629e51',
            'name': 'Exp',
            'tags': 'Type: exp',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#e5ac0e',
            'name': 'Стоп-Кран',
            'tags': 'emergency_break',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#b877d9',
            'name': 'flags.json',
            'tags': 'exp:flags_json',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#ff9830',
            'name': 'Push',
            'tags': 'type:mcrm-push',
            'type': 'tags'
        },
    ]


def make_links():
    return [
        {
            'asDropdown': True,
            'icon': 'external link',
            'keepTime': True,
            'tags': [
                'market_report_global_metrics'
            ],
            'title': 'Global',
            'type': 'dashboards'
        },
        {
            'asDropdown': True,
            'icon': 'external link',
            'keepTime': True,
            'tags': [
                'market_report_cluster_metrics'
            ],
            'title': 'Cluster',
            'type': 'dashboards'
        },
        {
            'asDropdown': True,
            'icon': 'external link',
            'keepTime': True,
            'tags': [
                'market_report_host_metrics'
            ],
            'title': 'Host',
            'type': 'dashboards'
        },
        {
            'asDropdown': True,
            'icon': 'external link',
            'keepTime': True,
            'tags': [
                'market_report_host_memory_metrics'
            ],
            'title': 'Host',
            'type': 'dashboards'
        },
        {
            'icon': 'external link',
            'tags': [],
            'title': 'Generator',
            'type': 'link',
            'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/market/report/GeneratePerHostDashboards/grafana_host_metric_dashboard.py'
        }
    ]


def get_garafana_uid(single_host):
    return 'TfrhVtJmk' if single_host else 'erXys3oiz'


def get_garafana_mem_dashboard_uid(single_host):
    return 'zaBxIUu05' if single_host else 'FOkEj7sE6'


def retry(max_try_count=5, initial_delay=2, exp_backoff=2):

    def retry_decorator(function):

        @functools.wraps(function)
        def retry_wrapper(*args, **kwargs):
            try_count = 0
            retry_delay = initial_delay
            while True:
                try:
                    return function(*args, **kwargs)
                except Exception:
                    try_count += 1
                    if try_count == max_try_count:
                        raise
                    logging.exception('Execution failed %d times, retrying...', try_count)
                    time.sleep(retry_delay)
                    retry_delay *= exp_backoff
        return retry_wrapper

    return retry_decorator


def make_dashboard(single_host, nanny_auth_token, publish):
    title = 'Standalone Report Host Metrics' if single_host else 'Aggregated Report Host Metrics'
    title += ' [autogenerated on {date}]'.format(
        date=datetime.datetime.now().strftime('%Y-%m-%d %H:%M'),
    )
    return {
        'annotations': {'list': make_annotations()},
        'graphTooltip': 1,
        'links': make_links(),
        'panels': make_panels(single_host, KNOWN_METRIC_GROUPS),
        'refresh': '1m',
        'schemaVersion': 16,
        'tags': [
            'market_report',
            'market_report_host_metrics',
        ] if publish else [],
        'templating': make_templating(single_host, nanny_auth_token),
        'title': title,
        'uid': get_garafana_uid(single_host)
    }


def make_mem_dashboard(single_host, nanny_auth_token, publish):
    title = 'Standalone Report Host Memory Metrics' if single_host else 'Aggregated Report Host Memory Metrics'
    title += ' [autogenerated on {date}]'.format(
        date=datetime.datetime.now().strftime('%Y-%m-%d %H:%M'),
    )
    return {
        'annotations': {'list': make_annotations()},
        'graphTooltip': 1,
        'links': make_links(),
        'panels': make_panels(single_host, MEM_DASHBOARD_METRIC_GROUPS),
        'refresh': '1m',
        'schemaVersion': 16,
        'tags': [
            'market_report',
            'market_report_host_memory_metrics',
        ] if publish else [],
        'templating': make_templating(single_host, nanny_auth_token),
        'title': title,
        'uid': get_garafana_mem_dashboard_uid(single_host)
    }


@retry()
def publish_dashboard(dashboard, single_host, auth_token):
    headers = {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': 'OAuth ' + auth_token
    }
    body = {
        'dashboard': dashboard,
        'folderId': 0,
        'overwrite': True
    }
    conn = httplib.HTTPSConnection('grafana.yandex-team.ru', timeout=10)
    conn.connect()
    try:
        # conn.request(
        #     'DELETE',
        #     '/api/dashboards/uid/{uid}'.format(uid=get_garafana_uid(single_host)),
        #     headers=headers
        # )
        # conn.getresponse().read()
        conn.request('POST', '/api/dashboards/db', json.dumps(body), headers)
        response = conn.getresponse()
        response_data = response.read()
        if response.status != 200:
            raise Exception('Grafana API returned error: {}\n{}'.format(response.status, response_data))
    finally:
        conn.close()
    return response_data


def get_grafana_token():
    auth_token = os.environ.get('GRAFANA_AUTH_TOKEN')
    if auth_token:
        return auth_token
    token_file_path = os.path.expanduser('~/.robot-market-st.tokens')
    if not os.path.isfile(token_file_path):
        return None
    with open(token_file_path, 'r') as token_file:
        tokens = json.loads(token_file.read())
        return tokens.get('grafana')


def get_nanny_token():
    auth_token = os.environ.get('NANNY_AUTH_TOKEN')
    if auth_token:
        return auth_token
    token_file_path = os.path.expanduser('~/.robot-market-st.tokens')
    if not os.path.isfile(token_file_path):
        return None
    with open(token_file_path, 'r') as token_file:
        tokens = json.loads(token_file.read())
        return tokens.get('nanny')


@retry()
def send_nanny_api_request(query, auth_token=None):
    headers = dict()
    headers['Accept'] = 'application/json'
    if auth_token:
        headers['Authorization'] = 'OAuth {0}'.format(auth_token)
    conn = httplib.HTTPConnection('nanny.yandex-team.ru')
    conn.request('GET', query, headers=headers)
    try:
        response = conn.getresponse()
        if response.status != 200:
            if response.status == 404:
                return None
            raise Exception('Nanny API returned error: {} {}\n{}'.format(response.status, response.reason, query))
        json_str = response.read()
    finally:
        conn.close()
    return json.loads(json_str)


def get_nanny_group_list(auth_token):
    group_list = list()
    nanny_data = send_nanny_api_request('/v2/services/?exclude_runtime_attrs=1&category=/market/report', auth_token)
    for group_info in nanny_data['result']:
        group_name = group_info['_id']
        group_list.append(group_name)
    logging.info('Found %d Nanny groups', len(group_list))
    return group_list


def get_host_list_for_group(group_name):
    logging.info('Processing Nanny group %s', group_name)
    nanny_data = send_nanny_api_request('/v2/services/{0}/current_state/instances/'.format(group_name))
    if nanny_data is None:
        return list()
    shard_matcher = re.compile(r'^a_shard_(\d+)$')
    replica_matcher = re.compile(r'^itag_replica_(\d+)$')
    host_list = list()
    for host_info in nanny_data['result']:
        cluster_index = None
        host_index = None
        for itag in host_info['itags']:
            shard_match = shard_matcher.match(itag)
            if shard_match:
                cluster_index = int(shard_match.group(1))
            replica_match = replica_matcher.match(itag)
            if replica_match:
                host_index = int(replica_match.group(1))
        if cluster_index is None or host_index is None:
            continue
        host_name = host_info['container_hostname']
        host_list.append((cluster_index, host_index, host_name))
    logging.info('Found %d hosts in group %s', len(host_list), group_name)
    return host_list


def parse_nanny_group_name(group_name):
    parts = group_name.split('_')
    if len(parts) < 4 or parts[1] != 'report':
        return None
    env_type = parts[0]
    dc = parts[-1]
    is_snippet = parts[-2] == 'snippet'
    report_type = '_'.join(parts[2:-2 if is_snippet else -1])
    return (env_type, report_type, dc, is_snippet)


def get_host_list(nanny_auth_token):
    nanny_group_list = get_nanny_group_list(nanny_auth_token)
    host_list = list()
    for group_name in nanny_group_list:
        if not parse_nanny_group_name(group_name):
            continue
        env_type, report_type, dc, _ = parse_nanny_group_name(group_name)
        for cluster_index, host_index, host_name in get_host_list_for_group(group_name):
            host_list.append((env_type, report_type, dc, cluster_index, host_index, host_name))
    return sorted(host_list)


def get_hp_host_list():
    def is_hp_host(host):
        return bool(re.match(r'msh\d\dhp.market.yandex.net', host))

    def is_hp_snippet_host(host):
        return bool(re.match(r'msh-off\d\dhp.market.yandex.net', host))

    def get_host_id(host):
        return int(re.search(r'\d\d', host).group()) - 1

    def get_cluster_and_host_indices(host):
        host_id = get_host_id(host)
        if is_hp_host(host):
            return host_id // 8, host_id % 8
        elif is_hp_snippet_host(host):
            return host_id, 8
        else:
            return None

    conn = httplib.HTTPConnection('c.yandex-team.ru')
    hosts = list()

    for group in CONDUCTOR_GROUPS:
        try:
            conn.request('GET', '/api/groups2hosts/{}'.format(group))
            resp = conn.getresponse()
            if resp.status != 200:
                if resp.status == 404:
                    return None
                raise Exception('Conductor API returned error: {} {}'.format(resp.status, resp.reason))

            for host in resp.read().split():
                cluster_index, host_index = get_cluster_and_host_indices(host)
                hosts.append(('hp', 'market', 'sas', cluster_index, host_index, host))
        finally:
            conn.close()

    return sorted(hosts)


def main():
    arg_parser = argparse.ArgumentParser(description='Generate host metric dashboards for Grafana.')
    arg_parser.add_argument('--per-host', action='store_true', help='Create per-host dashboard')
    arg_parser.add_argument('--dashboards', choices=['common', 'memory'], nargs='+', default=['common'], help='Dashboard to generate')
    arg_parser.add_argument('--publish', action='store_true', help='Publish dashboard via Grafana API')
    args = arg_parser.parse_args()
    nanny_auth_token = None
    if args.per_host:
        nanny_auth_token = get_nanny_token()
        if not nanny_auth_token:
            print '''Obtain Nanny auth token here: https://nanny.yandex-team.ru/ui/#/oauth/
Set environment variable NANNY_AUTH_TOKEN=<token>
Another way is to get .robot-market-st.tokens from your colleague with 'nanny' token in it.'''
            return
    if args.publish:
        grafana_auth_token = get_grafana_token()
        if not grafana_auth_token:
            print '''Obtain Grafana auth token here: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=9590d99772e74c7386610688d4a5c1b3
Examine Response Headers->Location in browser network debugger, it is going to look like this: https://grafana.yandex-team.ru#access_token=<token>&token_type=bearer
Set environment variable GRAFANA_AUTH_TOKEN=<token>
Another way is to get .robot-market-st.tokens from your colleague with 'grafana' token in it.'''
            return

    for name in args.dashboards:
        if name == 'memory':
            dashboard = make_mem_dashboard(args.per_host, nanny_auth_token, args.publish)
        elif name == 'common':
            dashboard = make_dashboard(args.per_host, nanny_auth_token, args.publish)
        else:
            raise Exception('unknown dashboard ' + name)
        if args.publish:
            print publish_dashboard(dashboard, args.per_host, grafana_auth_token)
        else:
            print json.dumps(dashboard, indent=2)


if __name__ == '__main__':
    main()
