#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import datetime
import functools
import httplib
import json
import logging
import os
import re
import time

GRAPH_WIDTH = 6
GRAPH_HEIGHT = 8
ROW_HEIGHT = 1
ROW_WIDTH = 24

HOSTS_COUNT = 16.
SHARDS_COUNT = 16.

RTY_MEM_GAP = 7.4
FREE_MEM_GAP = 5.

CONDUCTOR_GROUPS = ['market_search-full-perf-testing']

SMOOTH_LEVELS = [
    ('No', ''),
    ('15 minutes', '|movingAverage("15min")'),
    ('30 minutes', '|movingAverage("30min")'),
    ('1 hour', '|movingAverage("1h")'),
    ('4 hours', '|movingAverage("4h")'),
    ('1 day', '|movingAverage("1d")'),
]

KNOWN_METRIC_GROUPS = [
    {
        'name': 'System Memory',
        'metrics': [
            ('porto_stats.memory_left', 'min', 'bytes', 1),
            ('porto_stats.memory_limit_total', 'max', 'bytes', 2),
            ('porto_stats.cache_usage', 'min', 'bytes', 7),
            ('memory_stats.shmem_pmd_mapped', 'max', 'bytes', 9),
            ('memory_stats.mlocked', 'max', 'bytes', 146),
        ]
    },
    {
        'name': 'Tmpfs Memory',
        'metrics': [
            ('memory_stats.tmpfs_memory_used', 'max', 'bytes', 20),
            ('memory_stats.tmpfs_memory_left', 'min', 'bytes', 139),
            ('memory_stats.tmpfs_index', 'max', 'bytes', 140),
            ('memory_stats.tmpfs_mmap_files', 'max', 'bytes', 21),
            ('memory_stats.tmpfs_model_partN', 'max', 'bytes', 22),
            ('memory_stats.tmpfs_partN', 'max', 'bytes', 23),
            ('memory_stats.tmpfs_part_blue', 'max', 'bytes', 24),
            ('memory_stats.tmpfs_rty', 'max', 'bytes', 26)
        ]
    },
    {
        'name': 'Documents',
        'metrics': [
            ('report_versions.offer_count', 'max', 'locale', 141),
            ('report_versions.blue_offer_count', 'max', 'locale', 150),
            ('rty_server.searchable_docs', 'max', 'locale', 142),
        ]
    },
    {
        'name': 'Index Files',
        'metrics': [
            ('report_files.index_part_1.*', 'max', 'bytes', 94),
            ('report_files.index_part_blue.*', 'max', 'bytes', 95),
            ('report_files.index_part_model.*', 'max', 'bytes', 96),
            ('report_files.index_part_book.*', 'max', 'bytes', 116),
            ('report_files.index_part_cards.*', 'max', 'bytes', 117),
            ('report_files.index_part_wizard.*', 'max', 'bytes', 118),
            ('report_files.mmap.*', 'max', 'bytes', 97),
            ('report_files.report-data.*', 'max', 'bytes', 98)
        ],
        'graph_width': ROW_WIDTH / 3,
        'graph_height': 12
    },
]


def get_scale_for_index():
    gib = 1024. ** 3
    mil = 10. ** 6
    return mil * SHARDS_COUNT / gib


def get_scale_for_mmap():
    gib = 1024. ** 3
    mil = 10. ** 6
    return mil * HOSTS_COUNT / gib


def gigabyte_to_bytes(gib):
    return gib * 1024 ** 3


def get_sub_memory():
    return -gigabyte_to_bytes(RTY_MEM_GAP + FREE_MEM_GAP)


def sum_expr_tmplt(aliases, coefs):
    scale = []
    for alias, coef in zip(aliases, coefs):
        scale.append('scale(' + ','.join([''.join(['{', alias, '}']), str(coef)]) + ')')
    return 'sumSeries(' + ','.join(scale) + ')'


INDEX_ALIASES = ['index_size', 'mmap', 'blue', 'model', 'cards', 'book', 'wizard', 'offer_count']
INDEX_COEFS = [get_scale_for_index()] + [get_scale_for_mmap()] * (len(INDEX_ALIASES) - 1)


def get_optimistic_memory_for_offers(offer_millions):
    return 'scale(scale(divideSeries(scale({{index_size}}, {scale_index}), {{offer_count}}), {inv_shard}), {offers})'.format(offers=offer_millions, inv_shard=1./SHARDS_COUNT, scale_index=get_scale_for_index())


def get_pessimistic_memory_for_offers(offer_millions):
    return 'scale(scale(divideSeries({pes}, {{offer_count}}), {inv_shard}), {offers})'.format(offers=offer_millions, inv_shard=1./SHARDS_COUNT, pes=sum_expr_tmplt(INDEX_ALIASES[:2], INDEX_COEFS[:2]))


MEM_DASHBOARD_METRIC_GROUPS = [
    {
        'name': 'Offers and Memory',
        'metrics': [
            {
                'title': 'Memory Usage (Gb) per 1M offers',
                'metrics': {
                    'index_size': ('report_files.index_part_1.total', 'max'),
                    'mmap': ('report_files.mmap.total', 'max'),
                    'blue': ('report_files.index_part_blue.total', 'max'),
                    'model': ('report_files.index_part_model.total', 'max'),
                    'cards': ('report_files.index_part_cards.total', 'max'),
                    'book': ('report_files.index_part_book.total', 'max'),
                    'wizard': ('report_files.index_part_wizard.total', 'max'),
                    'offer_count': ('report_versions.offer_count', 'max'),
                },
                'expressions': [
                    'divideSeries({pessimistic}, {{offer_count}})'.format(pessimistic=sum_expr_tmplt(INDEX_ALIASES[:2], INDEX_COEFS[:2])),
                    'scale(divideSeries({{index_size}}, {{offer_count}}), {scale_index})'.format(scale_index=get_scale_for_index()),
                    'divideSeries({super_pessimistic}, {{offer_count}})'.format(super_pessimistic=sum_expr_tmplt(INDEX_ALIASES, INDEX_COEFS)),
                ],
                'units': 'locale',
                'labels': [
                    'pessimistic',
                    'optimistic',
                    'super_pessimistic'
                ],
                'panel_id': 1
            },
            {
                'title': 'Allowed Additional Offers (M) from Memory Left (GB)',
                'metrics': {
                    'index_size': ('report_files.index_part_1.total', 'max'),
                    'mmap': ('report_files.mmap.total', 'max'),
                    'blue': ('report_files.index_part_blue.total', 'max'),
                    'model': ('report_files.index_part_model.total', 'max'),
                    'cards': ('report_files.index_part_cards.total', 'max'),
                    'book': ('report_files.index_part_book.total', 'max'),
                    'wizard': ('report_files.index_part_wizard.total', 'max'),
                    'offer_count': ('report_versions.offer_count', 'max'),
                    'mem_left': ('porto_stats.memory_left', 'min')
                },
                'expressions': [
                    'movingMin(multiplySeries(scale(add({{mem_left}}, {sub_memory}), {scale_tmpfs}), divideSeries({{offer_count}}, {pessimistic})), \"6hour\")'.format(pessimistic=sum_expr_tmplt(INDEX_ALIASES[:2], INDEX_COEFS[:2]), scale_tmpfs=get_scale_for_mmap() / 10 ** 6, sub_memory=get_sub_memory()),
                    'movingMin(multiplySeries(scale(add({{mem_left}}, {sub_memory}), {scale_tmpfs}), divideSeries({{offer_count}}, scale({{index_size}}, {scale_index}))), \"6hour\")'.format(scale_index=get_scale_for_index(), scale_tmpfs=get_scale_for_mmap() / 10 ** 6, sub_memory=get_sub_memory()),
                    'movingMin(multiplySeries(scale(add({{mem_left}}, {sub_memory}), {scale_tmpfs}), divideSeries({{offer_count}}, {super_pessimistic})), \"6hour\")'.format(super_pessimistic=sum_expr_tmplt(INDEX_ALIASES, INDEX_COEFS), scale_tmpfs=get_scale_for_mmap() / 10 ** 6, sub_memory=get_sub_memory()),
                ],
                'units': 'locale',
                'labels': [
                    'pessimistic',
                    'optimistic',
                    'super_pessimistic',
                ],
                'panel_id': 2
            },
            {
                'title': 'Memory Usage for 400/800/1200M offers',
                'metrics': {
                    'index_size': ('report_files.index_part_1.total', 'max'),
                    'mmap': ('report_files.mmap.total', 'max'),
                    'offer_count': ('report_versions.offer_count', 'max'),
                    'tmpfs_memory': ('memory_stats.tmpfs_memory_left', 'min')
                },
                'expressions': [
                                   get_optimistic_memory_for_offers(offers) for offers in [400, 800, 1200]
                               ] + [
                                   get_pessimistic_memory_for_offers(offers) for offers in [400, 800, 1200]
                               ],
                'units': 'locale',
                'labels': [
                              "optimistic_per_shard_{offers}".format(offers=offers) for offers in [400, 800, 1200]
                          ] + [
                              "pessimistic_per_shard_{offers}".format(offers=offers) for offers in [400, 800, 1200]
                          ],
                'panel_id': 3
            },
            {
                'title': 'All Offers Prediction (M)',
                'metrics': {
                    'index_size': ('report_files.index_part_1.total', 'max'),
                    'mmap': ('report_files.mmap.total', 'max'),
                    'blue': ('report_files.index_part_blue.total', 'max'),
                    'model': ('report_files.index_part_model.total', 'max'),
                    'cards': ('report_files.index_part_cards.total', 'max'),
                    'book': ('report_files.index_part_book.total', 'max'),
                    'wizard': ('report_files.index_part_wizard.total', 'max'),
                    'offer_count': ('report_versions.offer_count', 'max'),
                    'mem_left': ('porto_stats.memory_left', 'min')
                },
                'expressions': [
                    'movingMin(sumSeries(scale({{offer_count}}, {inv_mil}), multiplySeries(scale(add({{mem_left}}, {sub_memory}), {scale_tmpfs}), divideSeries({{offer_count}}, {pessimistic}))), \"6hour\")'.format(pessimistic=sum_expr_tmplt(INDEX_ALIASES[:2], INDEX_COEFS[:2]), scale_tmpfs=get_scale_for_mmap() / 10 ** 6, sub_memory=get_sub_memory(), inv_mil=1./10**6),
                    'movingMin(sumSeries(scale({{offer_count}}, {inv_mil}), multiplySeries(scale(add({{mem_left}}, {sub_memory}), {scale_tmpfs}), divideSeries({{offer_count}}, scale({{index_size}}, {scale_index})))), \"6hour\")'.format(scale_index=get_scale_for_index(), scale_tmpfs=get_scale_for_mmap() / 10 ** 6, sub_memory=get_sub_memory(), inv_mil=1./10**6),
                    'movingMin(sumSeries(scale({{offer_count}}, {inv_mil}), multiplySeries(scale(add({{mem_left}}, {sub_memory}), {scale_tmpfs}), divideSeries({{offer_count}}, {super_pessimistic}))), \"6hour\")'.format(super_pessimistic=sum_expr_tmplt(INDEX_ALIASES, INDEX_COEFS), scale_tmpfs=get_scale_for_mmap() / 10 ** 6, sub_memory=get_sub_memory(), inv_mil=1./10**6),
                ],
                'units': 'locale',
                'labels': [
                    'pessimistic',
                    'optimistic',
                    'super_pessimistic',
                ],
                'panel_id': 4,
            },
        ]
    },
]


def make_row(name, y, panel_id):
    return {
        'collapsed': True,
        'gridPos': {
            'h': ROW_HEIGHT,
            'w': ROW_WIDTH,
            'x': 0,
            'y': y
        },
        'id': panel_id,
        'title': name,
        'type': 'row'
    }


def make_metric_expression(metric, agg):
    metric_id = metric.replace('.', '_')
    metric_exp = 'one_min.report_metrics.{agg}.$host_type.env.$env.sub_role.$sub_role.location.$location.cluster.$cluster.metric.{metric_id}'
    if agg != 'sum':
        if agg == 'max':
            metric_exp += '.1'
        else:
            metric_exp += '.0'
    metric_exp = metric_exp.format(agg=agg, metric_id=metric_id)
    return metric_exp


def prepare_expression(raw_expression, label, exp_dict):
    metric_exp = raw_expression.format(**exp_dict)
    return metric_exp + '|removeEmptySeries()[[smooth]]|alias("{label}")'.format(label=label)


def make_graph_for_compound_expression(metric_dict, x, y, w, h):
    exp_dict = dict()
    for key, entry in metric_dict['metrics'].iteritems():
        metric, agg = entry
        exp_dict[key] = make_metric_expression(metric, agg)

    graph = {
        'datasource': 'market',
        'fill': 0,
        'gridPos': {
            'h': h,
            'w': w,
            'x': x,
            'y': y
        },
        'id': metric_dict['panel_id'],
        'maxDataPoints': '2000',
        'targets': [],
        'title': metric_dict['title'],
        'tooltip': {
            'sort': 2
        },
        'type': 'graph',
        'yaxes': [
            {
                'format': metric_dict['units'],
                'show': True
            },
            {
                'format': metric_dict['units'],
                'show': True
            }
        ],
    }

    if 'expression' in metric_dict:
        graph['targets'].append({
            'refId': 'A',
            'target': prepare_expression(metric_dict['expression'], metric_dict['label'], exp_dict)
        })

    if 'expressions' not in metric_dict:
        return graph

    for raw_expression, label in zip(metric_dict['expressions'], metric_dict['labels']):
        current_letter = 'A'
        if len(graph['targets']) > 0:
            current_letter = chr(ord(graph['targets'][-1]['refId']) + 1)

        graph['targets'].append({
            'refId': current_letter,
            'target': prepare_expression(raw_expression, label, exp_dict)
        })

    return graph


def make_graph(metric, x, y, w, h, agg, units, panel_id):
    multi_graph = '*' in metric

    metric_id = metric.replace('.', '_')
    metric_exp = 'one_min.report_metrics.{agg}.$host_type.env.$env.sub_role.$sub_role.location.$location.cluster.$cluster.metric.{metric_id}'
    if multi_graph:
        label = r'.*\.([^.]+)\.env\.([^.]+)\.sub_role\.([^.]+)\.location\.([^.]+)\.cluster\.([^.]+)\.metric\.{metric}([^.]*).*'.format(metric=metric_id.rstrip('*'))
        if agg != 'sum':
            if agg == 'max':
                metric_exp += '.1'
            else:
                metric_exp += '.0'
    elif agg == 'sum':
        label = r'.*\.([^.]+)\.env\.([^.]+)\.sub_role\.([^.]+)\.location\.([^.]+)\.cluster\.([^.]+)\.metric\.([^,]*).*'
    else:
        label = r'.*\.([^.]+)\.env\.([^.]+)\.sub_role\.([^.]+)\.location\.([^.]+)\.cluster\.([^.]+)\.metric\.[^.]+\.([^,]+).*'
        metric_exp += '.$p'
    metric_exp += '|removeEmptySeries()[[smooth]]|aliasSub("{label}", "${{label:csv}}")|sortByName(True)'
    metric_exp = metric_exp.format(agg=agg, metric_id=metric_id, label=label)

    graph = {
        'datasource': 'market',
        'fill': 0,
        'gridPos': {
            'h': h,
            'w': w,
            'x': x,
            'y': y
        },
        'id': panel_id,
        'maxDataPoints': '2000',
        'targets': [
            {
                'refId': 'A',
                'target': metric_exp
            }
        ],
        'title': metric,
        'tooltip': {
            'sort': 2
        },
        'type': 'graph',
        'yaxes': [
            {
                'format': units,
                'show': True
            },
            {
                'format': units,
                'show': True
            }
        ],
    }
    if multi_graph:
        graph['seriesOverrides'] = [
            {
                'alias': 'total',
                'yaxis': 2
            }
        ]
    return graph


def make_templating(nanny_auth_token):
    var_list = list()
    var_list.extend(
        [
            {
                'current': {
                    'text': 'report',
                    'value': 'report'
                },
                'datasource': 'market',
                'includeAll': True,
                'label': 'host type',
                'multi': True,
                'name': 'host_type',
                'query': 'one_min.report_metrics.max.*',
                'refresh': 1,
                'sort': 0,
                'type': 'query'
            },
            {
                'current': {
                    'text': 'production',
                    'value': 'production'
                },
                'datasource': 'market',
                'includeAll': True,
                'label': 'environment',
                'multi': True,
                'name': 'env',
                'query': 'one_min.report_metrics.max.$host_type.env.*',
                'refresh': 1,
                'sort': 0,
                'type': 'query'
            },
            {
                'current': {
                    'text': 'market',
                    'value': 'market'
                },
                'datasource': 'market',
                'includeAll': True,
                'label': 'report type',
                'multi': True,
                'name': 'sub_role',
                'query': 'one_min.report_metrics.max.$host_type.env.$env.sub_role.*',
                'refresh': 1,
                'sort': 0,
                'type': 'query'
            },
            {
                'current': {
                    'text': 'ALL',
                    'value': 'ALL'
                },
                'datasource': 'market',
                'includeAll': True,
                'multi': True,
                'name': 'location',
                'query': 'one_min.report_metrics.max.$host_type.env.$env.sub_role.$sub_role.location.*',
                'refresh': 1,
                'sort': 0,
                'type': 'query'
            },
            {
                'current': {
                    'text': 'ALL',
                    'value': 'ALL'
                },
                'datasource': 'market',
                'includeAll': True,
                'multi': True,
                'name': 'cluster',
                'query': 'one_min.report_metrics.max.$host_type.env.$env.sub_role.$sub_role.location.$location.cluster.*',
                'refresh': 1,
                'sort': 3,
                'type': 'query'
            }
        ]
    )
    var_list.append(
        {
            'current': {
                'text': 'All',
                'value': '$__all'
            },
            'datasource': 'market',
            'includeAll': True,
            'label': 'percentiles',
            'multi': True,
            'name': 'p',
            'query': 'one_min.report_metrics.{min,max}.report.env.ALL.sub_role.ALL.location.ALL.cluster.ALL.metric.porto_stats_{memory_left,memory_usage}.*',
            'refresh': 1,
            'sort': 0,
            'type': 'query'
        }
    )
    LABEL_FORMATS = (
        (
            'host type',
            r'\1'
        ),
        (
            'environment',
            r'\2'
        ),
        (
            'report type',
            r'\3'
        ),
        (
            'location',
            r'\4'
        ),
        (
            'cluster',
            r'\5'
        ),
        (
            'percentile',
            r'\6'
        ),
    )
    SELECTED_LABEL_INDEX = 5
    var_list.append(
        {
            'current': {
                'text': LABEL_FORMATS[SELECTED_LABEL_INDEX][0],
                'value': LABEL_FORMATS[SELECTED_LABEL_INDEX][1]
            },
            'label': 'label format',
            'multi': True,
            'name': 'label',
            'options': [
                {
                    'selected': index == SELECTED_LABEL_INDEX,
                    'text': label_format[0],
                    'value': label_format[1]
                }
                for index, label_format in enumerate(LABEL_FORMATS)
            ],
            'query': ','.join([label_format[1] for label_format in LABEL_FORMATS]),
            'type': 'custom'
        }
    )
    var_list.append(
        {
            'current': {
                'text': SMOOTH_LEVELS[0][0],
                'value': SMOOTH_LEVELS[0][1]
            },
            'name': 'smooth',
            'options': [
                {
                    'selected': name == SMOOTH_LEVELS[0][0],
                    'text': name,
                    'value': value
                }
                for name, value in SMOOTH_LEVELS
            ],
            'query': ','.join(value for _, value in SMOOTH_LEVELS),
            'type': 'custom'
        }
    )
    return {
        'list': var_list
    }


def make_panels(metric_groups):
    rows = list()
    x = 0
    y = 0
    existing_panel_ids = set()
    for metric_group in metric_groups:
        for entry in metric_group['metrics']:
            if isinstance(entry, dict):
                panel_id = entry['panel_id']
            else:
                _, _, _, panel_id = entry
            if panel_id in existing_panel_ids:
                raise Exception('Duplicate panel id: {}'.format(panel_id))
            existing_panel_ids.add(panel_id)
    unique_panel_id = max(existing_panel_ids) + 1

    for metric_group in metric_groups:
        row = make_row(metric_group['name'], y, unique_panel_id)
        unique_panel_id += 1
        y += ROW_HEIGHT
        graph_width = metric_group.get('graph_width', GRAPH_WIDTH)
        graph_height = metric_group.get('graph_height', GRAPH_HEIGHT)
        panels = list()
        for entry in metric_group['metrics']:
            if isinstance(entry, dict):
                panel = make_graph_for_compound_expression(entry, x, y, graph_width, graph_height)
            else:
                metric, agg, units, panel_id = entry
                panel = make_graph(metric, x, y, graph_width, graph_height, agg, units, panel_id)
            panels.append(panel)
            x += graph_width
            if x >= ROW_WIDTH:
                x = 0
                y += graph_height
        if x != 0:
            x = 0
            y += graph_height
        row['panels'] = panels
        rows.append(row)
    return rows


def make_annotations():
    return [
        {
            'builtIn': 1,
            'datasource': '-- Grafana --',
            'enable': False,
            'hide': True,
            'iconColor': '#70dbed',
            'name': 'Notes',
            'type': 'dashboard'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#447ebc',
            'name': 'Front',
            'tags': 'nanny-resource:MARKET_DESKTOP_FRONT',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#ba43a9',
            'name': 'Touch',
            'tags': 'nanny-resource:MARKET_TOUCH_FRONT',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#64b0c8',
            'name': 'Report',
            'tags': 'package:yandex-market-report',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#629e51',
            'name': 'Exp',
            'tags': 'Type: exp',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#e5ac0e',
            'name': 'Стоп-Кран',
            'tags': 'emergency_break',
            'type': 'tags'
        },
        {
            'datasource': 'market',
            'enable': False,
            'hide': False,
            'iconColor': '#b877d9',
            'name': 'flags.json',
            'tags': 'exp:flags_json',
            'type': 'tags'
        },
    ]


def make_links():
    return [
        {
            'icon': 'external link',
            'tags': [],
            'title': 'Generator',
            'type': 'link',
            'url': 'https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/market/report/GeneratePerHostDashboards/offer_memory_expectation_dashboard.py'
        },
    ]


def get_garafana_mem_dashboard_uid():
    return "Th4rVQyGz"


def retry(max_try_count=5, initial_delay=2, exp_backoff=2):

    def retry_decorator(function):

        @functools.wraps(function)
        def retry_wrapper(*args, **kwargs):
            try_count = 0
            retry_delay = initial_delay
            while True:
                try:
                    return function(*args, **kwargs)
                except Exception:
                    try_count += 1
                    if try_count == max_try_count:
                        raise
                    logging.exception('Execution failed %d times, retrying...', try_count)
                    time.sleep(retry_delay)
                    retry_delay *= exp_backoff
        return retry_wrapper

    return retry_decorator


def make_mem_dashboard(nanny_auth_token, publish):
    title = 'Aggregated Offer Capacity Prediction'
    title += ' [autogenerated on {date}]'.format(
        date=datetime.datetime.now().strftime('%Y-%m-%d %H:%M'),
    )
    return {
        'annotations': {'list': make_annotations()},
        'graphTooltip': 1,
        'links': make_links(),
        'panels': make_panels(MEM_DASHBOARD_METRIC_GROUPS),
        'refresh': '1m',
        'schemaVersion': 16,
        'tags': [
            'market_report',
            'market_report_host_memory_metrics',
        ] if publish else [],
        'templating': make_templating(nanny_auth_token),
        'title': title,
        'uid': get_garafana_mem_dashboard_uid()
    }


@retry()
def publish_dashboard(dashboard, auth_token):
    headers = {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': 'OAuth ' + auth_token
    }
    body = {
        'dashboard': dashboard,
        'folderId': 0,
        'overwrite': True
    }
    conn = httplib.HTTPSConnection('grafana.yandex-team.ru', timeout=10)
    conn.connect()
    try:
        conn.request('POST', '/api/dashboards/db', json.dumps(body), headers)
        response = conn.getresponse()
        response_data = response.read()
        if response.status != 200:
            raise Exception('Grafana API returned error: {}\n{}'.format(response.status, response_data))
    finally:
        conn.close()
    return response_data


def get_grafana_token():
    auth_token = os.environ.get('GRAFANA_AUTH_TOKEN')
    if auth_token:
        return auth_token
    token_file_path = os.path.expanduser('~/.robot-market-st.tokens')
    if not os.path.isfile(token_file_path):
        return None
    with open(token_file_path, 'r') as token_file:
        tokens = json.loads(token_file.read())
        return tokens.get('grafana')


def get_nanny_token():
    auth_token = os.environ.get('NANNY_AUTH_TOKEN')
    if auth_token:
        return auth_token
    token_file_path = os.path.expanduser('~/.robot-market-st.tokens')
    if not os.path.isfile(token_file_path):
        return None
    with open(token_file_path, 'r') as token_file:
        tokens = json.loads(token_file.read())
        return tokens.get('nanny')


@retry()
def send_nanny_api_request(query, auth_token=None):
    headers = dict()
    headers['Accept'] = 'application/json'
    if auth_token:
        headers['Authorization'] = 'OAuth {0}'.format(auth_token)
    conn = httplib.HTTPConnection('nanny.yandex-team.ru')
    conn.request('GET', query, headers=headers)
    try:
        response = conn.getresponse()
        if response.status != 200:
            if response.status == 404:
                return None
            raise Exception('Nanny API returned error: {} {}\n{}'.format(response.status, response.reason, query))
        json_str = response.read()
    finally:
        conn.close()
    return json.loads(json_str)


def get_nanny_group_list(auth_token):
    group_list = list()
    nanny_data = send_nanny_api_request('/v2/services/?exclude_runtime_attrs=1&category=/market/report', auth_token)
    for group_info in nanny_data['result']:
        group_name = group_info['_id']
        group_list.append(group_name)
    logging.info('Found %d Nanny groups', len(group_list))
    return group_list


def get_host_list_for_group(group_name):
    logging.info('Processing Nanny group %s', group_name)
    nanny_data = send_nanny_api_request('/v2/services/{0}/current_state/instances/'.format(group_name))
    if nanny_data is None:
        return list()
    shard_matcher = re.compile(r'^a_shard_(\d+)$')
    replica_matcher = re.compile(r'^itag_replica_(\d+)$')
    host_list = list()
    for host_info in nanny_data['result']:
        cluster_index = None
        host_index = None
        for itag in host_info['itags']:
            shard_match = shard_matcher.match(itag)
            if shard_match:
                cluster_index = int(shard_match.group(1))
            replica_match = replica_matcher.match(itag)
            if replica_match:
                host_index = int(replica_match.group(1))
        if cluster_index is None or host_index is None:
            continue
        host_name = host_info['container_hostname']
        host_list.append((cluster_index, host_index, host_name))
    logging.info('Found %d hosts in group %s', len(host_list), group_name)
    return host_list


def parse_nanny_group_name(group_name):
    parts = group_name.split('_')
    if len(parts) < 4 or parts[1] != 'report':
        return None
    env_type = parts[0]
    dc = parts[-1]
    is_snippet = parts[-2] == 'snippet'
    report_type = '_'.join(parts[2:-2 if is_snippet else -1])
    return (env_type, report_type, dc, is_snippet)


def get_host_list(nanny_auth_token):
    nanny_group_list = get_nanny_group_list(nanny_auth_token)
    host_list = list()
    for group_name in nanny_group_list:
        if not parse_nanny_group_name(group_name):
            continue
        env_type, report_type, dc, _ = parse_nanny_group_name(group_name)
        for cluster_index, host_index, host_name in get_host_list_for_group(group_name):
            host_list.append((env_type, report_type, dc, cluster_index, host_index, host_name))
    return sorted(host_list)


def get_hp_host_list():
    def is_hp_host(host):
        return bool(re.match(r'msh\d\dhp.market.yandex.net', host))

    def is_hp_snippet_host(host):
        return bool(re.match(r'msh-off\d\dhp.market.yandex.net', host))

    def get_host_id(host):
        return int(re.search(r'\d\d', host).group()) - 1

    def get_cluster_and_host_indices(host):
        host_id = get_host_id(host)
        if is_hp_host(host):
            return host_id // 8, host_id % 8
        elif is_hp_snippet_host(host):
            return host_id, 8
        else:
            return None

    conn = httplib.HTTPConnection('c.yandex-team.ru')
    hosts = list()

    for group in CONDUCTOR_GROUPS:
        try:
            conn.request('GET', '/api/groups2hosts/{}'.format(group))
            resp = conn.getresponse()
            if resp.status != 200:
                if resp.status == 404:
                    return None
                raise Exception('Conductor API returned error: {} {}'.format(resp.status, resp.reason))

            for host in resp.read().split():
                cluster_index, host_index = get_cluster_and_host_indices(host)
                hosts.append(('hp', 'market', 'sas', cluster_index, host_index, host))
        finally:
            conn.close()

    return sorted(hosts)


def main():
    arg_parser = argparse.ArgumentParser(description='Generate host metric dashboards for Grafana.')
    arg_parser.add_argument('--publish', action='store_true', help='Publish dashboard via Grafana API')
    args = arg_parser.parse_args()
    nanny_auth_token = None
    if args.publish:
        grafana_auth_token = get_grafana_token()
        if not grafana_auth_token:
            print '''Obtain Grafana auth token here: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=9590d99772e74c7386610688d4a5c1b3
Examine Response Headers->Location in browser network debugger, it is going to look like this: https://grafana.yandex-team.ru#access_token=<token>&token_type=bearer
Set environment variable GRAFANA_AUTH_TOKEN=<token>
Another way is to get .robot-market-st.tokens from your colleague with 'grafana' token in it.'''
            return

    dashboard = make_mem_dashboard(nanny_auth_token, args.publish)

    if args.publish:
        print publish_dashboard(dashboard, grafana_auth_token)
    else:
        print json.dumps(dashboard, indent=2)


if __name__ == '__main__':
    main()
