# -*- coding: utf-8 -*-

import logging
from sandbox.common import telegram
from sandbox.projects.market import resources
from sandbox.projects import resource_types
from sandbox.projects import WizardRuntimeBuild, WizardRuntimeBuildPatched
from sandbox.projects.websearch.begemot.tasks.BegemotYT.common import CypressCache
from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel
from sandbox.common.errors import TaskError
from sandbox.common.types.task import Status
from sandbox.common.utils import get_task_link, get_resource_link
from sandbox.sandboxsdk.svn import Arcadia
import hashlib
import os
import re
import shutil
import subprocess

from sandbox.projects.WizardRuntimeBuild.ya_make import YaMake

YaMake = YaMake.YaMake
_1_MB = 1 * 1024 * 1024
_2_MB = 2 * 1024 * 1024

WIZARD_RUNTIME_BUILD_PATCHED = 'WIZARD_RUNTIME_BUILD_PATCHED'
GZT_TASK_FILES = set([
    'categories.gzt',
    'shops.gzt',
])


class FailTemplates:
    COMMON_TEMPLATE = "Sandbox task %s of type %s called from host [%s] in *%s* error:\n"
    COULD_NOT_DROP_RESOURCES_TEMPLATE = ("*Couldn't drop resources*\n"
                                         "%s")
    COMMIT_FAILED_TEMPLATE = ("*Commit failed* with exception %s\n"
                              "Patch url: %s")
    PATCH_FAILED_TEMPLATE = ("*Failed to patch arcadia* with retcode %d\n"
                             "Stdout [%s]\n"
                             "Stderr [%s]\n"
                             "Patch url: %s")
    SUBTASK_FAILED_TEMPLATE = ("*Subtask %s failed*\n"
                               "Link: %s, status %s")


class MarketCommitWizardData(sdk2.Task):
    """
    Commit fresh data for wizard Market rule to wizard runtime.
    Check wizard runtime build with the changes before commit.
    """

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 3600

        # custom parameters
        wizard_data = sdk2.parameters.Resource(
            "Market wizard data resource id",
            required=True,
            resource_type=resources.MARKET_WIZARD_DATA)

        generation = sdk2.parameters.String(
            'Market indexer generation',
            required=True)

        telegram_chat_id = sdk2.parameters.Integer(
            "Telegram chat for unity bot to warn about errors",
            required=True)

        use_real_fresh = sdk2.parameters.Bool(
            "Commit to real production fresh instead of testing",
            default=False,
            required=False)

        caller = sdk2.parameters.String(
            'Host where the task was called',
            default='unspecified',
            required=False)

    @staticmethod
    def __should_be_sandboxed(file_entry, was_sandboxed):
        if file_entry.name.rsplit('.', 1)[-1] in ['trie', 'bin', 'data']:
            return True
        if was_sandboxed:
            return file_entry.stat().st_size >= _1_MB
        else:
            return file_entry.stat().st_size > _2_MB

    @staticmethod
    def __get_wizard_data_resource_md5(resource_id):
        if not resource_id:
            return None
        for resource in sdk2.Resource["WIZARD_DATA"].find(id=resource_id).limit(1):
            return resource.md5
        return None

    @staticmethod
    def __calc_file_md5(file_path):
        m = hashlib.md5()
        with open(file_path, 'r') as input:
            m.update(input.read())
        return m.hexdigest()

    @staticmethod
    def __make_paths_in_patch_relative(patch):
        return re.sub(r'(\s)\/\S+((?:(?:market)|(?:MarketGztBuilder))\S+)(\s)',
                      r'\1\2\3', patch)

    def __create_wizard_data_resource(self, from_file, path):
        logging.info("Creating resource for file %s at %s", from_file, path)

        wizard_data_file = resource_types.WIZARD_DATA(
            self,
            '{} for Market generation {}'.format(path, self.Parameters.generation),
            path,
            ttl="inf",
            owner=self.owner)

        wizard_data_file_resource = sdk2.ResourceData(wizard_data_file)
        with open(from_file, 'r') as input:
            wizard_data_file.path.write_bytes(input.read())
        wizard_data_file_resource.ready()

        return wizard_data_file.id

    @staticmethod
    def __make_copy_only_from(value, file_name, yamake, needs_copied_before):
        if 'copy' not in yamake.__dict__:
            yamake.__dict__['copy'] = []

        copied_before = any([file_name in section for section in yamake.copy])
        if needs_copied_before and not copied_before:
            return

        necessary_parts = ['FROM', value, 'DESTINATION', '${RESULT}']
        got_something = False

        for i, section in enumerate(yamake.copy):
            if all([part in section for part in necessary_parts]):
                if file_name not in section and not got_something:
                    yamake.copy[i] = [file_name] + section
                    got_something = True
            else:
                yamake.copy[i] = [item for item in section if item != file_name]

        if not got_something:
            new_section = [file_name] + necessary_parts
            yamake.copy += [new_section]

    def __patch_working_path(self, new_files_dir, working_market_path,
                             working_gzt_path):
        """
        Add new files, patch existing, upload bigger and binary ones to sb
        Don't delete old unused files here, this can be done manually when
        necessary. Don't add non-trivial logic (like gzt compilation) for
        new files, this can also be done manually
        """

        market_yamake_file = os.path.join(str(working_market_path), 'ya.make')
        market_yamake = YaMake(market_yamake_file)
        gzt_yamake_file = os.path.join(str(working_gzt_path), 'ya.make')
        gzt_yamake = YaMake(gzt_yamake_file)

        self.Context.files_to_remove = []
        self.Context.files_to_add = []
        self.Context.resources_to_remove = []
        self.Context.resources_added = []

        for entry in new_files_dir.iterdir():
            is_gzt = entry.name in GZT_TASK_FILES
            yamake = gzt_yamake if is_gzt else market_yamake
            working_path = working_gzt_path if is_gzt else working_market_path

            from_file = os.path.join(str(new_files_dir), entry.name)
            to_file = os.path.join(str(working_path), entry.name)
            existed_before = os.path.exists(to_file)

            cls = MarketCommitWizardData
            sandboxed_resource_id = yamake.get_sandboxed(entry.name).resource
            should_be_sandboxed = cls.__should_be_sandboxed(entry, sandboxed_resource_id)
            mark_resource_to_remove = False

            if should_be_sandboxed:
                if sandboxed_resource_id:
                    md5_old = cls.__get_wizard_data_resource_md5(sandboxed_resource_id)
                    md5_new = cls.__calc_file_md5(from_file)
                    if md5_old == md5_new:
                        logging.debug("File %s has not been changed, skip", entry.name)
                        continue

                elif existed_before:
                    logging.info(("File %s becomes sandboxed, mark "
                                  "its old version in arcadia deleted"),
                                 entry.name)
                    Arcadia.delete(to_file)
                    self.Context.files_to_remove += [entry.name]
                    cls.__make_copy_only_from('${BINDIR}', entry.name, yamake,
                                              True)

                new_resource_id = self.__create_wizard_data_resource(from_file,
                                                                     entry.name)
                self.Context.resources_added += [new_resource_id]
                logging.info("Changing resource for file %s from %d to %d",
                             entry.name, sandboxed_resource_id, new_resource_id)
                yamake.update_sandbox_resource(entry.name,
                                               id=new_resource_id,
                                               compression='FILE')
                if new_resource_id != sandboxed_resource_id:
                    mark_resource_to_remove = True
            else:
                shutil.copy(from_file, to_file)
                if sandboxed_resource_id or not existed_before:
                    logging.info(("File %s didn't exist or was sandboxed, so "
                                  "add it to arcadia and remove from ya.make"),
                                 entry.name)
                    yamake.remove_sandbox_resource(entry.name)
                    mark_resource_to_remove = True
                    Arcadia.add(to_file, parents=True)
                    self.Context.files_to_add += [entry.name]
                    cls.__make_copy_only_from('${CURDIR}', entry.name, yamake,
                                              existed_before or sandboxed_resource_id)

            if mark_resource_to_remove and sandboxed_resource_id:
                self.Context.resources_to_remove += [sandboxed_resource_id]

        with open(market_yamake_file, 'w') as y:
            market_yamake.dump(y)

        with open(gzt_yamake_file, 'w') as y:
            gzt_yamake.dump(y)

    @staticmethod
    def __print_ex(ex):
        return "%s: %s" % (str(type(ex)), ex.message)

    def __send_to_telegram(self, template, *args):
        class_name = type(self).__name__
        env = 'PRODUCTION' if self.Parameters.use_real_fresh else 'TESTING'
        host = self.Parameters.caller
        msg = (FailTemplates.COMMON_TEMPLATE % (get_task_link(self.id), class_name, host, env)) + (template % args)
        chat_id = self.Parameters.telegram_chat_id
        try:
            bot = telegram.TelegramBot(sdk2.Vault.data("MARKET", "robot_market_io_telegram_token"))
            bot.send_message(chat_id, msg, parse_mode=telegram.ParseMode.MARKDOWN)
        except Exception as ex:
            logging.error("Couldn't send error message to chat %d; exception %s; message [%s]",
                          chat_id, MarketCommitWizardData.__print_ex(ex), msg)

    def __drop_all_resources(self, resource_ids):
        not_dropped_resources = []
        for resource_id in resource_ids:
            try:
                channel.sandbox.drop_resource_attribute(resource_id, 'ttl')
            except Exception as ex:
                logging.error(("TTL for resource with id %d was not dropped "
                               "with exception: %s"), resource_id,
                               MarketCommitWizardData.__print_ex(ex))
                not_dropped_resources += [(get_resource_link(resource_id),
                                           MarketCommitWizardData.__print_ex(ex))]

        if not_dropped_resources:
            not_dropped_info = '\n'.join(['%s | %s' % (r, msg) for r, msg in not_dropped_resources])
            self.__send_to_telegram(FailTemplates.COULD_NOT_DROP_RESOURCES_TEMPLATE, not_dropped_info)

    def __fail(self, template, *args):
        logging.error(template % args)
        self.__send_to_telegram(template, *args)

        logging.info("Drop created resources")
        self.__drop_all_resources(self.Context.resources_added)

        raise TaskError(template % args)

    def on_execute(self):
        logging.info("on_execute")

        working_fresh_path = self.path('fresh')
        working_market_path = working_fresh_path.joinpath('market')
        working_gzt_path = working_fresh_path.joinpath('MarketGztBuilder')

        repo_fresh_path = '{}/market/testing/begemot_data/fresh/'.format(
                              WizardRuntimeBuild.RuntimeDataSource.ArcadiaRoot)
        if self.Parameters.use_real_fresh:
            repo_fresh_path = WizardRuntimeBuild.RuntimeDataSource.Arcadia

        repo_market_path = os.path.join(repo_fresh_path, 'market')

        current_revision = self.Context.latest_market_data_rev
        logging.info('Current revision is %s', str(current_revision))

        Arcadia.checkout(repo_fresh_path, str(working_fresh_path),
                         depth='empty', revision=current_revision)
        fresh_latest_revision = int(Arcadia.info(repo_fresh_path)['commit_revision'])
        Arcadia.update(str(working_market_path), depth='immediates',
                       revision=fresh_latest_revision)
        Arcadia.update(str(working_gzt_path), depth='immediates',
                       revision=fresh_latest_revision)

        with self.memoize_stage.create_children:
            self.Context.has_diff = False

            wizard_data_path = sdk2.ResourceData(self.Parameters.wizard_data).path

            self.__patch_working_path(wizard_data_path, working_market_path, working_gzt_path)

            generated_patch_resource = resource_types.WIZDATA_PATCH(
                self,
                'new Market data patch from {}'.format(self.Parameters.generation),
                'market_wizard_data.patch'
            )
            resource_id = generated_patch_resource.id

            diff = Arcadia.diff(str(working_fresh_path))

            diff_relative_paths = MarketCommitWizardData.__make_paths_in_patch_relative(diff)

            generated_patch = sdk2.ResourceData(generated_patch_resource)
            generated_patch.path.write_bytes(diff_relative_paths)
            generated_patch.ready()

            if not diff:
                logging.info("Nothing changed in wizard data, skip")
                return
            self.Context.has_diff = True

            latest = int(Arcadia.info(repo_fresh_path)['commit_revision'])
            self.Context.latest_market_data_rev = latest

            runtime_input_parameters = {
                WizardRuntimeBuild.BegemotShards.name: 'Wizard',
                WizardRuntimeBuild.RuntimeDataSource.name: repo_fresh_path,
                WizardRuntimeBuild.ArcadiaRevision.name: latest,
                WizardRuntimeBuild.DoCommit.name: False,
                WizardRuntimeBuild.BuildForProduction.name: False,
                WizardRuntimeBuildPatched.ArcadiaPatchParam.name: resource_id,
                CypressCache.name: '',
                'do_not_restart': True,
                'fail_on_any_error': True
            }

            WizardRuntimeBuildPatchedClass = sdk2.Task[WIZARD_RUNTIME_BUILD_PATCHED]
            runtime_build_task = WizardRuntimeBuildPatchedClass(
                self,
                description='Validating and commiting patch for Market generation {}'.format(
                    self.Parameters.generation),
                priority=self.Parameters.priority,
                create_sub_task=True,
                owner=self.owner,
                **runtime_input_parameters
            ).enqueue()

            raise sdk2.WaitTask(
                [runtime_build_task],
                Status.Group.FINISH | Status.Group.BREAK,
                wait_all=True,
                timeout=2100)

        if not self.Context.has_diff:
            logging.info("Nothing changed in wizard data, skip")
            return

        runtime_build_task = self.find(children=True).first()
        if runtime_build_task.status != Status.SUCCESS:
            self.__fail(
                FailTemplates.SUBTASK_FAILED_TEMPLATE,
                WIZARD_RUNTIME_BUILD_PATCHED,
                get_task_link(runtime_build_task.id),
                str(runtime_build_task.status))

        logging.info("Wizard runtime data build succeeded, commit new data")

        generated_patch = resource_types.WIZDATA_PATCH.find(task=self).first()
        generated_patch_data = sdk2.ResourceData(generated_patch)
        generated_patch_path = str(generated_patch_data.path.absolute())

        logging.info("Obtaining patch from %s", generated_patch_path)
        final_diff = ''
        with open(generated_patch_path, 'r') as input:
            final_diff = input.read()

        patch = subprocess.Popen(['patch', '-p0'],
                                 cwd=str(working_fresh_path),
                                 stdin=subprocess.PIPE,
                                 stdout=subprocess.PIPE,
                                 stderr=subprocess.PIPE)
        stdout, stderr = patch.communicate(final_diff)
        retcode = patch.poll()

        if retcode:
            self.__fail(FailTemplates.PATCH_FAILED_TEMPLATE, retcode, stdout, stderr,
                        get_resource_link(generated_patch.id))

        logging.info("Mark removed and added files in svn")

        def mark_svn(working_path):
            for entry in working_path.iterdir():
                full_path = os.path.join(str(working_path), entry.name)
                if entry.name in self.Context.files_to_remove:
                    Arcadia.revert(full_path)
                    Arcadia.delete(full_path)
                if entry.name in self.Context.files_to_add:
                    Arcadia.add(full_path)

        mark_svn(working_market_path)
        mark_svn(working_gzt_path)

        logging.info("Ready to commit %s", str(working_fresh_path))
        try:
            Arcadia.commit(str(working_fresh_path),
                           'Commiting new data for generation {} with sandbox task {} SKIP_CHECK'.format(
                                self.Parameters.generation, get_task_link(self.id)),
                           'zomb-sandbox-rw')
        except Exception as ex:
            self.__fail(FailTemplates.COMMIT_FAILED_TEMPLATE,
                        MarketCommitWizardData.__print_ex(ex),
                        get_resource_link(generated_patch.id))
        else:
            logging.info("Data commited successfully, drop removed resources")
            self.__drop_all_resources(self.Context.resources_to_remove)
