import logging
import json
import time
import requests
import requests.adapters
from sandbox import sdk2
from sandbox.common.utils import Enum
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import ReleaseStatus
from requests.packages.urllib3.util.retry import Retry
from sandbox.projects.common.juggler import jclient
from sandbox.projects.resource_types import TASK_LOGS
from sandbox.projects.market.report.common.pinger_data import PingerData


class Environment(Enum):
    UNSTABLE = ReleaseStatus.UNSTABLE
    TESTING = ReleaseStatus.TESTING
    PRESTABLE = ReleaseStatus.PRESTABLE
    STABLE = ReleaseStatus.STABLE
    EXPERIMENTAL = 'productionexp1'


class MarketDeployWithReductor(sdk2.Task):
    """Task to deploy packages using reductor and backctld
    """
    CHECK_PERIOD = 10
    MAX_RETRIES = 3
    _installer_url = None

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 2 * 60 * 60
        environment = sdk2.parameters.String(
            'Environment',
            required=True,
            choices=[(value.capitalize(), value) for value in list(Environment)],
            default=Environment.TESTING)
        package = sdk2.parameters.Resource('Package', required=True)
        wait_for_reload = sdk2.parameters.Bool('Wait for reload', default=False)
        release_ticket = sdk2.parameters.String('Release ticket', required=False)

        with sdk2.parameters.Group("Juggler") as juggler_group:
            send_juggler_events = sdk2.parameters.Bool(
                "Send juggler events",
                default_value=False
            )

            with send_juggler_events.value[True]:
                juggler_host = sdk2.parameters.String("Juggler host")
                juggler_service = sdk2.parameters.String("Juggler service")


    def on_execute(self):
        self._installer_url = self._get_master_indexer_url() + '/yandex/reductor/installer_api.py'
        logging.info('Using installer {}'.format(self._installer_url))

        self.Parameters.package.ttl = 'inf'
        self._deploy_package()

        if self.Parameters.wait_for_reload:
            self._wait_for_package(self.Parameters.environment, self.Parameters.package.resource_name, self.Parameters.package.resource_version)

    def on_success(self, prev_status):
        self.send_juggler_event(status="OK", description="Successful release: https://sandbox.yandex-team.ru/task/{}".format(sdk2.Task.current.id))
        super(MarketDeployWithReductor, self).on_success(prev_status)

    def _get_master_indexer_url(self):
        if self.Parameters.environment == Environment.TESTING:
            return 'http://mi01ht.market.yandex.net'

        indexers = {
            'stratocaster': 'http://mi01h.market.yandex.net',
            'gibson': 'http://mi01v.market.yandex.net',
        }
        session = self._make_session()
        master = None
        for url in indexers.values():
            logging.info('Requesting {} to determine master indexer'.format(url))
            try:
                response = session.get(url + '/marketindexer/whoismaster.py')
                response.raise_for_status()
                master = response.text.strip()
                break
            except requests.exceptions.RequestException as error:
                logging.warn('Error requesting {}: {}'.format(url, error))

        if not master:
            raise TaskFailure('Could not determine master indexer')
        master_url = indexers[master]
        logging.info('Master indexer is {} ({})'.format(master, master_url))
        return master_url

    def _deploy_package(self):
        name = self.Parameters.package.resource_name
        version = self.Parameters.package.resource_version
        skynet_link = self.Parameters.package.skynet_id
        sandbox_task_id = self.Parameters.package.task_id
        environment = self.Parameters.environment
        logging.info('Deploying package {name} {ver} to {env} ({sky}). Parent task: {task_id}'
                     .format(name=name, ver=version, env=environment, sky=skynet_link, task_id=sandbox_task_id))

        self._run_task({
            'command': 'deploy_package',
            'environment': environment,
            'package': name,
            'version': version,
            'skynet_link': skynet_link,
            'sandbox_task_id': sandbox_task_id,
        })

    def _wait_for_package(self, env, package, version, timeout=2400):
        pinger = PingerData(env)
        sleep_time = 10
        retries = int(timeout / sleep_time)
        while True:
            value = 0.0
            try:
                pinger.update()
                value = pinger.get_installed_package_percent(package, version)
            except Exception as e:
                logging.exception("Failed to update pinger data, error: ", e)
            if (value >= 0.9):
                # 90% hosts with target version
                break
            elif retries <= 1:
                logging.info('Package {} version {} deploy timeout'.format(package, version))
                break
            else:
                retries -= 1
                time.sleep(sleep_time)

    def _run_task(self, parameters):
        request_data = dict(parameters, action='create_task')
        task = self._request_installer(request_data)
        logging.info('Created task {}'.format(task['id']))

        while task['is_running']:
            time.sleep(self.CHECK_PERIOD)
            logging.info('Checking task {}'.format(task['id']))
            task = self._request_installer({
                'action': 'get_task',
                'task_id': task['id'],
            })

        if not task['is_successful']:
            raise TaskFailure('Task {id} ({cmd}) failed: {res}'
                              .format(id=task['id'], cmd=task['command'], res=task['result']))

        # mark TASK_LOGS resource with ttl=inf(CSADMIN-24236)
        logs_resource = sdk2.Resource.find(type=TASK_LOGS, task_id=self.id).first()
        logs_resource.ttl = 'inf'
        logging.info('Task {} finished successfully'.format(task['id']))

    def _request_installer(self, parameters):
        session = self._make_session()
        request_data = json.dumps(parameters)
        logging.debug('Requesting installer at {}: {}'.format(self._installer_url, request_data))
        response = session.post(self._installer_url, request_data)
        response.raise_for_status()
        logging.debug('Got response from installer: {}'.format(response.text))

        response_json = response.json()
        if not response_json.get('is_request_successful'):
            error = response_json.get('request_error')
            raise TaskFailure('Request to installer failed: {}'.format(error))
        return response_json.get('result')

    @classmethod
    def _make_session(cls):
        session = requests.Session()
        session.mount('http://', requests.adapters.HTTPAdapter(max_retries=Retry(
                total=cls.MAX_RETRIES,
                backoff_factor=1.0,
                status_forcelist=[500, 502, 503, 504],
            )))
        return session

    def send_juggler_event(self, status, description):
        if not self.Parameters.send_juggler_events:
            return
        juggler_host = self.Parameters.juggler_host
        juggler_service = self.Parameters.juggler_service

        if not juggler_host or not juggler_service:
            return

        jclient.send_events_to_juggler(juggler_host, juggler_service, status, description)
        self.Context.juggler_status = status
        self.set_info("Send juggler event. Host='{}', service='{}', status='{}', description='{}'".format(
            juggler_host,
            juggler_service,
            status,
            description
        ))

