import os

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.market.report.common.stat import (
    LOG_DIR,
    SOLOMON_COMMON_LABELS,
    Vault,
    get_rounded_time,
    calc,
    send_stat,
    get_table_name_time
)

SENSOR_MAP = {
    'stat': 'percent',
    'count': 'abs',
    'spend_time': 'cpu_time_us',
    'mean_spend_time': 'mean_cpu_time_us'
}


class MarketReportAutomaticStat(sdk2.Task):
    """Automatically calc and send stat"""

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment("yandex-yt"),
        ]

    class Parameters(sdk2.Parameters):
        description = "Automatically calc and send stat"
        kill_timeout = 20 * 60  # seconds

        vault = Vault()

        clusters = sdk2.parameters.List('Clusters', default=['arnold', 'hahn'])

        log_dir = sdk2.parameters.String('YT log dir', default=LOG_DIR)
        stat_dir = sdk2.parameters.String('Stat dir', default='//tmp/user/stat/')

        custom_sensor = sdk2.parameters.Dict('Map Stat Data to sensors', default=SENSOR_MAP)

        advanced_stat_params = sdk2.parameters.List('Advanced yt_stat params', description='Try --help')

        common_labels = sdk2.parameters.Dict("Solomon common labels", default=SOLOMON_COMMON_LABELS)

    def calc_stat(self, cluster, log, output, table_time):
        inputs = ['-i {}'.format(log)]
        vault = self.Parameters.vault
        labels = self.Parameters.common_labels
        advanced = list(self.Parameters.advanced_stat_params)

        calc(self, cluster, inputs, output, advanced, 'stat')

        for value_column in self.Parameters.custom_sensor:
            sensor_name = self.Parameters.custom_sensor[value_column]
            send_stat(vault, cluster, output, sensor_name, value_column, labels, table_time)

    def on_execute(self):
        table_time = get_rounded_time()
        table_name = get_table_name_time(table_time)
        input_name = self.Parameters.log_dir + table_name
        output = self.Parameters.stat_dir + table_name
        clusters = list(self.Parameters.clusters)

        vault = self.Parameters.vault

        tokens = vault.tokens.data()
        os.environ['YT_TOKEN'] = tokens[vault.yt_token_name]

        failures_count = 0
        for cluster in clusters:
            try:
                self.calc_stat(cluster, input_name, output, table_time)
            except sdk2.helpers.subprocess.CalledProcessError:
                failures_count += 1
                continue
            break

        if failures_count == len(clusters):
            raise Exception('Statistics has not been sent')
