import logging

import sandbox.common.types.task as ctt
import sandbox.projects.common.constants as consts
from sandbox import sdk2
from sandbox.common import errors
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.market.report.common import helpers
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import binary_task


logger = logging.getLogger('MarketReportCacheHeater')

YT_PROXY = 'hahn.yt.yandex.net'

COMMON_HEATER_OPTIONS = dict(
    __requirements__={
        'cores': 56
    },
    allow_revision=True,  # Allow to heat specific revision
    build_system=consts.SEMI_DISTBUILD_BUILD_SYSTEM,
    build_type=consts.RELEASE_BUILD_TYPE,
    binary_executor_release_type='stable',
    check_return_code=False,
    distbuild_pool="//vla/users/market/report",
    env_vars='YA_TOKEN=$(vault:value:mkoterev:ya_token)',  # For distbuild
    failed_tests_cause_error=False,
    ignore_recurses=False,
    keep_on=True,
    ya_yt_proxy='hahn.yt.yandex.net',
    ya_yt_dir='//home/market-report-dist-cache',
    ya_yt_put=True,
    ya_yt_store_codec='zstd08_1',
    ya_yt_replace_result=True,
    ya_yt_replace_result_add_objects=True,
    ya_yt_max_cache_size=2089072092774,  # 1.9Tb
    ya_yt_store_threads=32,
    result_single_file=False,
)


class MarketReportCacheHeater(binary_task.LastBinaryTaskRelease, sdk2.Task):
    '''Heat report cache if there is appropriate revision'''

    class Parameters(sdk2.Task.Parameters):
        store_binaries = sdk2.parameters.Bool('Add binaries to cache', default=False)
        first_run = sdk2.parameters.Bool('Yt tables are empty and we want to fill them with actual revision', default='false')
        dont_heat_dependencies = sdk2.parameters.Bool('Heat only binaries of report', default=False)

        wait_time_if_no_revision = sdk2.parameters.Integer(
            'Time to wait if no revisions from TSUM found (in seconds)', default=30
        )

        binary_release = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Tokens'):
            yt_store_token = sdk2.parameters.YavSecret(
                label='Yav YT token', description='YT_STORE_TOKEN from Sandbox Vault is used by default'
            )

    def on_execute(self):
        with self.memoize_stage.run_report_heater:
            self._run_first_heat_task()
            self._begin_wait_for_heater()

        rev_to_heat = getattr(self.Context, 'revision_to_heat', None)
        # Strange if statement but classic "rev_to_heat is None" didn't work
        if type(rev_to_heat) != int:
            self.set_info('Task finished after wait because rev_to_heat=%r' % rev_to_heat)
            return

        heater_task = sdk2.Task[self.Context.heater_task_id]
        if heater_task.status not in ctt.Status.Group.SUCCEED:
            raise errors.TaskFailure('ERROR! Child heater task failed')

        if self.Parameters.dont_heat_dependencies:
            self.set_info('Task finished sucsessfuly')
            return

        with self.memoize_stage.run_deps_heater:
            self._run_second_heat_task()
            self._begin_wait_for_heater()

        self.set_info('Task finished sucsessfuly')

    def _run_first_heat_task(self):
        new_report_revision = None
        if self.Parameters.first_run:
            new_report_revision = max(self._get_new_revisions(paths=('/market/report', '/market/library'), rev_to=None, limit=2))
        else:
            last_revision_with_yt_cache = self._get_last_revision_with_yt_cache()
            new_report_revisions = self._get_new_revisions(
                paths=('/market/report', '/market/library'),
                rev_from=last_revision_with_yt_cache,
            )
            if len(new_report_revisions) == 0:
                wait_time = self.Parameters.wait_time_if_no_revision
                self.set_info('No new revisions found, sleep for %i seconds, then finish' % wait_time)
                raise sdk2.WaitTime(wait_time)  # Sleep

            new_report_revision = max(new_report_revisions)

            if last_revision_with_yt_cache >= new_report_revision:
                wait_time = self.Parameters.wait_time_if_no_revision
                self.set_info('No new revisions found, sleep for %i seconds, then finish' % wait_time)
                raise sdk2.WaitTime(wait_time)  # Sleep

        self.Context.revision_to_heat = new_report_revision

        self.set_info('Found new revision from TSUM: %i, starting YT_HEATER task' % new_report_revision)

        logger.info('We have new revisions, launching YT_HEATER task')

        self._run_heater_task()
        logger.info('Task for heating report created, waiting for finish')
        self.set_info(
            'Waiting for report cache heater : {}'.format(
                lb.task_link(self.Context.heater_task_id)
            ),
            do_escape=False
        )

    def _run_second_heat_task(self):
        self.set_info('Running second task for revision %i' % self.Context.revision_to_heat)
        self._run_heater_task_for_deps()
        logger.info('Task for heating report\'s package deps created, waiting for finish')
        self.set_info(
            'Waiting for report\'s package deps cache heater : {}'.format(
                lb.task_link(self.Context.heater_task_id)
            ),
            do_escape=False
        )

    def _run_heater_task(self):
        report_heat_task = sdk2.Task['YT_HEATER'](
            self,
            description='Cache heater for market/report/report_base and market/report/report_meta (with optimisations)',
            kill_timeout=1*60*60,  # One hour timeout
            checkout_arcadia_from_url=Arcadia.ARCADIA_TRUNK_URL + '@' + str(self.Context.revision_to_heat),
            heater_name='MarketReportCacheHeater_report',
            ya_yt_token_yav_secret=self.Parameters.yt_store_token,
            ya_yt_replace_result_rm_binaries=not self.Parameters.store_binaries,
            thinlto=True,
            definition_flags='-DCFLAGS="-mllvm -inline-threshold=1000 -fno-omit-frame-pointer" -DLDFLAGS="-Wl,--thinlto-jobs=all"',  # TODO(MSI-1580)
            targets='market/report/report_base;market/report/report_meta',
            **COMMON_HEATER_OPTIONS
        )
        report_heat_task.enqueue()
        self.Context.heater_task_id = report_heat_task.id

    def _run_heater_task_for_deps(self):
        targets = (
            'market/report/lite;'
            'market/report/basesearch;'
            'market/report/runtime_cloud/report_ctl;'
            'market/report/runtime_cloud/report_ctl/tests;'
            'market/report/runtime_cloud/report_config/tests;'
            'market/report/runtime_cloud/scripts;'
            'market/tools/reanimator'
        )
        description = (
            'Cache heater for dependencies of packages market/report/runtime_cloud/yandex-market-report.json'
            ' and market/report/runtime_cloud/yandex-market-report-meta.json'
        )
        logger.debug('Targets (deps) : %r' % targets.split(';'))
        deps_heat_task = sdk2.Task['YT_HEATER'](
            self,
            description=description,
            kill_timeout=1*60*60,  # 60 minutes timeout
            checkout_arcadia_from_url=Arcadia.ARCADIA_TRUNK_URL + '@' + str(self.Context.revision_to_heat),
            force_build_depends=True,
            heater_name='MarketReportCacheHeater_deps',
            ya_yt_token_yav_secret=self.Parameters.yt_store_token,
            ya_yt_replace_result_rm_binaries=False,  # Store deps binaries
            targets=targets,
            **COMMON_HEATER_OPTIONS
        )
        deps_heat_task.enqueue()
        self.Context.heater_task_id = deps_heat_task.id

    def _begin_wait_for_heater(self):
        raise sdk2.WaitTask(self.Context.heater_task_id, (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))

    def _get_new_revisions(self, paths, rev_from=None, rev_to='HEAD', limit=100):
        new_revisions = []
        for path in paths:
            new_revisions.extend(
                Arcadia.log(
                    url=Arcadia.ARCADIA_TRUNK_URL + path,
                    revision_from=rev_from,
                    revision_to=rev_to,
                    limit=limit,
                )
            )
        only_revisions = [i.get('revision') for i in new_revisions]
        logger.info('Revisions from %r : %r' % (paths, only_revisions))
        return only_revisions

    def _get_last_revision_with_yt_cache(self):
        logger.debug('Trying to read yt table')
        from yt.wrapper import YtClient
        from yt.wrapper.format import YsonFormat

        ytc = YtClient(proxy=YT_PROXY, token=helpers.get_yt_store_token(self))

        # Stat table is static and we can't use select_rows :(
        # https://yt.yandex-team.ru/docs/api/commands.html#select_rows
        # (only for dynamic tables)
        stat_table = ytc.read_table(
            '//home/market-report-dist-cache/stat', format=YsonFormat(format='text', require_yson_bindings=False)
        )

        last_revision = (
            max(stat_table, key=lambda x: x.get('timestamp') if x.get('key') == 'heater' else 0)
            .get('value')
            .get('svn_revision')
        )

        logger.info('Last revision from yt = %r' % last_revision)
        return last_revision
