from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.client import Tag
from sandbox.common.types import task
from sandbox.projects.common import link_builder
from sandbox.projects.common.nanny import nanny
import sandbox.projects.common.build.parameters as build_parameters
from sandbox.projects.market.report.MarketReportRelease2 import MarketReportRelease2
from sandbox.projects.market.report.MarketReportGenerateChangelog import svn, generate_changelog
import os


class BuildVersionParameter(sdk2.parameters.String):
    name = 'build_version'
    description = 'Build version'


class MarketReportCdRelease2(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 4

    class Parameters(sdk2.Task.Parameters):
        build_system = build_parameters.BuildSystem()
        checkout_arcadia_from_url = build_parameters.ArcadiaUrl()
        packages = sdk2.parameters.String("Package path", required=True)
        resource_type = build_parameters.ResourceType()
        run_tests = build_parameters.RunTests()
        run_long_tests = build_parameters.RunLongTests()
        startrek_ticket_ids = nanny.StartrekTicketIdsParameter2("Startrek ticket ids")
        build_version = BuildVersionParameter()

    def on_execute(self):
        with self.memoize_stage.wait_build:
            arcadia_patch = self.get_arcadia_patch()
            resource = sdk2.Resource[self.Parameters.resource_type](
                self,
                self.Parameters.packages,
                os.path.join(os.getcwd(), "{}.{}.tar.gz".format(
                    self.get_package_name(self.Parameters.packages),
                    self.Parameters.build_version
                )),
            )
            subtask_parameters = {
                "build_system": self.Parameters.build_system,
                "checkout_arcadia_from_url": self.Parameters.checkout_arcadia_from_url,
                "packages": self.Parameters.packages,
                "resource_type": self.Parameters.resource_type,
                "run_tests": self.Parameters.run_tests,
                "run_long_tests": self.Parameters.run_long_tests,
                "startrek_ticket_ids": self.Parameters.startrek_ticket_ids,
                "arcadia_patch": arcadia_patch,
                "resource_id": str(resource.id)
            }
            build_task = MarketReportRelease2(
                self,
                description="Build {resource_type} release version {build_version}".format(
                    resource_type=self.Parameters.resource_type,
                    build_version=self.Parameters.build_version
                ),
                hints=list(self.hints),
                **subtask_parameters
            )
            build_task.enqueue()
            self.Context.build_task_id = build_task.id
            self.set_info('INFO: Build subtask enqueued {}'.format(link_builder.task_link(build_task.id)), do_escape=False)
            raise sdk2.WaitTask(build_task.id, task.Status.Group.FINISH | task.Status.Group.BREAK)
        build_task = sdk2.Task[self.Context.build_task_id]
        if build_task.status not in task.Status.Group.SUCCEED:
            raise errors.TaskFailure('ERROR: Building report release failed')
        self.set_info('INFO: Report release built successfully')

    def get_package_name(self, path):
        return path[path.rfind('/')+1:path.rfind('.')]

    def get_arcadia_patch(self):

        if not self.Parameters.packages or not self.Parameters.build_version:
            return None

        branch_path = self.Parameters.checkout_arcadia_from_url
        at_index = branch_path.index('@')
        if at_index >= 0:
            branch_path = branch_path[:at_index]

        build_version = self.Parameters.build_version
        package_name = self.Parameters.packages
        package_name = package_name[package_name.rfind('/')+1:package_name.rfind('.')]
        ya_package_patch = svn.patch_ya_package_version(branch_path, build_version, should_commit=False, package_name=package_name)

        changelog_content = generate_changelog(build_version, build_version, [], package_name=package_name)
        _, changelog_patch = svn.update_changelog(branch_path, changelog_content, should_commit=False)

        self.set_info('INFO: Arcadia patch generated successfully')
        return ya_package_patch + '\n' + changelog_patch
