# coding: utf-8

import json
import logging
import os

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task as ctt
from sandbox.common.types import client as ctc
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import task_env
from sandbox.projects.common.constants import constants
from sandbox.projects.market.report.common import helpers
from sandbox.projects.market.report.common.lite_test_runner import LiteTestYaMakeRunner


logger = logging.getLogger(__name__)


def get_project_targets(project_path):
    with open(project_path) as f:
        project = json.load(f)
        if 'targets' in project['build']:
            targets = project['build']['targets']
        else:
            targets = []
            for build_key in project['build']:
                targets += project['build'][build_key]['targets']
        targets += ['market/report/lite']
        return targets


class MarketReportCdReleaseTests(sdk2.Task):
    class Requirements(task_env.BuildRequirements):
        client_tags = (
            (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC)
            & (ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL | ctc.Tag.LINUX_BIONIC)
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Input parameters'):
            attempts_number = sdk2.parameters.Integer('Number of attempts to run tests', default=5)
            checkout_arcadia_from_url = sdk2.parameters.String('Svn url for arcadia', required=True)
            packages = sdk2.parameters.String('Package path, related to arcadia', required=True)
            run_tests = sdk2.parameters.Bool('Run tests', default=True)
            run_long_tests = sdk2.parameters.Bool('Run long tests', default=True)

        with sdk2.parameters.Group('Tokens') as build_group:
            arc_token = sdk2.parameters.YavSecret(
                label='Yav ARC token', description='ARC_TOKEN from Sandbox Vault is used by default')
            yt_store_token = sdk2.parameters.YavSecret(
                label='Yav YT token', description='YT_STORE_TOKEN from Sandbox Vault is used by default')
            ya_token = sdk2.parameters.YavSecret(
                label='Yav YA token', description='YA_TOKEN from Sandbox Vault is used by default')

        with sdk2.parameters.Group('Debug options'):
            debug_options = sdk2.parameters.Bool('Enable debug options', default=False)
            with debug_options.value[True]:
                test_filters = sdk2.parameters.String('Only run tests that match this string (space separated list)')
                rearr_factors = sdk2.parameters.String(
                    'Rearrangement factors (semicolon separated pairs, for example "factor1=0;factor2=text")')

    def on_execute(self):
        if not self.Parameters.run_tests and not self.Parameters.run_long_tests:
            return

        arc_token = helpers.get_arc_token(self)
        yt_store_token = helpers.get_yt_store_token(self)
        ya_token = helpers.get_ya_token(self)

        with LiteTestYaMakeRunner(
            self,
            arc_token=arc_token,
            yt_store_token=yt_store_token,
            ya_token=ya_token,
            arcadia_url=self.Parameters.checkout_arcadia_from_url,
            attempts_number=self.Parameters.attempts_number,
        ) as runner:
            targets = get_project_targets(os.path.join(runner.source_root, self.Parameters.packages))
            lite_test_args = {'with_rearr': self.Parameters.rearr_factors or None}
            test_filters = self.Parameters.test_filters or None
            test_result = runner.run_tests(
                targets=targets,
                lite_test_args=lite_test_args,
                test_filters=test_filters,
                distbuild_pool='//vla/users/market/report'
            )
            if not test_result:
                raise errors.TaskFailure('ERROR: All attempts to run tests were failed')
            self.set_info('INFO: Project tests completed successfully')
