# coding: utf-8

import json
import logging
import os

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task as ctt
from sandbox.common.types import client as ctc
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import task_env
from sandbox.projects.common.constants import constants
from sandbox.projects.market.report.common import helpers
from sandbox.projects.market.report.common.lite_test_runner import LiteTestYaMakeRunner
from sandbox.projects.market.report.MarketReportCdRelease import MarketReportCdRelease
from sandbox.projects.market.report.MarketReportCdMetaRelease import MarketReportCdMetaRelease


logger = logging.getLogger(__name__)


MARKET_META_REPORT = 'MARKET_META_REPORT'


def get_project_targets(project_path):
    with open(project_path) as f:
        project = json.load(f)
        if 'targets' in project['build']:
            targets = project['build']['targets']
        else:
            targets = []
            for build_key in project['build']:
                targets += project['build'][build_key]['targets']
        targets += ['market/report/lite']
        return targets


class MarketReportCdReleaseWrapper(sdk2.Task):
    class Requirements(task_env.BuildRequirements):
        client_tags = (
            (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC)
            & (ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL | ctc.Tag.LINUX_BIONIC)
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Input parameters'):
            attempts_number = sdk2.parameters.Integer('Number of attempts to run tests', default=5)
            # All relevant parameters of MarketReportCdRelease task
            checkout_arcadia_from_url = sdk2.parameters.String('Svn url for arcadia', required=True)
            packages = sdk2.parameters.String('Package path, related to arcadia', required=True)
            resource_type = sdk2.parameters.String('Created package resource type', required=True)
            run_tests = sdk2.parameters.Bool('Run tests', default=True)
            run_long_tests = sdk2.parameters.Bool('Run long tests', default=True)
            startrek_ticket_ids = sdk2.parameters.String('Startrek ticket ids', required=True)
            build_version = sdk2.parameters.String('Build version', required=True)

        with sdk2.parameters.Group('Tokens') as build_group:
            arc_token = sdk2.parameters.YavSecret(
                label='Yav ARC token', description='ARC_TOKEN from Sandbox Vault is used by default')
            yt_store_token = sdk2.parameters.YavSecret(
                label='Yav YT token', description='YT_STORE_TOKEN from Sandbox Vault is used by default')
            ya_token = sdk2.parameters.YavSecret(
                label='Yav YA token', description='YA_TOKEN from Sandbox Vault is used by default')

        with sdk2.parameters.Group('Debug options'):
            debug_options = sdk2.parameters.Bool('Enable debug options', default=False)
            with debug_options.value[True]:
                test_filters = sdk2.parameters.String('Only run tests that match this string (space separated list)')
                rearr_factors = sdk2.parameters.String(
                    'Rearrangement factors (semicolon separated pairs, for example "factor1=0;factor2=text")')

    def on_execute(self):
        with self.memoize_stage.run_tests:
            self._start_building_package()
            self._run_all_tests()
            self._begin_waiting_for_package()

        build_task = sdk2.Task[self.Context.build_task_id]
        if build_task.status not in ctt.Status.Group.SUCCEED:
            raise errors.TaskFailure('ERROR: Building report release failed')

        self.set_info('INFO: Report release built successfully')

    def _run_all_tests(self):
        if not self.Parameters.run_tests and not self.Parameters.run_long_tests:
            return

        arc_token = helpers.get_arc_token(self)
        yt_store_token = helpers.get_yt_store_token(self)
        ya_token = helpers.get_ya_token(self)

        with LiteTestYaMakeRunner(
            self,
            arc_token=arc_token,
            yt_store_token=yt_store_token,
            ya_token=ya_token,
            arcadia_url=self.Parameters.checkout_arcadia_from_url,
            attempts_number=self.Parameters.attempts_number,
        ) as runner:
            targets = get_project_targets(os.path.join(runner.source_root, self.Parameters.packages))
            lite_test_args = {'with_rearr': self.Parameters.rearr_factors or None}
            test_filters = self.Parameters.test_filters or None
            test_result = runner.run_tests(
                targets=targets,
                lite_test_args=lite_test_args,
                test_filters=test_filters,
                distbuild_pool='//vla/users/market/report'
            )
            if not test_result:
                sdk2.Task[self.Context.build_task_id].stop()
                raise errors.TaskFailure('ERROR: All attempts to run tests were failed')
            self.set_info('INFO: Project tests completed successfully')

    def _start_building_package(self):
        task_type = MarketReportCdMetaRelease.type if (self.Parameters.resource_type == MARKET_META_REPORT) else MarketReportCdRelease.type
        task = sdk2.Task[task_type](
            self,
            description=self.Parameters.description,
            build_system=constants.SEMI_DISTBUILD_BUILD_SYSTEM,
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            packages=self.Parameters.packages,
            resource_type=self.Parameters.resource_type,
            run_tests=False,
            run_long_tests=False,
            startrek_ticket_ids=self.Parameters.startrek_ticket_ids.split(','),
            build_version=self.Parameters.build_version,
        )
        task.enqueue()
        self.Context.build_task_id = task.id
        self.set_info('INFO: Report release build was started, task {}'.format(lb.task_link(task.id)), do_escape=False)

    def _begin_waiting_for_package(self):
        raise sdk2.WaitTask(self.Context.build_task_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)
