import json

import release_changes
from sandbox import sdk2
import svn

from changelog_helper import generate_changelog


class ReportTsumJsonChangelogResource(sdk2.Resource):
    share = True


class ReportTsumBuildInfo(sdk2.Resource):
    share = True


class ReportDebianChangelogResource(sdk2.Resource):
    share = True


class ReportReleasePatch(sdk2.Resource):
    share = True
    ttl = 'inf'


class MarketReportGenerateChangelog(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        previous_branch = sdk2.parameters.ArcadiaUrl('previous version branch', group='Arcadia')
        current_branch = sdk2.parameters.ArcadiaUrl('current version branch', group='Arcadia')

        with sdk2.parameters.Output:
            changelog_resource = sdk2.parameters.Resource(
                'debian/changelog',
                resource_type=ReportDebianChangelogResource,
            )
            tsum_changelog_resource = sdk2.parameters.Resource(
                'package.json',
                resource_type=ReportTsumJsonChangelogResource,
            )

            tsum_build_info = sdk2.parameters.Resource(
                'build_info.json',
                resource_type=ReportTsumBuildInfo,
            )

            report_release_patch = sdk2.parameters.Resource(
                'report_release_patch.patch',
                resource_type=ReportReleasePatch,
            )

    def on_execute(self):
        release_commits = release_changes.get_commits_between_branches(self.Parameters.previous_branch,
                                                                       self.Parameters.current_branch)

        report_version = release_changes.get_version_from_branch_url(self.Parameters.current_branch)
        build_version = release_changes.get_next_build_version(self.Parameters.current_branch)

        changelog_content = generate_changelog(build_version, report_version, release_commits.market_commits)

        changelog_content = svn.update_changelog(self.Parameters.current_branch, changelog_content)
        svn.patch_ya_package_version(self.Parameters.current_branch, build_version)

        tsum_changes_format = release_changes.generate_tsum_changelog(build_version, release_commits.market_commits)

        self.Parameters.tsum_changelog_resource = ReportTsumJsonChangelogResource(
            self, 'packages.json', 'packages.json'
        )
        app_res = sdk2.ResourceData(self.Parameters.tsum_changelog_resource)
        app_res.path.write_bytes(
            json.dumps(tsum_changes_format)
        )
        app_res.ready()

        self.Parameters.changelog_resource = ReportDebianChangelogResource(
            self, 'changelog', 'changelog'
        )

        app_res = sdk2.ResourceData(self.Parameters.changelog_resource)
        app_res.path.write_bytes(changelog_content)
        app_res.ready()

        self.Parameters.report_release_patch = ReportReleasePatch(
            self, 'report_release_patch.patch', 'report_release_patch.patch'
        )
        app_res = sdk2.ResourceData(self.Parameters.report_release_patch)
        app_res.path.write_bytes(svn.generate_release_branch_path(self.Parameters.current_branch))
        app_res.ready()

        self.Parameters.tsum_build_info = ReportTsumBuildInfo(
            self, 'build_info.json', 'build_info.json'
        )

        app_res = sdk2.ResourceData(self.Parameters.tsum_build_info)
        app_res.path.write_bytes(
            json.dumps({
                'branchPath': self.Parameters.current_branch,
                'branchRevision': svn.get_last_revision(self.Parameters.current_branch),
                'baseTrunkRevision': str(svn.get_copied_from_revision(self.Parameters.current_branch)),
                'buildVersion': build_version
            })
        )
        app_res.ready()
