# coding=utf-8
import json
import os
import re
import tempfile

import shutil

import svn


def get_version_from_branch_url(branch_url):
    return re.search(r'/(report-)?(?P<version>(\d|\.)+)/', branch_url).group('version')


class ExtCommit:
    REVIEW_REGEX = r'(?i)\bREVIEW:\s*(\d+)\b'
    TICKET_REGEX = r'\b([A-Z]+-\d+)\b'
    MARKET_TICKET_REGEX = r'\b(MARKETOUT-(\d+))\b'

    def __init__(self, commit):
        self.revision = commit.revision
        self.message = unicode(commit.message.replace('\n', ' ').encode('utf-8'), 'utf-8')
        self.author = commit.author
        self.tickets = [name for name, num in re.findall(ExtCommit.MARKET_TICKET_REGEX, commit.message) if int(num) > 0]
        m = re.search(ExtCommit.REVIEW_REGEX, commit.message)
        self.review = m.group(1) if m is not None else None
        self.text = unicode(re.sub(ExtCommit.REVIEW_REGEX, '',
                           re.sub(ExtCommit.TICKET_REGEX, '', commit.message)).strip().replace('\n', ' ').encode(
            'utf-8'), 'utf-8')

        def get_wiki_text_from_commit_message(msg):
            result = msg
            result = re.sub(ExtCommit.TICKET_REGEX, r'https://st.yandex-team.ru/\1', result)
            result = re.sub(ExtCommit.REVIEW_REGEX, r'(((https://a.yandex-team.ru/arc/review/\1/ review)))', result)
            result = unicode(result.strip().replace('\n', ' ').encode('utf-8'), 'utf-8')
            return result

        self.wiki_text = get_wiki_text_from_commit_message(commit.message)

    def __repr__(self):
        return u'{revision} {author:20} {msg}'.format(
            revision=self.revision,
            author=self.author,
            msg=self.message[:100]
        )

    def wiki(self):
        return '((https://a.yandex-team.ru/arc/commit/{revision} r{revision})) ({author}) {msg}'.format(
            revision=self.revision, author=self.author, msg=self.wiki_text)


class ExtCommits:
    """Все коммиты сделанные после отведения релиза after включая релиз up_to_release
       или до последнего коммита, если up_to_release = None
    """

    def __init__(self, after_release=None, up_to_release=None):
        self.after = after_release
        self.up_to = up_to_release
        self.first_trunk_revision = svn.get_release_revision(self.after) + 1
        self.last_trunk_revision = svn.get_release_revision(self.up_to) if self.up_to is not None else None

        self.market_commits, self.non_market_commits = svn.get_commits(
            svn.TRUNK,
            revision_interval=(self.first_trunk_revision, self.last_trunk_revision),
            also_non_market_commits=True
        )

        if self.up_to:
            self.market_commits += svn.get_version_commits(self.up_to)

        self.market_commits = [ExtCommit(c) for c in self.market_commits]
        self.non_market_commits = [ExtCommit(c) for c in self.non_market_commits]

        self.market_tickets = list(sorted(set([ticket for commit in self.market_commits for ticket in commit.tickets])))
        self.non_market_tickets = list(
            sorted(set([ticket for commit in self.non_market_commits for ticket in commit.tickets])))

    def description(self):
        return 'Commits made after release {ver1} up to {ver2} inclusive (trunk revisions r{rev1}:r{rev2})'.format(
            ver1=get_version_from_branch_url(self.after),
            ver2='release {}'.format(get_version_from_branch_url(self.up_to)) if self.up_to else 'last commit',
            rev1=self.first_trunk_revision,
            rev2=self.last_trunk_revision or 'HEAD')


def get_commits_between_branches(svn_branch_from, svn_branch_to):
    return ExtCommits(after_release=svn_branch_from, up_to_release=svn_branch_to)


def generate_tsum_changelog(build_version, market_commits):
    return {
        'packages': [
            {
                'moduleName': 'market/report',
                'version': build_version,
                'changelog': [
                    {
                        'version': build_version,
                        'change': u'*  {0}. Revision: {1} ({2})'.format(market_commit.message, market_commit.revision, market_commit.author),
                        'author': market_commit.author,
                        'revision': str(market_commit.revision)
                    } for market_commit in market_commits
                ]

            }
        ]
    }


def get_next_build_version(release_branch):
    tmpdir = tempfile.mkdtemp()
    try:
        svn.ReportSvn.checkout(release_branch + '/market/report/runtime_cloud', tmpdir)
        major_version = get_version_from_branch_url(release_branch)
        package_path = os.path.join(tmpdir, 'yandex-market-report.json')

        with open(package_path) as package_file:
            package_json = json.load(package_file)
            package_version = package_json['meta']['version']
            if major_version in package_version:
                version_parts = package_version.split('.')
                minor_version = version_parts[3]
                next_minor_version = str(int(minor_version) + 1)
                return '{major}.{minor}'.format(major=major_version, minor=next_minor_version)
            else:
                return major_version + '.1'
    finally:
        shutil.rmtree(tmpdir)
