import collections
import json
import os
import tempfile

import logging

import shutil

from sandbox import sdk2
from lxml import etree

TRUNK = sdk2.svn.Arcadia.ARCADIA_TRUNK_URL

Commit = collections.namedtuple('Commit', ['revision', 'message', 'author'])


class ReportSvn(sdk2.svn.Arcadia):
    @classmethod
    def get_svn_log(cls, release_branch, revision_interval=None, verbose=True):
        opts = cls.SvnOptions()
        opts.xml = True
        opts.stop_on_copy = True
        opts.verbose = True

        if revision_interval is not None:
            revision_from, revision_to = revision_interval
            opts.revision = '{}:{}'.format(revision_from, revision_to or 'HEAD')

        svn_url = cls.svn_url(release_branch)
        svn_log_output = cls.svn('log', opts, url=svn_url, timeout=sdk2.svn.Svn.SVN_DEFAULT_TIMEOUT)

        return etree.fromstring(svn_log_output.stdout)


def get_copied_from_revision(branch_url):
    xml_out = ReportSvn.get_svn_log(branch_url)
    copied_from = xml_out.xpath('/log/logentry/paths/path[@copyfrom-rev]')
    if not copied_from:
        raise Exception(
            'Failed to determine the starting revision for branch "{0}"'
                .format(branch_url))
    # Assume the last copyfrom-rev points to the revision the branch was
    # created from.
    # This should work well enough for finding where the release
    # originated from.
    return int(copied_from[-1].attrib['copyfrom-rev'])


def generate_release_branch_path(branch_url):
    revision_from = get_copied_from_revision(branch_url) + 1
    revision_to = get_last_revision(branch_url)
    return ReportSvn.diff(branch_url, change="{}-{}".format(revision_from, revision_to))


def get_release_revision(relese_branch_url):
    return get_copied_from_revision(relese_branch_url)


def get_last_revision(branch_url):
    return ReportSvn.get_revision(branch_url)


def get_commits(branch_path, revision_interval=None, also_non_market_commits=False):
    def to_commit(elem):
        revision = elem.attrib['revision']
        message = elem.find('msg').text if elem.find('msg') is not None else ''
        author = elem.find('author').text
        return Commit(revision, message, author)

    def about_report(elem):
        report_paths = [
            '/market/report/src',
            '/market/report/data',
            '/market/report/clusters_bin',
            '/market/report/configs/generator',
            '/market/report/basesearch',
            '/market/report/runtime_cloud',
            '/market/library/libmarket',
            '/market/library/libreport',
            '/market/library/fixed_point_number',
            '/market/library/currency',
            '/market/library/currency_exchange',
            '/market/library/recom',
            '/market/library/ichwill',
            '/market/library/crypta',
            '/market/library/trees',
            '/market/library/xml',
            '/market/library/pictures',
            '/market/devtools/build.mk',
        ]

        for path in elem.xpath('paths/path'):
            for rpath in report_paths:
                if rpath in path.text:
                    return True
        return False

    xml_out = ReportSvn.get_svn_log(
        branch_path, revision_interval=revision_interval, verbose=True)
    market_logentries = [le
                         for le in xml_out.xpath('/log/logentry')
                         if about_report(le)]
    market_commits = map(to_commit, market_logentries)

    non_market_commits = []
    if also_non_market_commits:
        non_market_logentries = [le
                                 for le in xml_out.xpath('/log/logentry')
                                 if not about_report(le)]
        non_market_commits = map(to_commit, non_market_logentries)

    return market_commits, non_market_commits


def get_version_commits(branch_path):
    xml_out = ReportSvn.get_svn_log(branch_path, verbose=False)
    revisions = xml_out.xpath('/log/logentry[@revision]')
    if not revisions or len(revisions) < 2:
        return []

    rev_interval = (revisions[0].attrib['revision'],
                    revisions[-1].attrib['revision'])
    commits, _ = get_commits(branch_path, rev_interval)
    return commits


def update_changelog(release_branch, changelog_content, should_commit=True):
    tmpdir = tempfile.mkdtemp()
    try:
        debian_dir_path = 'market/report/debian'
        _checkout_path(release_branch, tmpdir, debian_dir_path)
        changelog_path = os.path.join(tmpdir, debian_dir_path, 'changelog')

        with open(changelog_path, 'r') as changelog:
            logging.info("Changelog was:")
            old_changelog_content = unicode(changelog.read(), 'utf-8')
            logging.info(old_changelog_content)

        with open(changelog_path, 'w+') as changelog:
            new_changelog_content = u'{}\n{}'.format(changelog_content, old_changelog_content)
            changelog.write(new_changelog_content.encode('utf-8'))

        with open(changelog_path, 'r') as changelog:
            logging.info("Changelog now:")
            new_changelog_content = changelog.read()
            logging.info(new_changelog_content)

            if should_commit:
                _commit_file("update market/report changelog [no merge]", changelog_path)
                return new_changelog_content
            else:
                return new_changelog_content, _create_diff(tmpdir)

    finally:
        shutil.rmtree(tmpdir)


def patch_ya_package_version(release_branch, package_version, should_commit=True, package_name='yandex-market-report'):
    tmpdir = tempfile.mkdtemp()
    try:
        runtime_cloud_dir_path = 'market/report/runtime_cloud'
        _checkout_path(release_branch, tmpdir, runtime_cloud_dir_path)
        package_path = os.path.join(tmpdir, runtime_cloud_dir_path, package_name + '.json')

        with open(package_path) as package_file:
            logging.info("Ya Package was:")
            file_content = package_file.read()
            logging.info(file_content)

            package_json = json.loads(file_content, object_pairs_hook=collections.OrderedDict)

        package_json['meta']['version'] = package_version
        content = json.dumps({
            'version': package_version,
            'package': package_name,
            'sandbox_task_id': '{sandbox_task_id}'
        })
        package_json['data'].append({
            'source': {
                'type': 'INLINE',
                'content': '{' + content + '}'  # escaping outer '{' and '}' of serialized JSON object
            },
            'destination': {
                'path': '/' + package_name + '.json'
            }
        })

        with open(package_path, 'w') as package_file:
            json_text = json.dumps(package_json, indent=4, separators=(',', ': '))
            package_file.write(json_text + '\n')

        with open(package_path, 'r') as package_file:
            logging.info("Ya Package now:")
            logging.info(package_file.read())

        if should_commit:
            _commit_file("set {0} version to {1}.json [no merge]".format(package_version, package_name),
                         package_path)
        else:
            return _create_diff(tmpdir)
    finally:
        shutil.rmtree(tmpdir)


def _commit_file(commit_message, path):
    try:
        ReportSvn.commit(path, commit_message, "lognick")
    except:
        logging.info("Failed to commit under lognick. Try commit under zomb-sandbox-rw")
        ReportSvn.commit(path, commit_message, "zomb-sandbox-rw")


def _create_diff(working_copy_root_dir):
    return ReportSvn.diff(working_copy_root_dir).replace(working_copy_root_dir + '/', '')


def _checkout_path(release_branch, tmpdir, path):
    runtime_cloud_dir_path_parts = path.split('/')
    ReportSvn.checkout(release_branch, tmpdir, depth='empty')

    for i in range(len(runtime_cloud_dir_path_parts)):
        ReportSvn.update(os.path.join(tmpdir, *runtime_cloud_dir_path_parts[0:(i + 1)]), depth='immediates')
