# coding: utf-8

import logging
import time

import dateutil

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import client as ctc
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt
from sandbox.projects.common import task_env
from sandbox.projects.market.report.common import helpers
from sandbox.projects.market.report.common.lite_test_runner import LiteTestYaMakeRunner
from sandbox.projects.market.report.MarketReportCdRelease import MarketReportCdRelease
from sandbox.sdk2.vcs.svn import Arcadia as arc


logger = logging.getLogger(__name__)

MAX_REVISION_CREATION_DELAY = 48 * 60 * 60


def get_revision_timestamp(revision):
    arc_url = arc.trunk_url(revision=revision)
    date = dateutil.parser.parse(arc.info(arc_url)['date']).replace(tzinfo=dateutil.tz.tzutc())
    date = date.astimezone(dateutil.tz.tzlocal())
    return int(time.mktime(date.timetuple()))


def find_proper_report_revision():
    task = helpers.find_last_task(MarketReportCdRelease.type, status=ctt.Status.Group.SUCCEED)
    resource = helpers.find_last_report_resourse(release_task=task)
    revision = resource.svn_revision
    ts = get_revision_timestamp(revision)
    logger.debug('Report resource found: %s (revision %s, timestamp: %s)', resource.id, revision, ts)
    if ts + MAX_REVISION_CREATION_DELAY < int(time.time()):
        logger.warn('Revision %s is too old, trunk will be used', revision)
        revision = 'trunk'
    elif hasattr(resource, 'hotfix') and resource.hotfix or resource.branch != 'trunk':
        logger.warn('Resource built from branch (hotfix), trunk will be used')
        revision = 'trunk'
    return revision


class MarketReportLiteTestsExp(sdk2.Task):
    class Requirements(task_env.BuildRequirements):
        client_tags = (
            (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC)
            & (ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL | ctc.Tag.LINUX_BIONIC)
        )

    class Parameters(sdk2.Task.Parameters):

        kill_timeout = 180 * 60

        with sdk2.parameters.Group('Input parameters'):
            run_tests = sdk2.parameters.Bool('Run tests', default=True)  # UPS-807
            rearr_factors = sdk2.parameters.String(
                'Rearrangement factors (semicolon separated pairs, for example "factor1=0;factor2=text")')
            attempts_number = sdk2.parameters.Integer('Number of attempts to run tests', default=3)

        with sdk2.parameters.Group('Building') as build_group:
            arc_token = sdk2.parameters.YavSecret(
                label='Yav ARC token', description='ARC_TOKEN from Sandbox Vault is used by default')
            yt_store_token = sdk2.parameters.YavSecret(
                label='Yav YT token', description='YT_STORE_TOKEN from Sandbox Vault is used by default')

        with sdk2.parameters.Group('Debug options'):
            debug_options = sdk2.parameters.Bool('Enable debug options', default=False)
            with debug_options.value[True]:
                revision = sdk2.parameters.String(
                    'Run tests from arcadia revision: trunk ("trunk") or svn revision ("r7944746")')
                test_filters = sdk2.parameters.String('Only run tests that match this string')
                disable_crash_only = sdk2.parameters.Bool('Disable crash only mode', default=False)

        notifications = [
            sdk2.Notification(
                statuses=[ctt.Status.FAILURE, ctt.Status.EXCEPTION, ctt.Status.TIMEOUT],
                recipients=['bmrph'],
                transport=ctn.Transport.TELEGRAM
            )
        ]

    def on_execute(self):
        if not self.Parameters.run_tests:
            self.set_info('WARN: Run test disabled, task skipped')
            return
        self._validate_input_params()

        self._arc_token = helpers.get_arc_token(self)
        self._yt_store_token = helpers.get_yt_store_token(self)

        revision = self._get_revision()
        self.set_info('INFO: Use report revision: {}'.format(revision))

        test_result = self._run_tests(revision)
        if not test_result:
            raise errors.TaskFailure('ERROR: All attempts to run tests were failed')

        self.set_info('INFO: Report LITE tests completed successfully')

    def _validate_input_params(self):
        if not self.Parameters.rearr_factors:
            raise errors.TaskFailure('Rearrangement factors is empty')

    def _get_revision(self):
        return self.Parameters.revision if self.Parameters.revision else find_proper_report_revision()

    def _run_tests(self, revision):
        with LiteTestYaMakeRunner(
            self,
            arc_token=self._arc_token,
            yt_store_token=self._yt_store_token,
            revision=revision,
            attempts_number=self.Parameters.attempts_number,
            failed_attempt_callback=self._on_failed_attempt,
        ) as runner:
            test_filters = self.Parameters.test_filters or None
            lite_test_args = {
                'crash_only': not self.Parameters.disable_crash_only,
                'with_rearr': self.Parameters.rearr_factors,
            }
            return runner.run_tests(
                lite_test_args=lite_test_args,
                test_filters=test_filters,
                test_type_filter=['py2test']
            )

    def _on_failed_attempt(self, failed_tests):
        break_next_attempt = False
        coredumps = failed_tests.get('fail', set())
        if not coredumps:
            self.set_info('INFO: Test with coredump not found, timeouts will be ignored')
            break_next_attempt = True
        return break_next_attempt
