# coding: utf-8

import logging
import os
import tempfile

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import client as ctc
from sandbox.projects.common import task_env
from sandbox.projects.market.report.common import helpers
from sandbox.projects.market.report.common.lite_test_runner import LiteTestYaMakeRunner
from sandbox.projects.market.report.MarketReportCdMetaRelease import MarketReportCdMetaRelease
from sandbox.projects.market.report.MarketReportCdRelease import MarketReportCdRelease
from sandbox.projects.market.resources import MARKET_REPORT, MARKET_META_REPORT


logger = logging.getLogger(__name__)


class MarketReportMetaBaseCompatibility(sdk2.Task):
    class Requirements(task_env.BuildRequirements):
        client_tags = (
            (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC)
            & ~(ctc.Tag.CORES16 | ctc.Tag.CORES24 | ctc.Tag.CORES32)
            & (ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL | ctc.Tag.LINUX_BIONIC)
        )

    class Parameters(sdk2.Task.Parameters):

        kill_timeout = 90 * 60

        with sdk2.parameters.Group('Input parameters'):
            attempts_number = sdk2.parameters.Integer('Number of attempts to run tests', default=3)
            svn_arcadia_url = sdk2.parameters.String('Svn arcadia url', required=True)
            test_base_search = sdk2.parameters.Bool('Use stable meta report for current base', default=False)

        with sdk2.parameters.Group('Tokens') as build_group:
            arc_token = sdk2.parameters.YavSecret(
                label='Yav ARC token', description='ARC_TOKEN from Sandbox Vault is used by default')
            yt_store_token = sdk2.parameters.YavSecret(
                label='Yav YT token', description='YT_STORE_TOKEN from Sandbox Vault is used by default')
            ya_token = sdk2.parameters.YavSecret(
                label='Yav YA token', description='YA_TOKEN from Sandbox Vault is used by default')

        with sdk2.parameters.Group('Debug options'):
            debug_options = sdk2.parameters.Bool('Enable debug options', default=False)
            with debug_options.value[True]:
                skip_download_report = sdk2.parameters.Bool(
                    'Skip download report binary, use report built from repository', default=False)
                report_release_resource = sdk2.parameters.Resource('Report release resource (will use stable if empty)')
                test_filters = sdk2.parameters.String('Only run tests that match this string')
                rearr_factors = sdk2.parameters.String(
                    'Rearrangement factors (semicolon separated pairs, for example "factor1=0;factor2=text")')
                disable_crash_only = sdk2.parameters.Bool('Disable crash only mode', default=False)

    def on_execute(self):
        self._arc_token = helpers.get_arc_token(self)
        self._yt_store_token = helpers.get_yt_store_token(self)
        self._ya_token = helpers.get_ya_token(self)
        report_bin_path = None
        if not self.Parameters.skip_download_report:
            report_bin_path = self._download_stable_meta_report() if self.Parameters.test_base_search else self._download_stable_base_report()
        self._run_tests(report_bin_path)
        self.set_info('INFO: Report LITE tests completed successfully')

    def _download_stable_report(self, task_type, resource_type):
        if self.Parameters.report_release_resource:
            logger.info('Use resource {} as report release', self.Parameters.report_release_resource)
            resource = self.Parameters.report_release_resource
        else:
            logger.info('Try to get report binary from last stable')
            task = helpers.find_last_task(task_type, release='stable')
            resource = helpers.find_last_report_resourse(release_task=task, resource_type=resource_type)

        bin_path = helpers.unpack_report_bin(tempfile.mkdtemp(), resource=resource)
        self.set_info('INFO: Using report release version: {}'.format(resource.resource_version))
        return bin_path

    def _download_stable_base_report(self):
        return self._download_stable_report(MarketReportCdRelease.type, MARKET_REPORT)

    def _download_stable_meta_report(self):
        return self._download_stable_report(MarketReportCdMetaRelease.type, MARKET_META_REPORT)

    def _run_tests(self, report_bin_path):
        with LiteTestYaMakeRunner(
            self,
            arc_token=self._arc_token,
            yt_store_token=self._yt_store_token,
            ya_token=self._ya_token,
            arcadia_url=self.Parameters.svn_arcadia_url,
            attempts_number=self.Parameters.attempts_number,
            failed_attempt_callback=self._on_failed_attempt,
        ) as runner:
            test_filters = self.Parameters.test_filters or None
            lite_test_args = {
                'crash_only': not self.Parameters.disable_crash_only,
                'with_rearr': self.Parameters.rearr_factors or None,
            }
            ya_make_flags = []

            if report_bin_path:
                if self.Parameters.test_base_search:
                    self.set_info('INFO: Run test to check of meta/base compatibility: test base / stable meta')
                    lite_test_args.update({
                        'meta_bin': report_bin_path
                    })
                    ya_make_flags.append('-D=SKIP_META')
                else:
                    self.set_info('INFO: Run test to check of meta/base compatibility: stable base / test meta')
                    lite_test_args.update({
                        'fresh_offers_bin': report_bin_path,
                        'base_bin': report_bin_path
                    })
                    ya_make_flags.append('-D=SKIP_BASE')
            else:
                self.set_info('INFO: Run test to check of meta/base compatibility: test base / test meta')

            test_result = runner.run_tests(
                lite_test_args=lite_test_args,
                test_filters=test_filters,
                ya_make_extra_parameters=ya_make_flags,
                distbuild_pool='//vla/users/market/report'
            )
            if not test_result:
                raise errors.TaskFailure('ERROR: All attempts to run tests were failed')

    def _on_failed_attempt(self, failed_tests):
        break_next_attempt = False
        coredumps = failed_tests.get('fail', set())
        if not coredumps:
            self.set_info('INFO: Test with coredump not found, timeouts will be ignored')
            break_next_attempt = True
        return break_next_attempt
