# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task
from sandbox.projects.market.report.common import kombat_client

import sandbox.common.types.notification as ctn


STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
STARTREK_USER_AGENT = 'Sandbox MARKET_REPORT_PERF_TEST_EXP'


class MarketReportPerfTestExp(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        run_tests = sdk2.parameters.Bool('Run tests', default=True)
        rearr_factors = sdk2.parameters.String(
            'Rearrangement factors (semicolon separated pairs, for example "factor1=0;factor2=text")', required=True)
        ticket = sdk2.parameters.String('Experiment ticket', required=True)
        tokens = sdk2.parameters.YavSecret('YAV secret identifier (with optional version)', default='sec-01feejn5ptvqgv6h47m178q8y2', required=True)
        startrek_token_name = sdk2.parameters.String('Startrek OAuth token name', default='startrek-token', required=True)

    def on_execute(self):
        super(MarketReportPerfTestExp, self).on_execute()
        if not self.Parameters.run_tests:
            self.set_info('WARN: Run test disabled, task skipped')
            return

        try:
            self._validate_input_params()
            self._run_kombat_battles()
        except Exception as e:
            self.set_info('ERROR: Failed to run kombat battles: {}'.format(e))
            self._notify_juggler(text='Creation kombat battles failed', status=ctn.JugglerStatus.CRIT)
            return

        self._notify_juggler(text='OK', status=ctn.JugglerStatus.OK)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "MARKET_REPORT_PERF_TEST_EXP", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def _validate_input_params(self):
        if not self.Parameters.rearr_factors:
            raise errors.TaskFailure('Rearrangement factors is empty')

    def _run_kombat_battles(self):
        self._create_startrek_client()
        kombat_ticket = self._create_ticket_for_shooting()
        rearr_factors = self.Parameters.rearr_factors.split(';')
        battles = kombat_client.test_exp_batch(kombat_ticket, self.Parameters.ticket, rearr_factors, owner='AB')
        self._write_start_message(battles)

    def _create_ticket_for_shooting(self):
        issue = self._startrek.issues.create(
            queue='MARKETKOMBAT',
            summary='Стрельбы эксперимента {}'.format(self.Parameters.ticket),
            links=[{'relationship': 'relates', 'issue': self.Parameters.ticket}]
        )
        return issue.key

    def _create_startrek_client(self):
        from startrek_client import Startrek

        startrek_token = self.Parameters.tokens.data()[self.Parameters.startrek_token_name]
        self._startrek = Startrek(useragent=STARTREK_USER_AGENT, base_url=STARTREK_URL, token=startrek_token)

    def _write_start_message(self, battles):
        battle_info_links = kombat_client.create_battle_info_links(battles)
        issue = self._startrek.issues[self.Parameters.ticket]
        issue.comments.create(text='Стрельбы по эксперименту запущены: {}'.format(', '.join(battle_info_links)))

    def _notify_juggler(self, text, status):
        try:
            self._try_notify_juggler(text, status)
        except Exception as e:
            self.set_info('ERROR: Failed to notify about fail: {}'.format(e))

    def _try_notify_juggler(self, text, status):
        self.server.notification(
            body=text,
            recipients=["host=sandbox_market_report_perf_test_exp&service=failure_to_create_battles"],
            transport=ctn.Transport.JUGGLER,
            check_status=status
        )
