# coding: utf-8

import collections
import json
import logging
import os

import requests
from urllib3.util.retry import Retry

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import client as ctc
from sandbox.common.types import notification as ctn
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.vcs import arc
from sandbox.projects.market.report.common import kombat_client


logger = logging.getLogger(__name__)

ARCANUM_URL = 'https://arcanum.yandex.net/api/v1'
STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
STARTREK_USER_AGENT = 'Sandbox MARKET_REPORT_PRECOMMIT_PERF_TEST'

STARTED_BATTLES_COMMENT_TEMPL = """
Стрельбы по изменениям в `market/report` запущены: {battle_links}
Результаты стрельб можно посмотреть в тикете {kombat_ticket}
Больше информации про прекоммитные стрельбы [здесь](https://wiki.yandex-team.ru/market/report/infra/optimizacija-timetomarket/).
"""
STARTED_BATTLES_COMMENT_SHORT_TEMPL = """
Стрельбы по изменениям в `market/report` запущены: {battle_links}
"""
FAILED_BATTLES_COMMENT_TEMPL = """
Стрельбы завершились с ошибкой: {battle_links} (подробности в [тикете](https://st.yandex-team.ru/{kombat_ticket})).
Что с этим делать написано [здесь](https://wiki.yandex-team.ru/market/report/infra/optimizacija-timetomarket/).
"""


def contains_common_prefix(prefix, paths):
    for path in paths:
        if prefix == os.path.commonprefix([prefix, path]):
            return True
    return False


def get_ci_context_value(context, *args):
    context_part = context or {}
    paths = collections.deque(args)
    while len(paths) != 1:
        path = paths.popleft()
        context_part = context_part.get(path, {})
    return context_part.get(paths[0])


def requests_retry_session(retries=5, backoff_factor=1.0):
    session = requests.Session()
    retry = Retry(total=retries, backoff_factor=backoff_factor, status_forcelist=(500, 502, 503, 504))
    adapter = requests.adapters.HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


class ArcanumClient(object):
    def __init__(self, token):
        self.__token = token
        self.__session = requests_retry_session()

    def comment_pull_request(self, pull_request_id, message, issue=True):
        query = 'pull-requests/{}/comments'.format(pull_request_id)
        data = {
            'content': message,
            'issue': issue,
        }
        self.__send_request(query, data=json.dumps(data))

    def __send_request(self, query, data=None):
        url = '{proxy}/{query}'.format(proxy=ARCANUM_URL, query=query)
        headers = {
            'Authorization': 'OAuth {}'.format(self.__token),
            'Content-Type': 'application/json',
        }
        logger.debug('Send request to Arcanum: %s', url)
        response = self.__session.post(url, headers=headers, data=data)
        if response.status_code != 200:
            raise RuntimeError('Arcanum request {url} failed({status_code}): {text}'.format(
                url=response.url, status_code=response.status_code, text=response.text
            ))
        return response


class MarketReportPrecommitPerfTest(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        client_tags = (
            (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC)
            & (ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL | ctc.Tag.LINUX_BIONIC)
        )

    class Parameters(sdk2.Parameters):
        kill_timeout = None

        with sdk2.parameters.Group('Input parameters'):
            controlled_paths = sdk2.parameters.List(
                'List of paths to run Kombat battles', value_type=sdk2.parameters.String, required=True)
            pull_request_id = sdk2.parameters.Integer('Pull request id', required=True)
            battle_priority = sdk2.parameters.Integer('Kombat battle priority', default=300)
            fail_fast = sdk2.parameters.Bool('Finish task after first fail', default=True)
            wait_time = sdk2.parameters.Integer('Wait time between check battle status (seconds)', default=300)

        with sdk2.parameters.Group('Auth'):
            arc_secret = sdk2.parameters.YavSecret('Yav ARC secret', required=True)
            arcanum_secret = sdk2.parameters.YavSecret('Yav Arcanum secret', required=True)
            startrek_secret = sdk2.parameters.YavSecret('Yav Startrek secret', required=True)

        with sdk2.parameters.Group('Other'):
            binary_release = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Debug options'):
            debug_options = sdk2.parameters.Bool('Enable debug options', default=False)
            with debug_options.value[True]:
                ci_context = sdk2.parameters.String('CI context')

    def on_execute(self):
        with self.memoize_stage.prepare:
            self._validate_input_params()

        self._ci_context = self.Context.__CI_CONTEXT or json.loads(self.Parameters.ci_context)

        try:
            with self.memoize_stage.check_controlled_paths:
                if not self._pr_contains_controlled_path():
                    self.set_info("INFO: Pull request doesn't contain any files from controlled paths")
                    return
            if not self.Context.battle_ids:
                self._run_battles()
            else:
                self._check_battles()
        except sdk2.WaitTime:
            raise
        except errors.TaskFailure:
            self._cancel_battles()
            raise
        except Exception as e:
            logger.exception(e)
            self.set_info('WARN: Exception has occurred: {}. Wait {}s'.format(e, self.Parameters.wait_time))
            raise sdk2.WaitTime(self.Parameters.wait_time)

    def on_break(self, *unused):
        self._cancel_battles()

    def on_terminate(self):
        self._cancel_battles()

    def _validate_input_params(self):
        assert self.Parameters.controlled_paths, 'controlled_paths is empty'
        assert self.Parameters.pull_request_id, 'pull_request_id must be > 0'
        assert self.Parameters.battle_priority, 'battle_priority must be > 0'
        assert self.Parameters.wait_time > 0, 'wait_time must be > 0'

    def _pr_contains_controlled_path(self):
        changed_files = self._get_changed_files()
        for controlled_path in self.Parameters.controlled_paths:
            if contains_common_prefix(controlled_path, changed_files):
                logger.info('Pull request contains controlled path, %s', controlled_path)
                return True
        logger.info("Pull request doesn't contain controlled path")
        return False

    def _get_changed_files(self):
        vcs_info = get_ci_context_value(self._ci_context, 'launch_pull_request_info', 'vcs_info')
        logger.debug('vcs_info: %s', vcs_info)
        feature_branch = vcs_info.get('feature_branch')
        if not feature_branch:
            path = [self.Parameters.controlled_paths[0]]
            logger.warn('Feature branch not found, set changed files as %s', path)
            return path

        arc_token = self.Parameters.arc_secret.data()[self.Parameters.arc_secret.default_key]
        arc_client = arc.Arc(arc_oauth_token=arc_token)
        with arcadiasdk.mount_arc_path(
            'arcadia-arc:/#{}'.format(feature_branch),
            use_arc_instead_of_aapi=True,
            arc_oauth_token=arc_token
        ) as arcadia_path:
            commits = arc_client.log(
                arcadia_path,
                start_commit=vcs_info['upstream_revision_hash'],
                end_commit=vcs_info['feature_revision_hash'],
                as_dict=True,
                name_only=True
            )

            changed_files = set()
            for commit in commits:
                paths = [
                    name['path']
                    for name in commit.get('names')
                    if 'path' in name
                ]
                changed_files.update(paths)
            logger.debug('Changed files of branch %s: %s', feature_branch, ', '.join(changed_files))
            return changed_files

    def _run_battles(self):
        self.Context.is_first_run = True
        self.Context.failed_battles = []
        base_revision = get_ci_context_value(self._ci_context, 'previous_revision', 'number')
        logger.info('Base revision for battle: %s', base_revision)
        self.Context.kombat_ticket = self._get_kombat_ticket()
        self.Context.battle_ids = kombat_client.test_precommit_batch(
            self.Context.kombat_ticket,
            base_revision,
            self.Parameters.pull_request_id,
            self.Parameters.battle_priority,
            owner='CI')
        links = kombat_client.create_battle_info_links(self.Context.battle_ids)
        self.set_info('INFO: Kombat battles launched successful: {}'.format(', '.join(links)), do_escape=False)
        self._comment_started_battles()
        raise sdk2.WaitTime(self.Parameters.wait_time)

    def _get_kombat_ticket(self):
        input_parameters = {'pull_request_id': self.Parameters.pull_request_id}
        task = sdk2.Task.find(
            task_type=self.type, owner=self.owner, author=self.author, children=False,
            input_parameters=input_parameters, extra_fields=['context']
        ).order(sdk2.Task.id).first()
        if task is not None and task.Context.kombat_ticket is not ctm.NotExists:
            self.Context.is_first_run = False
            logger.info('Reuse Kombat ticket from task %s: %s', task.id, task.Context.kombat_ticket)
            return task.Context.kombat_ticket
        logger.info('Previous task launch not found, create new Kombat ticket')
        return self._create_kombat_ticket()

    def _create_kombat_ticket(self):
        from startrek_client import Startrek
        pr_issues = get_ci_context_value(self._ci_context, 'launch_pull_request_info', 'issues')
        logger.debug('Pull request issues: %s', pr_issues)
        links = [
            {'relationship': 'relates', 'issue': issue.get('id')}
            for issue in pr_issues
            if issue.get('id') is not None
        ]
        token = self.Parameters.startrek_secret.data()[self.Parameters.startrek_secret.default_key]
        startrek = Startrek(useragent=STARTREK_USER_AGENT, base_url=STARTREK_URL, token=token)
        issue = startrek.issues.create(
            queue='MARKETKOMBAT',
            summary='Прекоммитные стрельбы по PR:{}'.format(self.Parameters.pull_request_id),
            links=links
        )
        logger.info('New Kombat ticket: %s', issue.key)
        return issue.key

    def _check_battles(self):
        actual_battles = []
        for battle_id in self.Context.battle_ids:
            logger.debug('Check battle: %s', battle_id)
            battle_result = kombat_client.get_battle_result(battle_id)
            if battle_result.is_ready:
                link = kombat_client.create_battle_info_link(battle_result.id)
                if battle_result.is_success:
                    self.set_info('INFO: Battle {} successfully finished'.format(link), do_escape=False)
                elif battle_result.is_cancelled:
                    self.Context.failed_battles.append(battle_result.id)
                    self.set_info('ERROR: Battle {} was cancelled'.format(link), do_escape=False)
                else:
                    self.Context.failed_battles.append(battle_result.id)
                    self.set_info('ERROR: Battle {} has failed'.format(link), do_escape=False)
            else:
                logger.info('Battle %s progess: %s%%', battle_result.id, battle_result.progress)
                actual_battles.append(battle_result.id)

        self.Context.battle_ids = actual_battles
        if self.Context.failed_battles and (self.Parameters.fail_fast or not self.Context.battle_ids):
            self._comment_failed_battles(self.Context.failed_battles)
            raise errors.TaskFailure('Some battle was failed')

        if self.Context.battle_ids:
            self.set_info('INFO: Still waiting battles: {}'.format(', '.join(self.Context.battle_ids)))
            raise sdk2.WaitTime(self.Parameters.wait_time)
        else:
            self.set_info('INFO: All battles successfully finished')

    def _cancel_battles(self):
        for battle_id in self.Context.battle_ids:
            try:
                logger.info('Cancel battle: %s', battle_id)
                kombat_client.cancel_battle(battle_id)
            except Exception as e:
                logger.exception(e)
        self.Context.battle_ids = []

    def _comment_started_battles(self):
        comment_args = {
            'battle_links': ', '.join(kombat_client.create_battle_info_links(self.Context.battle_ids)),
            'issue': False,
        }
        if self.Context.is_first_run:
            comment_args['kombat_ticket'] = self.Context.kombat_ticket
            self._comment_pull_request(STARTED_BATTLES_COMMENT_TEMPL, **comment_args)
        else:
            self._comment_pull_request(STARTED_BATTLES_COMMENT_SHORT_TEMPL, **comment_args)

    def _comment_failed_battles(self, battle_ids):
        self._comment_pull_request(
            FAILED_BATTLES_COMMENT_TEMPL,
            battle_links=', '.join(kombat_client.create_battle_info_links(battle_ids)),
            kombat_ticket=self.Context.kombat_ticket,
        )

    def _comment_pull_request(self, template, issue=True, **kwargs):
        try:
            message = template.format(**kwargs)
            token = self.Parameters.arcanum_secret.data()[self.Parameters.arcanum_secret.default_key]
            arcanum = ArcanumClient(token)
            arcanum.comment_pull_request(self.Parameters.pull_request_id, message, issue=issue)
        except Exception as e:
            logger.exception(e)
