import logging
import multiprocessing

from sandbox import sdk2
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.channel import channel
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import constants
from sandbox.projects.common.build import YaPackage
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.market.sre import MarketYaPackageLegacy
from sandbox.projects.market.report.MarketDeployWithReductor import Environment, MarketDeployWithReductor


class RestartReportParameter(SandboxBoolParameter):
    name = 'restart_report'
    description = 'Restart report after releases'
    default_value = False


class CheckoutParameter(build_parameters.CheckoutParameter):
    default_value = False


class BuildType(YaPackage.parameters.BuildType):
    required = False


class ReportHotfixParameter(SandboxBoolParameter):
    name = 'report_hotfix'
    description = 'Mark release as hotfix'
    default_value = False


class MarketReportRelease(MarketYaPackageLegacy.MarketYaPackageLegacy, object):
    """Task to package and release market report related packages
    """
    type = 'MARKET_REPORT_RELEASE'

    input_parameters = [
        RestartReportParameter,
        ReportHotfixParameter,
        YaPackage.parameters.ArcadiaUrl,
        BuildType,
        MarketYaPackageLegacy.PackagesParameter,
        YaPackage.RunTestsParameter,
        YaPackage.RunLongTestsParameter,
        build_parameters.TestsRetriesCount,
        MarketYaPackageLegacy.ResourceTypeParameter,
        MarketYaPackageLegacy.StartrekTicketIdsParameter,
        build_parameters.ArcadiaPatch,
        CheckoutParameter,
        build_parameters.ThinLTO,
        build_parameters.BuildSystem,
        build_parameters.ClearBuild,
        build_parameters.DistbuildPool,
        build_parameters.EnvironmentVarsParam,
    ]

    def on_enqueue(self):
        MarketYaPackageLegacy.MarketYaPackageLegacy.on_enqueue(self)
        if not self.ctx.get(BuildType.name):
            self.ctx[BuildType.name] = constants.RELEASE_BUILD_TYPE
        if ';' in self.ctx[MarketYaPackageLegacy.PackagesParameter.name]:
            raise TaskFailure('; in package json path')

    def on_prepare(self):
        checkout_parameter = self.ctx.get(CheckoutParameter.name)
        if self.ctx.get(build_parameters.BuildSystem.name) not in (constants.SEMI_DISTBUILD_BUILD_SYSTEM, constants.DISTBUILD_BUILD_SYSTEM):
            self.ctx[build_parameters.BuildSystem.name] = constants.SEMI_DISTBUILD_BUILD_SYSTEM
        super(MarketReportRelease, self).on_prepare()
        self.ctx[build_parameters.YtStoreThreads.name] = multiprocessing.cpu_count()

        self.ctx[build_parameters.YtStore.name] = True
        self.ctx[build_parameters.YtProxy.name] = 'hahn.yt.yandex.net'
        self.ctx[build_parameters.YtDir.name] = '//home/market-report-dist-cache'
        self.ctx[build_parameters.YtTokenVaultOwner.name] = self.owner
        self.ctx[build_parameters.YtTokenVaultName.name] = 'YT_STORE_TOKEN'

        self.ctx[YaPackage.parameters.UseArcadiaApiFuse.name] = True
        self.ctx[YaPackage.parameters.AllowArcadiaApiFallback.name] = True
        self.ctx[YaPackage.parameters.UseArcInsteadOfArcadiaApi.name] = True
        self.ctx[MarketYaPackageLegacy.TsumSendReportParameter.name] = True

        self.ctx[CheckoutParameter.name] = checkout_parameter
        if self.ctx.get(build_parameters.ArcadiaPatch.name):
            logging.info("Build with patch %s. Need to disable --checkout flag", self.ctx.get(build_parameters.ArcadiaPatch.name))
            self.ctx[build_parameters.CheckoutParameter.name] = False

    def on_finish(self):
        super(MarketReportRelease, self).on_finish()
        self._set_inf_ttl_to_junit_reports()
        if self.ctx.get(ReportHotfixParameter.name) is True:
            self._set_released_hotfix_to_market_reports()

    def _set_inf_ttl_to_junit_reports(self):
        junit_reports = channel.sandbox.list_resources(task_id=self.id, resource_type='JUNIT_REPORT')

        if junit_reports:
            for junit_report in junit_reports:
                channel.sandbox.set_resource_attribute(junit_report.id, 'ttl', 'inf')

    def _set_released_hotfix_to_market_reports(self):
        resource_type = self.ctx.get(MarketYaPackageLegacy.ResourceTypeParameter.name)
        market_reports = channel.sandbox.list_resources(task_id=self.id, resource_type=resource_type)

        if market_reports:
            for market_report in market_reports:
                channel.sandbox.set_resource_attribute(market_report.id, 'hotfix', 'true')
