import logging
import multiprocessing

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task, constants, task_env
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.nanny import nanny
from sandbox.projects.market.sre import MarketYaPackage2


REPORT_GROUP_NAME = "Report params"


class RestartReportParameter(sdk2.parameters.Bool):
    name = 'restart_report'
    description = 'Restart report after releases'
    default_value = False
    group = REPORT_GROUP_NAME


class ReportHotfixParameter(sdk2.parameters.Bool):
    name = 'report_hotfix'
    description = 'Mark release as hotfix'
    default_value = False
    group = REPORT_GROUP_NAME


class MarketReportRelease2(MarketYaPackage2.MarketYaPackage2, object):
    """Task to package and release market report related packages
    """

    class Requirements(MarketYaPackage2.MarketYaPackage2.Requirements):
        pass

    class Parameters(MarketYaPackage2.MarketYaPackage2.Parameters):
        restart_report = RestartReportParameter()
        report_hotfix = ReportHotfixParameter()
        build_type = build_parameters.BuildType(required = False)
        ext_params = binary_task.binary_release_parameters(none=True)


    def on_save(self):
        if not self.Parameters.build_type:
            self.Parameters.build_type = constants.RELEASE_BUILD_TYPE
        if ';' in self.Parameters.packages:
            raise TaskFailure('; in package json path')
        if self.Parameters.build_system not in (constants.SEMI_DISTBUILD_BUILD_SYSTEM, constants.DISTBUILD_BUILD_SYSTEM):
            self.Parameters.build_system = constants.SEMI_DISTBUILD_BUILD_SYSTEM

        self.Parameters.ya_yt_store = True
        self.Parameters.ya_yt_proxy = 'hahn.yt.yandex.net'
        self.Parameters.ya_yt_dir = '//home/market-report-dist-cache'
        self.Parameters.ya_yt_token_vault_owner = self.owner
        self.Parameters.ya_yt_token_vault_name = 'YT_STORE_TOKEN'

        self.Parameters.use_aapi_fuse = True
        self.Parameters.aapi_fallback = True
        self.Parameters.use_arc_instead_of_aapi = True
        self.Parameters.tsum_send_report = True

        if self.Parameters.arcadia_patch and self.Parameters.checkout:
            logging.info("Build with patch. Need to disable --checkout flag")
            self.Parameters.checkout = False

        super(MarketReportRelease2, self).on_save()

    def on_finish(self, prev_status, status):
        super(MarketReportRelease2, self).on_finish(prev_status, status)
        self._set_inf_ttl_to_junit_reports()
        if self.Parameters.report_hotfix:
            self._set_released_hotfix_to_market_reports()

    def _set_inf_ttl_to_junit_reports(self):
        junit_reports = sdk2.Resource.find(task_id=self.id, type='JUNIT_REPORT').limit(1)

        if junit_reports:
            for junit_report in junit_reports:
                setattr(junit_report.id, 'ttl', 'inf')

    def _set_released_hotfix_to_market_reports(self):
        market_reports = sdk2.Resource.find(task_id=self.id, type=self.Parameters.resource_type).limit(1)

        if market_reports:
            for market_report in market_reports:
                setattr(market_report.id, 'hotfix', 'true')
