import json
import logging
import multiprocessing as mp
import os
import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types import resource as ctr
from sandbox.projects.market.resources import MARKET_REPORT


logger = logging.getLogger(__name__)

PIGZ_BIN = '/usr/bin/pigz'


class MarketReportReleaseFromBuildServer(sdk2.Task):
    '''Task to find appropriate package in sandbox for given revision.'''

    class Parameters(sdk2.Task.Parameters):
        version = sdk2.parameters.String('Report version')
        svn_revision = sdk2.parameters.Integer('Revision id for release')

    class Requirements(sdk2.Requirements):
        ram = 32 << 10  # 32Gb
        client_tags = ctc.Tag.Group.LINUX
        container_resource = 2717091920  # Using custom LXC container with pigz

    # type = 'MARKET_REPORT_RELEASE_FROM_BUILD_SERVER'

    def on_execute(self):
        svn_revision = self.Parameters.svn_revision
        version = self.Parameters.version

        resource = self._download_last_report('OTHER_RESOURCE', svn_revision)
        self.set_info('INFO: Found resource of market-report package, id={}, decompressing'.format(resource.id))

        decompressed_fname = self._decompress_report(resource=resource)
        self.set_info('INFO: Package decompressed to {}. Adding version file and repacking'.format(decompressed_fname))
        repacked_report = self._patch_and_repack(decompressed_fname, version)

        self.set_info('INFO: Done. Uploading MARKET_REPORT resource')
        self._upload_updated_resource(repacked_report, svn_revision, version)

    def _download_last_report(self, resource_type, svn_revision):
        attrs = {'revision': svn_revision, 'package': 'yandex-market-report'}
        resource = (
            sdk2.Resource.find(resource_type=resource_type, state=ctr.State.READY, owner='mkoterev', attrs=attrs)
            .order(-sdk2.Resource.id)
            .first()
        )
        if resource is None:
            raise RuntimeError('Report resourse {} not found'.format(resource_type))
        return resource

    def _decompress_report(self, resource):
        tar_path = str(sdk2.ResourceData(resource).path)
        logger.info('Resource {} downloaded to {}'.format(resource.id, tar_path))

        if tar_path.endswith(('.tar.gz', '.tgz', '.gz')):
            self.compression_recived = 'gz'
        else:
            self.compression_recived = 'uc'

        decompressed_filename = 'yandex-market-report.tar'
        if self.compression_recived == 'gz':
            logger.info('Started gz decompression')
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('gzip_decomp')) as pl:
                with open(decompressed_filename, 'wb') as decompressed_tar:
                    sp.check_call(
                        [PIGZ_BIN, '-dcv', '--processes', str(mp.cpu_count()), str(tar_path)],
                        stdout=decompressed_tar,
                        stderr=pl.stderr,
                        cwd=str(self.path())
                    )

            return decompressed_filename

        logger.info('Started uc decompression')
        # Cause task is binary. See https://docs.yandex-team.ru/sandbox/dev/binary-task#dependencies
        from library.python.compress import decompress

        decompress(tar_path, decompressed_filename, threads=mp.cpu_count())

        return decompressed_filename

    def _patch_and_repack(self, decompressed_fname, report_version, codec_to_compress=None):
        os.mkdir('bin')
        path_to_version_file = os.path.join('bin', 'version')
        with open(path_to_version_file, 'w') as version_file:
            version_file.write(report_version)

        logger.info('Created file {} with version={}'.format(path_to_version_file, report_version))

        path_to_sox_file = 'yandex-market-report.json'
        with open(path_to_sox_file, 'w') as sf:
            json.dump(
                {'version': str(report_version), 'sandbox_task_id': str(self.id), 'package': 'yandex-market-report'},
                fp=sf,
            )

        logger.info('Adding files to tar {}'.format(decompressed_fname))
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('tar_append')) as pl:
            sp.check_call(
                ['tar', '--append', '-f', str(decompressed_fname), str(path_to_version_file), str(path_to_sox_file)],
                stderr=pl.stderr,
                cwd=str(self.path())
            )

        if codec_to_compress is None:
            codec_to_compress = 'lzma-2' if self.compression_recived == 'uc' else 'gz'

        compressed_tar_path = 'yandex-market-report.{}.tar.{}'.format(report_version, codec_to_compress)

        if self.compression_recived == 'gz' and codec_to_compress == 'gz':
            logger.info('Started gz compression')
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('gzip_comp')) as pl:
                with open(compressed_tar_path, 'wb') as compressed_tar:
                    sp.check_call(
                        [PIGZ_BIN, '-cv', '--processes', str(mp.cpu_count()), str(decompressed_fname)],
                        stdout=compressed_tar,
                        stderr=pl.stderr,
                        cwd=str(self.path())
                    )
        else:
            logger.info('Started uc compression')
            # Cause task is binary. See https://docs.yandex-team.ru/sandbox/dev/binary-task#dependencies
            from library.python.compress import compress

            compress(decompressed_fname, compressed_tar_path, codec=codec_to_compress, threads=mp.cpu_count())

        return compressed_tar_path

    def _upload_updated_resource(self, repacked_report, svn_revision, version):
        logger.info('Uploading new report to sandbox')

        MARKET_REPORT(
            self,
            'Updated (with version in file) MARKET_REPORT from build server',
            os.path.relpath(repacked_report),
            svn_revision=svn_revision,
            resource_version=version,
        )

        logger.info('MARKET_REPORT resource was uploaded to sandbox')
