# coding=utf-8

import json
import logging
import requests
from time import sleep
from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.errors import TaskError


class PublishMarketAccessResource(sdk2.Task):
    """Publish Market Access resource from Sandbox resource
    """

    _KEY_OWNER = 'MARKET'

    class Parameters(sdk2.Task.Parameters):

        sandbox_resource = sdk2.parameters.Resource(
            'Sandbox resource ID',
            required=True
        )
        resource_name = sdk2.parameters.String(
            'Resource name',
            required=True
        )
        endpoint = sdk2.parameters.String(
            'Endpoint to publish',
            description='Values: prod, test',
            required=True,
            default_value='prod'
        )

    def __create_version(self, host, resource_name, rbtorrent, tvm_ticket):
        path = '{host}/v1/resources/{resource_name}/versions'.format(host=host, resource_name=resource_name)
        body = {
            "change": [0],
            "resource_name": resource_name,
            "storage": [{
                "location": {
                    "access": {
                        "rbtorrent": rbtorrent
                    }
                }
            }]
        }
        body = json.dumps(body)

        logging.info('Using {path}'.format(path=path))
        logging.info(body)

        for _ in range(10):
            try:
                r = requests.post(
                    path,
                    headers={
                        'Content-Type': 'application/json',
                        'X-Ya-Service-Ticket': tvm_ticket,
                    },
                    data=body,
                    timeout=10
                )
                logging.info("Response: {}".format(r.text))
                response = r.json()
                if r.status_code == requests.codes.ok and ('status' not in response or response['status'] != 'error'):
                    return True
            except Exception as exception:
                logging.error(exception)
            sleep(1)
        return False

    def __get_tvm_ticket(self, self_tvm_id, self_tvm_secret, dst_tvm_id):
        from tvmauth import TvmClient, TvmApiClientSettings
        tvm_client = TvmClient(
            TvmApiClientSettings(
                self_tvm_id=self_tvm_id,
                self_secret=self_tvm_secret,
                dsts=[dst_tvm_id],
            )
        )
        return tvm_client.get_service_ticket_for(tvm_id=dst_tvm_id)

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={"Name": "PublishMarketAccessResource", "release": ctt.ReleaseStatus.STABLE},
        ).first()

    def on_execute(self):
        logging.info("on_execute")

        resource = sdk2.Resource[self.Parameters.sandbox_resource]
        skynet_id = resource.skynet_id
        logging.info("Resource skynet_id: {}".format(skynet_id))

        secret_id = "sec-01fmj4fs67sje50kzzwhgnakhj"
        secret = sdk2.yav.Secret(secret_id).data()

        self_tvm_id = int(secret["self_tvm_id"])
        self_tvm_secret = secret["self_tvm_secret"]
        if self.Parameters.endpoint.lower() == 'prod':
            host = 'https://access.vs.market.yandex.net'
            dst_tvm_id = int(secret["dst_tvm_id_prod"])
        else:
            host = 'https://access.tst.vs.market.yandex.net'
            dst_tvm_id = int(secret["dst_tvm_id_test"])
        tvm_ticket = self.__get_tvm_ticket(self_tvm_id, self_tvm_secret, dst_tvm_id)

        if self.__create_version(host, self.Parameters.resource_name, skynet_id, tvm_ticket):
            logging.info("Publish {resource_name} in market access is successful".format(resource_name=self.Parameters.resource_name))
        else:
            raise TaskError("Error publishing {resource_name} in market access".format(resource_name=self.Parameters.resource_name))
