import logging
import os
import tarfile

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import resource as ctr

from sandbox.projects.market.resources import MARKET_REPORT


logger = logging.getLogger(__name__)


def find_last_task(task_type, status=None, release=None, author='robot-market-infra'):
    task = sdk2.Task.find(
        task_type=sdk2.Task[task_type], status=status, release=release, author=author, children=False
    ).order(-sdk2.Task.id).first()
    child_task = sdk2.Task.find(
        task_type=sdk2.Task[task_type], status=status, release=release, author=author, children=True
    ).order(-sdk2.Task.id).first()
    if task is None and child_task is None:
        raise RuntimeError('Task {} not found'.format(task_type))
    if task is None or child_task is None:
        return task if task is not None else child_task
    return task if task.id > child_task.id else child_task


def find_last_report_resourse(release_task=None, svn_revision=None, state=ctr.State.READY, resource_type=MARKET_REPORT):
    attrs = None
    if svn_revision is not None:
        attrs = {'svn_revision': svn_revision}
    resource = sdk2.Resource.find(
        resource_type=resource_type, task=release_task, state=state, attrs=attrs
    ).order(-sdk2.Resource.id).first()
    if resource is None:
        raise RuntimeError('Report resourse {} not found'.format(resource_type))
    return resource


def unpack_report_bin(output_path, resource=None, tar_path=None):
    if not tar_path:
        if not resource:
            raise RuntimeError('Neither resource nor tar_path are specified')
        tar_path = str(sdk2.ResourceData(resource).path)
        logger.info('Resource {} downloaded to {}'.format(resource.id, tar_path))
    member = 'bin/report'
    with tarfile.open(tar_path, 'r:gz') as tar:
        tar.extract(member, output_path)
    report_bin_path = os.path.join(output_path, member)
    logger.info('Report binary unpacked to {}'.format(report_bin_path))
    return report_bin_path


def get_token(yav_token, vault_owner, vault_item=None):
    if yav_token:
        return yav_token.data()[yav_token.default_key]
    return sdk2.Vault.data(vault_owner, vault_item) if vault_item else None


def get_arc_token(task):
    try:
        return get_token(task.Parameters.arc_token, task.owner, vault_item='ARC_TOKEN')
    except Exception as e:
        logger.exception(e)
        raise errors.TaskFailure('ERROR: Failed to get ARC token: {}'.format(e))


def get_yt_store_token(task):
    token = None
    try:
        token = get_token(task.Parameters.yt_store_token, task.owner, vault_item='YT_STORE_TOKEN')
    except Exception as e:
        msg = 'Unable to get YT token, YT store disabled'
        logger.warning('{}: {}'.format(msg, e))
        task.set_info('WARN: {}'.format(msg))
    return token


def get_ya_token(task):
    try:
        token = get_token(task.Parameters.ya_token, task.owner, vault_item='YA_TOKEN')
    except Exception as e:
        msg = 'Unable to get YA token'
        logger.warning('{}: {}'.format(msg, e))
        task.set_info('WARN: {}'.format(msg))
    return token
