# coding: utf-8

import logging

import requests
from urllib3.util.retry import Retry


logger = logging.getLogger(__name__)

KOMBAT_PROXY = 'http://kombat.vs.market.yandex.net'
BATTLE_INFO_LINK_TEMPL = '<a href="{kombat_proxy}/battle_info?id={battle_id}">{battle_id}</a>'


class BattleResult(object):
    def __init__(self, id, state):
        self.__id = id
        self.__state = state

    @property
    def id(self):
        return self.__id

    @property
    def state(self):
        return self.__state

    @property
    def status(self):
        return self.__state.get('status')

    @property
    def progress(self):
        return self.__state.get('progress', 0)

    @property
    def is_ready(self):
        return self.status == 'complete' or self.is_cancelled

    @property
    def is_cancelled(self):
        return self.status == 'cancel'

    @property
    def is_success(self):
        return self.is_ready and self.__state.get('success', False)

    @property
    def next_battle_id(self):
        return self.__state.get('next_battle_id')


def test_exp_batch(kombat_ticket_id, exp_ticket_id, rearr_factors, owner='sandbox'):
    params = [
        ('ticket', kombat_ticket_id),
        ('experiment-ticket', exp_ticket_id),
        ('rearr-factor', rearr_factors),
        ('owner', owner)
    ]
    formated_params = format_params(params)
    query = 'test_exp_batch?{}'.format(formated_params)
    response = send_request(query)
    return response.json()


def test_precommit_batch(kombat_ticket_id, base_revision, pull_request_id, priority, owner='sandbox'):
    params = [
        ('ticket', kombat_ticket_id),
        ('br', base_revision),
        ('pr', pull_request_id),
        ('priority', priority),
        ('owner', owner),
    ]
    formated_params = format_params(params)
    query = 'test_precommit_batch?{}'.format(formated_params)
    response = send_request(query)
    return response.json()


def get_battle_result(battle_id, recursive=True):
    query = 'test_report_result?id={}'.format(battle_id)
    response = send_request(query)
    result = BattleResult(battle_id, response.json())
    if not recursive:
        return result
    while result.next_battle_id is not None:
        logger.info('Next battle id for {} is {}', battle_id, result.next_battle_id)
        result = get_battle_result(result.next_battle_id)
    return result


def cancel_battle(battle_id):
    battle_result = get_battle_result(battle_id)
    if not battle_result.is_ready:
        query = 'cancel_battle?id={}'.format(battle_result.id)
        send_request(query)
    else:
        logger.debug('Battle %s is already finished', battle_result.id)


def send_request(query):
    session = requests_retry_session()
    url = '{proxy}/{query}'.format(proxy=KOMBAT_PROXY, query=query)
    logger.debug('Send request to Kombat: %s', url)
    response = session.get(url)
    if response.status_code != 200:
        raise RuntimeError('Kombat request {url} failed({status_code}): {text}'.format(
            url=response.url, status_code=response.status_code, text=response.text
        ))
    return response


def requests_retry_session(retries=5, backoff_factor=1.0):
    session = requests.Session()
    retry = Retry(total=retries, backoff_factor=backoff_factor, status_forcelist=(500, 502, 503, 504))
    adapter = requests.adapters.HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


def format_params(params):
    params_str = ('{}={}'.format(name, value) for name, value in params)
    return '&'.join(params_str)


def create_battle_info_link(battle_id):
    return BATTLE_INFO_LINK_TEMPL.format(kombat_proxy=KOMBAT_PROXY, battle_id=battle_id)


def create_battle_info_links(battle_ids):
    return [create_battle_info_link(id) for id in battle_ids]
