import requests
import logging

from requests.exceptions import RequestException

PINGER_REPORT_JSON_URL = 'http://marketindexer.http.yandex.net:33131/yandex/report-pinger/report.json'
PINGER_TESTING_JSON_URL = 'http://mi01ht.market.yandex.net/yandex/report-pinger/report.json'

logger = logging.getLogger(__name__)


class PingerData(object):

    def __init__(self, environment):
        self.environment = self.check_environment(environment)
        self.pinger_json_url = PINGER_TESTING_JSON_URL if (self.environment == 'testing') else PINGER_REPORT_JSON_URL
        self.data = None

    def check_environment(self, environment):
        if environment == 'stable':
            return 'production'
        return environment

    def update(self):
        try:
            self.data = Clusters(requests.get(self.pinger_json_url, timeout=3).json(), self.environment)
        except Exception as e:
            self.data = None
            logging.exception("Failed to download: %s", e)

    def get_consistent_package_percent(self, package, version):
        if not self.data:
            return 0.0
        total = 0
        target = 0
        for cluster in self.data.clusters:
            if cluster.ctype != self.environment:
                continue
            if cluster.get_consistent_version(package) == version:
                target += 1
            total += 1
        if total == 0:
            return 0.0
        return target / total

    def get_installed_package_percent(self, package, version):
        if not self.data:
            return 0.0
        total = 0
        target = 0
        for cluster in self.data.clusters:
            if cluster.ctype != self.environment:
                continue
            for host in cluster.hosts:
                if host.get_package_version(package) == version:
                    target += 1
                total += 1
        if total == 0:
            return 0.0
        return float(target) / float(total)


class Clusters(object):

    def __init__(self, response, environment):
        self.clusters = list()
        self.clusters_by_id = dict()
        if not response:
            return
        for cluster_data in response['clusters']:
            cluster = Cluster(cluster_data)
            if cluster.dc in ('hd', 'hp'):
                continue
            self.clusters_by_id[cluster.id] = cluster
            self.clusters.append(cluster)

        for host in response['hosts']:
            cluster_id = host.get('cluster_id', '')
            cluster = self.clusters_by_id.get(cluster_id)
            if cluster:
                cluster.hosts.append(Host(host))

        for cluster in self.clusters:
            cluster.detect_ctype()


class Cluster(object):
    def __init__(self, data):
        self.id = data['cluster_id']
        self.name = data['name']
        self.status = data['status']
        self.group = data['group']
        self.report_version = data.get('consistent_version','')
        self.formulas_version = data.get('consistent_formulas_version', '')
        self.dssm_version = data.get('consistent_dssm_version', '')
        self.generation = data.get('consistent_generation', '')
        self.env, self.type, self.dc, _ = self.id.split('@', 3)
        self.hosts = []
        self.description = '\n'.join([
            '{} ({})'.format(self.id, self.name),
            str(self.status),
            '{}/{}'.format(self.report_version, self.dssm_version),
            str(self.generation),
        ])
        self.ctype = ''

    def get_consistent_version(self, package):
        if package == 'yandex-market-report':
            return self.report_version
        elif package == 'yandex-market-report-formulas':
            return self.formulas_version
        elif package == 'yandex-market-report-dssm':
            return self.dssm_version
        return ''

    def detect_ctype(self):
        self.ctype = '' if not self.hosts else self.hosts[0].get_ctype()


class Host(object):

    def __init__(self, data):
        self.id = data.get('cluster_id', '')
        self.report_version = data.get('version', '')
        self.formulas_version = data.get('formulas_version', '')
        self.dssm_version = data.get('dssm_version', '')
        self.generation = data.get('generation', '')
        self.pinger_data = data

    def get_package_version(self, package):
        if package == 'yandex-market-report':
            return self.report_version
        elif package == 'yandex-market-report-formulas':
            return self.formulas_version
        elif package == 'yandex-market-report-dssm':
            return self.dssm_version
        return ''

    def get_ctype(self):
        nanny_info = self.pinger_data.get('nanny_info', None)
        if not nanny_info:
            return ''
        tags = nanny_info.get('tags', None)
        if not tags:
            return ''
        return tags.get('ctype', '')
