import datetime
import logging

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common.solomon import push_to_solomon_v2

LOG_DIR = '//logs/market-main-report-log/30min/'

SOLOMON_COMMON_LABELS = {
    'project': 'yt_log_stat',
    'cluster': 'cluster',
    'service': 'pusher',
}


def get_rounded_time():
    now = datetime.datetime.now()
    return now - datetime.timedelta(hours=1, minutes=30 + now.minute % 30)


def get_table_name_time(now):
    table_time = datetime.time(now.hour, now.minute)
    return '{}T{}'.format(now.date(), table_time)


def get_curr_prev_time(now):
    curr = get_table_name_time(now)
    prev = get_table_name_time(now - datetime.timedelta(minutes=30))
    return curr, prev


def get_binary_resource_path(resource_type):
    resource = sdk2.Resource.find(
        type=resource_type,
        attrs={
            "released": "stable",
        },
    ).first()

    if not resource:
        raise errors.TaskFailure("Could not find binary")

    data = sdk2.ResourceData(resource)
    path = data.path
    logging.info("Using binary %s from resource = %s created by %s" % (path, resource.id, resource.task_id))

    return str(path)


def run_command(task, command, path, logger):
    with sdk2.helpers.ProcessLog(task, logger=logger) as pl:
        sdk2.helpers.subprocess.check_call([command], stdout=pl.stdout, stderr=pl.stderr, shell=True, cwd=path)


def calc(task, cluster, inputs, output, advanced, mode):
    path = get_binary_resource_path('MARKET_TOOLS_YT_{}'.format(mode.upper()))
    command = ' '.join(['./yt_' + mode, '--cluster', cluster, '-o', output] + inputs + advanced)
    run_command(task, command, path, mode)


def get_table(cluster, token, table_path):
    from yt.wrapper import YtClient

    client = YtClient(proxy=cluster, token=token, config={"tabular_data_format": "dsv"})
    return client.read_table(table_path)


def create_sensors(table, table_time, sensor_name, value_column):
    sensors = []
    timestamp = int(table_time.strftime('%s'))
    for row in table:
        param, arg = row["param"], row["arg"]
        sensors.append(
            {
                'labels': {"sensor": sensor_name, "param": param, "arg": arg},
                'ts': timestamp,
                'value': float(row[value_column]),
            }
        )
    return sensors


def send_stat(vault, cluster, path, mode, value_column, labels, timestamp):
    tokens = vault.tokens.data()
    yt_token = tokens[vault.yt_token_name]
    table = get_table(cluster, yt_token, path)  # generator
    logging.info('{} mode'.format(mode))
    solomon_token = tokens[vault.solomon_token_name]
    sensors = create_sensors(table, timestamp, mode, value_column)
    push_to_solomon_v2(solomon_token, labels, sensors)


class Vault(sdk2.Parameters):

    tokens = sdk2.parameters.YavSecret(
        'YAV secret identifier (with optional version)', default='sec-01fd9apjvrv3j38wne5wjwvdhp', required=True
    )
    yt_token_name = sdk2.parameters.String('YT token name', default='yt-token', required=True)
    solomon_token_name = sdk2.parameters.String('Solomon token name', default='solomon-token', required=True)
