# -*- coding: utf-8 -*-
import logging

from sandbox import sdk2, common
import sandbox.projects.common.environments as env
from sandbox.projects.common import binary_task
from sandbox.common.types import resource as ctr

from .metrics import MetricFactory, MetricPeriod
from .repository import Clickhouse, YT, MetricsRepository, Migrations
from .template import template

debug = False


class CreateReportFromMarketAlarms(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            env.PipEnvironment("startrek_client", '2.5', custom_parameters=['--upgrade-strategy only-if-needed']),
            env.PipEnvironment('yql'),
            env.PipEnvironment('yandex-yt'),
            env.PipEnvironment('Jinja2', '2.8'),
            env.PipEnvironment('yoyo-migrations', '7.3.2'),
            env.PipEnvironment('psycopg2-binary', '2.7.7'),
        ]

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(custom=debug, stable=not debug)

        note = sdk2.parameters.String(
            "Примечание",
            multiline=True,
            default=""
        )

        recipients = sdk2.parameters.List(
            "Получатели",
            default=["andrewdm@yandex-team.ru" if debug else "market-logistics-reliability@yandex-team.ru"]
        )

        dry_run = sdk2.parameters.Bool("Dry Run", default=False)

        period_count = sdk2.parameters.Integer(
            "Количество периодов",
            default=1
        )

        secrets = sdk2.parameters.YavSecret(
            "Секреты",
            default="sec-01f7jx7fv8fwqbct8exyefgwp3"
        )

        enable_stability = sdk2.parameters.Bool("Оперативная доступность сервисов", default=False)

        with sdk2.parameters.Group("Курьерка") as courier:
            with sdk2.parameters.Group("Маршрутизация") as route:
                courier_route_week = sdk2.parameters.Float(
                    "Превышение времени на маршрутизацию за неделю (ключевые СЦ) (мин)",
                    default=None)
                courier_route_quarter = sdk2.parameters.Float(
                    "Превышение времени на маршрутизацию за квартал (ключевые СЦ) (мин)",
                    default=None)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "CREATE_REPORT_FROM_MARKET_ALARMS",
                "released": self.Parameters.binary_executor_release_type
            },
            "state": [ctr.State.READY]
        }

    def on_execute(self):
        secrets = self.Parameters.secrets.data()

        if not self.Parameters.dry_run:
            self._migrate(
                secrets=secrets
            )

        clickhouse_repo = Clickhouse(
            token=secrets["clickhouse.token"],
            query_path="query",
            dry_run=self.Parameters.dry_run
        )

        yt_repo = YT(
            token=secrets["clickhouse.token"],
            query_path="query",
            dry_run=self.Parameters.dry_run
        )

        metrics_repo = MetricsRepository(
            host=secrets["reporter.datasource.host"],
            port=secrets["reporter.datasource.port"],
            dbname=secrets["reporter.datasource.dbname"],
            user=secrets["reporter.datasource.username"],
            password=secrets["reporter.datasource.password"],
        )

        from .startrek import Startrek

        startrek_repo = Startrek(secrets["startrek.token"])

        factory = MetricFactory(
            clickhouse_repo=clickhouse_repo,
            yt_repo=yt_repo,
            startrek_repo=startrek_repo,
            metrics_repo=metrics_repo,
            thread_count=4,
            courier_route_week=self.Parameters.courier_route_week,
            courier_route_quarter=self.Parameters.courier_route_quarter
        )

        stability = factory.get_stability() if self.Parameters.enable_stability else []

        for i in range(0, self.Parameters.period_count):

            streams = factory.get_metrics(i + 1)
            logging.info('Processing period %s: streams: %s' % (i, streams))

            if not debug and not self.Parameters.dry_run:
                metrics_repo.save_history(streams, i + 1)

            self._send_notify(streams, stability, i + 1)

    def _send_notify(self, streams, stability, start_period_delta_days):
        if self.Parameters.dry_run:
            return

        subject = template.get_subject(start_period_delta_days)
        body = template.get_message_body(
            note=self.Parameters.note,
            streams=streams,
            stability=stability,
            start_period_delta_days=start_period_delta_days
        )
        self.server.notification(
            subject=subject,
            body=body,
            recipients=self.Parameters.recipients,
            transport=common.types.notification.Transport.EMAIL,
            urgent=True,
            type='html',
            task_id=self.id
        )

    @staticmethod
    def _migrate(secrets):
        migration_files = [
            "0001-initial.py",
            "0002-filling-streams.py",
            "0003-filling-groups.py",
            "0004-filling-metrics.py",
            "0005-history-index-create.py",
            "0006-add-sc-pvz-ajax.py",
            "0007-update-route-plan.py",
            "0008-rename-errors-metric.py",
            "0009-mqm-brand-new.py",
            "0010-add-resources-table.py",
            "0011-filling-resources-table.py",
            "0012-fix-capacity-metrics-unit.py",
            "0013-fix-mqm-metrics-name.py",
            "0014-courier-ajax.py",
            "0015-pvz-invisible-points.py",
            "0016-add-active-column.py",
            "0017-fix-sequence.py",
            "0018-add-yt-resource-type.py",
            "0019-add-pickup-point-availability.py",
            "0020-update-metrics-with-percent.py",
            "0021-add-pvz-sla-metrics.py",
            "0022-add-pvz-unavail-metrics.py",
            "0023-add-orders-after-cutoff-metrics.py",
            "0024-update-orders-after-cutoff-metrics.py",
            "0025-add-delays-after-cutoff-metrics.py",
            "0026-update-mqm-metrics.py",
            "0027-update-tracker-metrics.py",
            "0028-add-linehaul.py",
            "0029-courier-routing-calc-and-publish-time.py",
            "0030-add-linehaul-app-crash-free.py",
            "0031-up-pickup-point-availability.py",
            "0032-add-postamat.py",
            "0035-add-capacity-overflow.py",
            "0036-add-crash-free-sc.py",
            "0037-add-crash-free-courier.py",
            "0038-add-crash-free-magistral.py"
        ]

        migrations = Migrations(
            host=secrets["reporter.datasource.host"],
            port=secrets["reporter.datasource.port"],
            dbname=secrets["reporter.datasource.dbname"],
            user=secrets["reporter.datasource.username"],
            password=secrets["reporter.datasource.password"],
            migration_files=migration_files,
        )

        migrations.migrate()
