# -*- coding: utf-8 -*-

import logging
import threading

from .metrics import MetricGroup, StabilityMetric, MetricPeriod


class Task:
    def __init__(self, resource, factory, period_start_delta_in_days=1):
        self._resource = resource
        self._factory = factory
        self._period_start_delta_in_days = period_start_delta_in_days

    def execute(self):
        try:
            self._do_execute()
        except:
            logging.exception("Unable to execute query from %s" % self._resource._source)
            raise

    def _do_execute(self):
        if self._resource._resource_type == 'MANUAL':
            res = self._factory._input.get(self._resource._source)

        elif self._resource._resource_type == 'CLICKHOUSE':
            res = self._factory._clickhouse_repo.execute_query(
                self._resource._source,
                self._period_start_delta_in_days
            )

        elif self._resource._resource_type == 'YT':
            res = self._factory._yt_repo.execute_query(
                self._resource._source,
                self._period_start_delta_in_days
            )

        else:
            logging.error("Undefined resource type: " + self._resource._resource_type)
            return

        for ref in self._resource._resource_refs:
            current = None
            if not ref._positions:
                current = res
            elif res is not None and len(res) > 0:
                current = res
                for position in ref._positions:
                    try:
                        current = current[position]
                    except IndexError:
                        current = None
                        logging.warning("Error IN source " + self._resource._source, self._resource)
                        logging.warning("Undefined position " + str(position) + " in current", current)
            else:
                logging.warning("Error IN source " + self._resource._source + " result is None", self._resource)

            ref._metric._current = current


class MetricFactory:
    def __init__(
        self,
        clickhouse_repo,
        yt_repo,
        startrek_repo,
        metrics_repo,
        thread_count=4,
        **kwargs
    ):
        import queue

        self._clickhouse_repo = clickhouse_repo
        self._yt_repo = yt_repo
        self._startrek_repo = startrek_repo
        self._metrics_repo = metrics_repo

        self._group_names = {}
        self._stream_names = {}

        # Input variables
        self._input = kwargs

        # Multithreading
        self._results = {}
        self._thread_count = thread_count
        self._queue = queue.Queue()

    def _put(self, resource, periods_start_count):
        self._queue.put(
            Task(resource, self, periods_start_count)
        )

    def _work(self):
        while not self._queue.empty():
            logging.info("Run task")
            task = self._queue.get()

            try:
                task.execute()
            except Exception as e:
                logging.exception("Task execute error")

            logging.info("End task")
            self._queue.task_done()

    def _wait_execution(self):
        for i in range(self._thread_count):
            threading.Thread(target=self._work).start()
        self._queue.join()

    def get_metrics(self, periods_start_count=1):
        metrics = self._metrics_repo.get_metrics(periods_start_count)
        resources = self._metrics_repo.get_resources(metrics)

        for resource in resources:
            self._put(resource, periods_start_count)

        self._wait_execution()

        return self._metrics_repo.get_streams(metrics)

    def get_stability(self):
        streams = self._get_stability_groups()
        return streams

    def _get_stability_groups(self):
        streams = self._startrek_repo.get_downtimes()
        groups = []
        for stream in streams:
            metrics = []
            for group in streams[stream]:
                metrics.append(
                    StabilityMetric(
                        name=self._group_names[group] if group in self._group_names else group,
                        downtimes=streams[stream][group]
                    )
                )

            groups.append(
                MetricGroup(
                    id=0,
                    name=self._stream_names[stream] if stream in self._stream_names else stream,
                    metrics=metrics
                )
            )

        return groups


def set_metric_values(metrics, metric_id, current):
    for metric in metrics:
        if metric._id == metric_id:
            metric._current = current
            return


def get_metric(metrics, metric_id):
    for metric in metrics:
        if metric._id == metric_id:
            return metric
    return None
