import logging

from .exception import RepositoryException
from ..metrics import MetricStream, MetricGroup, Metric, Resource, ResourceRef, MetricPeriod


def with_cursor(method):
    def wrapper(self, *args):
        with self._conn.cursor() as cursor:
            return method(self, cursor, *args)

    return wrapper


class MetricsRepository:
    def __init__(self, host, port, dbname, user, password):
        import psycopg2

        self._conn = psycopg2.connect(
            host=host,
            port=port,
            dbname=dbname,
            user=user,
            password=password
        )

    def get_streams(self, metrics):
        if len(metrics) == 0:
            return []

        groups = self._get_groups(metrics)
        streams = self._get_streams(groups)
        return streams

    def get_metrics(self, periods_start_count=1):
        return self._get_metrics(
            periods_start_count=periods_start_count
        )

    def get_resources(self, metrics):
        return self._get_resources(metrics)

    def _get_metrics(self, periods_start_count):
        query = '''
            SELECT m.id, m.name, m.plan, m.unit, m.thrashold_type, h.value, m.periods, h.period FROM metrics m
            LEFT JOIN history h
                ON m.id = h.metric_id
                AND h.period = '{period}'
                AND h.start_period = date_trunc('{period_name}', current_date)::date - interval '{periods_start_count} {period_name}'
            WHERE m.active = TRUE and '{period}'::period = any(m.periods)
            ORDER BY m.id
        '''.format(
            period=MetricPeriod.ONE_DAY,
            period_name='day',
            periods_start_count=(periods_start_count + 1)
        )
        cursor = self._conn.cursor()
        cursor.execute(query)
        result = cursor.fetchall()
        metrics = {}
        for (id, name, plan, unit, thrashold_type, previous, m_periods, h_period) in result:
            more = thrashold_type == 'MORE'
            metrics[id] = Metric(
                id=id,
                name=name,
                plan=plan,
                unit=unit,
                more=more,
                previous=previous,
                periods=m_periods,
                period=h_period
            )

        return metrics

    @with_cursor
    def _get_groups(self, cursor, metrics):
        groups = {}
        if len(metrics) == 0:
            return groups

        query = '''
            SELECT g.id, g.name, g.stream_id, array_agg(m.id ORDER BY m.id) as metric_ids
            FROM groups g JOIN metrics m on g.id = m.group_id AND m.active = TRUE
            GROUP BY g.id ORDER BY g.id
        '''
        cursor.execute(query)
        result = cursor.fetchall()

        for (id, name, stream_id, metric_ids) in result:

            if stream_id not in groups:
                groups[stream_id] = []

            current_metrics = []
            for metric_id in metric_ids:
                if metric_id not in metrics:
                    continue
                current_metrics.append(metrics[metric_id])

            if len(current_metrics) > 0:
                groups[stream_id].append(MetricGroup(id, name, current_metrics))

        return groups

    @with_cursor
    def _get_streams(self, cursor, groups):
        streams = []
        if len(groups) == 0:
            return streams

        query = "SELECT id, name FROM streams ORDER BY id"
        cursor.execute(query)
        result = cursor.fetchall()

        for (id, name) in result:
            if id not in groups:
                continue
            streams.append(MetricStream(id, name, groups[id]))

        return streams

    @with_cursor
    def save_history(self, cursor, streams, start_period_delta_days=1):
        query = '''
            INSERT INTO history (metric_id, value, start_period, period)
            VALUES (
                %s,
                %s,
                date_trunc('{period_name}', current_date)::date - interval '{periods_start_count} {period_name}',
                '{period}'
            )
            ON CONFLICT (metric_id, start_period, period) DO UPDATE SET value = excluded.value, created_at = CURRENT_TIMESTAMP
            '''.format(
            period=MetricPeriod.ONE_DAY,
            period_name='day',
            periods_start_count=start_period_delta_days
        )

        values = []

        for stream in streams:
            for group in stream._groups:
                for metric in group._metrics:
                    if metric._current is None or metric._current is list:
                        logging.warning("Metric " + metric._name + " value is None or is list")
                        continue
                    values.append((metric._id, metric._current))

        cursor.executemany(query, values)
        self._conn.commit()

    @with_cursor
    def _get_resources(self, cursor, metrics):
        query = '''
            SELECT r.id, r.resource_type, r.source, rr.metric_id, rr.positions
            FROM resources r
            JOIN resources_ref rr on r.id = rr.resource_id
            ORDER BY r.id
        '''
        cursor.execute(query)
        result = cursor.fetchall()
        resources = {}
        for (id, resource_type, source, metric_id, positions) in result:
            if metric_id not in metrics:
                continue

            ref = ResourceRef(metrics[metric_id], positions)
            if id not in resources:
                resources[id] = Resource(id, resource_type, source, [ref])
            else:
                resources[id]._resource_refs.append(ref)

        res = []
        for i in resources:
            res.append(resources[i])

        return res
