# -*- coding: utf-8 -*-

import logging
from datetime import datetime


class Interval:
    def __init__(self, start_time, end_time, duration):
        self.start_time = start_time
        self.end_time = end_time
        self.duration = duration


class Startrek:
    def __init__(self, token):
        import startrek_client

        self._client = startrek_client.Startrek(
            useragent="",
            base_url="https://st-api.yandex-team.ru/v2/",
            token=token
        )

        logging.info("Connected to startrek")

    def get_downtimes(self):
        issues = self._get_issues()
        return self._append(issues)

    def _get_issues(self):
        return self._client.issues.find(
            query='Queue: "MARKETALARMS" and "Created": week() - "1w" and Tags: "delivery" "Sort By": created ASC',
            per_page=10000
        )

    def _append(self, issues):
        downtimes = {}
        intervals = {}
        for issue in issues:
            metric_group = None
            metric_subgroup = None

            for tag in issue.tags:
                items = tag.split(':')
                if len(items) != 2:
                    continue
                if items[0] == 'group':
                    metric_group = items[1]
                if items[0] == 'subgroup':
                    metric_subgroup = items[1]

                if metric_group is not None and metric_subgroup is not None and issue.vdt is not None:
                    if metric_group not in downtimes:
                        downtimes[metric_group] = {}
                    if metric_subgroup not in downtimes[metric_group]:
                        downtimes[metric_group][metric_subgroup] = []

                    if metric_group not in intervals:
                        intervals[metric_group] = {}
                    if metric_subgroup not in intervals[metric_group]:
                        intervals[metric_group][metric_subgroup] = []

                    start_time = issue.sreBeginTime
                    end_time = issue.sreEndTime
                    if start_time is None or end_time is None:
                        logging.error("Time is zero in ticket, %s", issue.key)
                        downtimes[metric_group][metric_subgroup].append(issue.vdt)
                        continue

                    try:
                        start_time = datetime.strptime(start_time, '%Y-%m-%dT%H:%M:%S.%f+0000')
                        end_time = datetime.strptime(end_time, '%Y-%m-%dT%H:%M:%S.%f+0000')
                    except Exception as e:
                        logging.error(e)
                        downtimes[metric_group][metric_subgroup].append(issue.vdt)
                        continue

                    current = Interval(start_time, end_time, issue.vdt)
                    self._merge_intervals(current, intervals[metric_group][metric_subgroup])

        for metric_group in intervals:
            for metric_subgroup in intervals[metric_group]:
                for interval in intervals[metric_group][metric_subgroup]:
                    downtimes[metric_group][metric_subgroup].append(interval.duration)

        return downtimes

    def _merge_intervals(self, current, intervals):
        for interval in intervals:
            if current.start_time < interval.start_time and current.end_time < interval.start_time:
                continue
            if current.start_time > interval.end_time and current.end_time > interval.end_time:
                continue
            if current.start_time <= interval.start_time and current.end_time >= interval.end_time:
                current.duration = max(interval.duration, current.duration)
                intervals.remove(interval)
                continue
            if current.start_time >= interval.start_time and current.end_time <= interval.end_time:
                interval.duration = max(interval.duration, current.duration)
                return
            if current.start_time < interval.start_time <= current.end_time:
                current.end_time = interval.end_time
                current.duration = max(interval.duration, current.duration)
                intervals.remove(interval)
                self._merge_intervals(current, intervals)
                return
            if current.start_time >= interval.start_time and current.end_time > interval.end_time:
                current.start_time = interval.start_time
                current.duration = max(interval.duration, current.duration)
                intervals.remove(interval)
                self._merge_intervals(current, intervals)
                return

        intervals.append(current)
