# -*- coding: utf-8 -*-

import logging
import os
import datetime
import calendar

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.market.saas.ferryman import (
    start_ferryman_process,
    wait_ferryman,
)

current_date = datetime.datetime.now().strftime("%y-%m-%d")
current_time = datetime.datetime.now().strftime("%H-%M")
ferryman_status_check_interval = 5 * 60


class RebuildSaasKvChatsData(sdk2.Task):
    """Run YQL query to import chats data from dialogs.yandex.ru tables and push them to saas-kv"""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 300
        environments = (
            # TODO: Зафиксировать версию
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql'),
        )

    class Parameters(sdk2.Task.Parameters):
        yql_token_name = sdk2.parameters.String('YQL token name', default='YQL_TOKEN')
        yt_proxy = sdk2.parameters.String('YT proxy', required=True, default='hahn')
        tables_path = sdk2.parameters.String('YT folder', required=True, default='home/market/production/chats')

        mapper_make_resource = sdk2.parameters.Resource(
            'YA_MAKE Resource ID for YT Mapper',
            required=True
        )

    def on_execute(self):
        import yql.api.v1.client

        yql_token = sdk2.Vault.data('robot-market-chats', self.Parameters.yql_token_name)
        tables_path = self.Parameters.tables_path
        yt_proxy = self.Parameters.yt_proxy
        self.Context.src_table_name = "temp/" + current_date + "_" + current_time

        # Токен используется еще и в скрипте маппера, через YT_TOKEN
        os.environ['YT_TOKEN'] = yql_token

        with self.memoize_stage.yt:

            yql_query = """
            use {yt_proxy};

            PRAGMA yt.InferSchema = '1';
            PRAGMA yson.DisableStrict;

            $TODAY = CurrentUtcDate();
            $MONTH_AGO = CAST($TODAY - DateTime::IntervalFromDays(30) AS String);

            $is_last_30_days_table = ($table_name) -> {{
                return $table_name > $MONTH_AGO;
            }};

            $parse_punycode = ($url) -> {{
                RETURN IF (Url::PunycodeToHostName($url) != "", Url::PunycodeToHostName($url), $url);
            }};

            $get_host = ($url) -> {{
                RETURN $parse_punycode(String::ReplaceAll(String::ToLower(COALESCE(Url::Parse($url).Host, $url)), 'www.', ''));
            }};

            $CHATS = (
                SELECT
                    Yson::LookupString(Dialog, 'bot_id') AS bot_id,
                    CAST(chats.schedule as String) as chat_schedule,
                    IF
                    (
                        $get_host(Yson::LookupString(Dialog, 'brand_website')) != "",
                        $get_host(Yson::LookupString(Dialog, 'brand_website')),
                        IF (
                            main_host == "",
                            $parse_punycode(String::JoinFromList(ListTake(String::SplitToList(Url::CutWWW(Url::CutScheme(brand_website)), "/"), 1), "")),
                            $parse_punycode(Url::CutWWW(Url::CutScheme(main_host)))
                        )
                    ) AS website
                FROM
                    hahn.`home/business-chat/dialogs/prod/dialogs` as dialogs
                LEFT JOIN
                    hahn.`home/business-chat/prod/export/url_rules` as chats
                ON
                    dialogs.DialogId == chats.org_id
                WHERE
                    Yson::ConvertToBool(Yson::Lookup(Yson::Parse(Dialog), "is_enabled")) == true
                    AND Yson::ConvertToBool(Yson::Lookup(Yson::Lookup(Yson::Parse(Dialog), "settings"), "use_chat_on_market")) == true
            );

            -- Получаем список магазинов маркета
            --
            $SHOPS = (
                SELECT
                    shops.shop_id as shop_id,
                    IF (shops.domain == "",
                        $parse_punycode(String::JoinFromList(ListTake(String::SplitToList(Url::CutWWW(Url::CutScheme(shops.url)), "/"), 1), "")),
                        $parse_punycode(Url::CutWWW(Url::CutScheme(shops.domain)))
                    ) AS website,
                    shops.is_enabled as is_enabled
                FROM
                    hahn.`home/market/production/indexer/stratocaster/in/shopsdat/recent` as shops
                LEFT ONLY JOIN
                    hahn.`{tables_path}/disabled_shops` as disabled_shops
                ON
                    disabled_shops.shop_id == shops.shop_id
                WHERE
                    is_enabled == true
            );

            -- Получаем список из id магазина и кол-во активных дней за период
            --
            $SHOPS_ACTIVITY = (
                SELECT
                    datasource_id as shop_id,
                    active_days_count
                FROM
                    (
                        SELECT
                            datasource_id,
                            count(*) as active_days_count
                        FROM
                            filter(`home/market/production/mstat/dictionaries/shop_activity`, $is_last_30_days_table)
                        GROUP BY
                            datasource_id
                    )
            );

            -- Получаем данные активных магазинов
            --
            $ACTIVE_SHOPS = (SELECT * from
                (
                    SELECT
                        shops.shop_id as shop_id,
                        shops.website as website,
                        shops.is_enabled as is_enabled
                    FROM
                        $SHOPS as shops
                    LEFT JOIN
                        $SHOPS_ACTIVITY as shops_activity
                    ON
                        shops.shop_id == shops_activity.shop_id
                    WHERE
                        active_days_count > 0
                )
                WHERE
                    is_enabled == true or shop_id is not NULL
            );

            -- Матчим активные магазины с данными диалогов по url-паттернам (url_rules) яндекс диалогов
            -- Записываем это всё в таблицу
            --
            INSERT INTO hahn.`{tables_path}/{dst_table}` WITH TRUNCATE
            SELECT
                unique_chats.shop_id as shop_id,
                unique_chats.chat_id as chat_id,
                CAST(unique_chats.chat_schedule as Yson) as chat_schedule
            FROM
                (
                    SELECT DISTINCT
                        chats.bot_id as chat_id,
                        shops.shop_id as shop_id,
                        chats.chat_schedule as chat_schedule,
                        shops.website as website
                    FROM
                        $CHATS as chats
                    INNER JOIN
                        $ACTIVE_SHOPS as shops
                    ON
                        shops.website == chats.website
                ) as unique_chats
            ;
            """.format(dst_table=self.Context.src_table_name, tables_path=tables_path, yt_proxy=yt_proxy)

            with yql.api.v1.client.YqlClient(token=yql_token) as yql_client:
                query = yql_client.query(yql_query, syntax_version=1)
                query.run()

                if not query.get_results().is_success:
                    raise common.errors.TaskFailure("YQL request has failed: " + '; '.join(str(error) for error in query.get_results().errors))

        with self.memoize_stage.mapper:
            self.Context.destination_table_path = "//{tables_path}/{table}_result".format(table=self.Context.src_table_name, tables_path=tables_path)

            mapper_path = os.path.join(str(sdk2.ResourceData(self.Parameters.mapper_make_resource).path), "bin", "chats_mapper")
            rich_check_call(
                [mapper_path, "--proxy", "hahn", "--src", "//{tables_path}/{table}".format(table=self.Context.src_table_name, tables_path=tables_path), "--dst", self.Context.destination_table_path],
                task=self, alias="prepare_data"
            )

        with self.memoize_stage.start_ferryman:
            timestamp = calendar.timegm(datetime.datetime.utcnow().utctimetuple())
            self.Context.ferryman_service_name = "market-shops-chats"
            self.Context.ferryman_batch_id = start_ferryman_process(self.Context.destination_table_path, "0", timestamp, self.Context.ferryman_service_name)

        with self.memoize_stage.wait_ferryman(10):
            wait_ferryman(self.Context.ferryman_service_name, self.Context.ferryman_batch_id, ferryman_status_check_interval)

        with self.memoize_stage.remove_ferryman_table:
            yql_query = "DROP TABLE hahn.`{tables_path}/{dst_table}_result`".format(tables_path=tables_path, dst_table=self.Context.src_table_name)

            with yql.api.v1.client.YqlClient(token=yql_token) as yql_client:
                query = yql_client.query(yql_query, syntax_version=1)
                query.run()

                if not query.get_results().is_success:
                    raise common.errors.TaskFailure("YQL request has failed: " + '; '.join(str(error) for error in query.get_results().errors))

        logging.info('Done')
