import logging

import requests

from sandbox import sdk2
from sandbox.projects.market.solomon_pusher.helpers import base_task
from sandbox.projects.market.solomon_pusher.helpers import common, sql
from sandbox.projects.market.solomon_pusher.helpers.common import PreviousRunResourceSaver
from sandbox.projects.market.solomon_pusher.helpers.database import DatabaseFetcher


class PostSensorViaHttp(base_task.BaseTask):
    class Parameters(base_task.BaseTask.Parameters):
        alert_channel_name = sdk2.parameters.String('Alert channel name', required=True)

    _SENDER_URL = 'https://sender.yandex-team.ru/api/0/market.inner/transactional/C91UMZ83-SV61/send'
    _SENDER_TEST_URL = 'https://test.sender.yandex-team.ru/api/0/market/transactional/0MN8QZ83-73T1/send'
    log = logging.getLogger(__name__)

    def on_execute(self):
        sql_args = self.Parameters.additional_args
        use_date = self.Parameters.use_date_range
        service, filename, path_to_sql = common.get_path_names(self.Parameters.path_to_sql)
        config = common.read_config(common.get_service_config_file(service))
        channel_name = self.Parameters.alert_channel_name

        with PreviousRunResourceSaver(filename, self, use_date) as dates, DatabaseFetcher(config) as fetcher:
            prepared_sql = sql.prepare_sql(path_to_sql, dates + tuple(sql_args))
            columns, rows = fetcher.execute_sql(prepared_sql)
            if len(rows) == 0:
                return
            csv_file_name = common.generate_csv(columns, rows)
            self.send_http_post(config, channel_name, csv_file_name)

    def send_http_post(self, config, channel_name, csv_file_name):

        url = config.get(channel_name, "url")

        data = common.read_file(csv_file_name)
        self.log.info('Sending CSV to {}'.format(url))

        headers = {'Content-Type': 'text/csv'}
        resp = requests.post(url, data=data, headers=headers, verify=False)

        if resp.raise_for_status():
            self.log.info('Sending CSV to HTTP failed, status code={}'.format(resp.status_code))
            raise Exception('Sending CSV to HTTP failed, status code=' + str(resp.status_code))
        else:
            self.log.info('CSV was successfully sent to {}'.format(url))
