import base64
import json
import logging

import requests

from sandbox import sdk2
from sandbox.projects.market.solomon_pusher.helpers import base_task
from sandbox.projects.market.solomon_pusher.helpers import common, sql
from sandbox.projects.market.solomon_pusher.helpers.common import PreviousRunResourceSaver
from sandbox.projects.market.solomon_pusher.helpers.database import DatabaseFetcher


class SendSensorsToEmail(base_task.BaseTask):
    class Parameters(base_task.BaseTask.Parameters):
        alert_channel_name = sdk2.parameters.String('Alert channel name', required=True)

    _SENDER_URL = 'https://sender.yandex-team.ru/api/0/market.inner/transactional/C91UMZ83-SV61/send'
    _SENDER_TEST_URL = 'https://test.sender.yandex-team.ru/api/0/market/transactional/0MN8QZ83-73T1/send'
    log = logging.getLogger(__name__)

    def on_execute(self):
        sql_args = self.Parameters.additional_args
        use_date = self.Parameters.use_date_range
        service, filename, path_to_sql = common.get_path_names(self.Parameters.path_to_sql)
        config = common.read_config(common.get_service_config_file(service))
        channel_name = self.Parameters.alert_channel_name

        with PreviousRunResourceSaver(filename, self, use_date) as dates, DatabaseFetcher(config) as fetcher:
            prepared_sql = sql.prepare_sql(path_to_sql, dates + tuple(sql_args))
            columns, rows = fetcher.execute_sql(prepared_sql)
            if len(rows) == 0:
                return
            csv_file_name = common.generate_csv(columns, rows)
            subject_idx = common.index_of(columns, '__smtp_subject', -1)
            subject = config.get(channel_name, "subject") if subject_idx < 0 else rows[0][subject_idx]
            self.send_email_with_csv(config, csv_file_name, subject)

    def send_email_with_csv(self, config, csv_file_name, subject):
        channel_name = self.Parameters.alert_channel_name
        if self.Parameters.is_testing:
            token = sdk2.Vault.data('solomon_pusher_rassylyator_test_token')
            url = self._SENDER_TEST_URL
            send_to = ['solomonpushertest@yandex-team.ru']
        else:
            url = self._SENDER_URL
            token = sdk2.Vault.data('solomon_pusher_rassylyator_token')
            send_to = [x.strip() for x in config.get(channel_name, "sendTo").split(',')]

        encoded_csv = base64.b64encode(common.read_file(csv_file_name))
        attachment = json.dumps([{'filename': 'data.csv', 'mime_type': 'text/csv', 'data': encoded_csv}])
        args = json.dumps({'subject': subject})

        data = {
            'args': args,
            'async': 'true',
            'from_email': 'solomon-pusher@yandex-team.ru',
            'attachments': attachment
        }

        for email_address in send_to:
            params = (('to_email', email_address),)
            self.log.info('Sending email to {}'.format(email_address))
            response = requests.post(url,
                                     params=params,
                                     data=data,
                                     auth=(token, ''))
            self.log.debug(response.text)
            self.log.info('Email sent to {}'.format(email_address))
            response.raise_for_status()
