import ConfigParser
import codecs
import csv
import logging
import shutil
import tempfile
from datetime import datetime, timedelta
from os import path

import pytz

from sandbox import sdk2
from sandbox.projects.market.solomon_pusher.resources.PreviousRunResource import PreviousRunResource

log = logging.getLogger('helpers.common.py')
__PROJECT_ROOT_PATH = 'arcadia:/arc/trunk/arcadia/market/solomon-pusher/'
TMP_DIR = 'pusher'
ARC_PATH_TO_PUSHER = 'arcadia:/arc/trunk/arcadia/market/solomon-pusher/'
SERVICES_ROOT_PATH = TMP_DIR + '/services'
DEFAULT_SEMAPHORE_NAME = 'SOLOMON_PUSHER'


def get_path_names(arc_path):
    cut_path = arc_path.replace(__PROJECT_ROOT_PATH, '').split('/')
    service = cut_path[1]
    sql_path = [SERVICES_ROOT_PATH]
    for i in range(1, len(cut_path)):
        sql_path.append('/{}'.format(cut_path[i]))
    relative_path = ''.join(sql_path)
    filename = __prepare_filename(relative_path)
    log.info('service {}, filename {}, relative path {}'.format(service, filename, relative_path))
    return service, filename, relative_path


def __prepare_filename(relative_path):
    dirname = path.dirname(relative_path)
    head, tail = path.split(dirname)
    filename = path.basename(relative_path)
    return '{}-{}'.format(tail, filename)


def index_of(collection, element, default):
    return collection.index(element) if element in collection else default


def read_file(filename):
    with open(filename, 'r') as f:
        return f.read()


def get_service_config_file(service):
    return path.join(SERVICES_ROOT_PATH, service, service + '.cfg')


def get_service_from_path(path_to_sql):
    if not path_to_sql.startswith(SERVICES_ROOT_PATH):
        raise Exception("Path is not in services hierarchy: " + path_to_sql)
    path_parts = path_to_sql[len(SERVICES_ROOT_PATH) + 1:len(path_to_sql)].split(path.sep, 1)
    if path.altsep:
        path_parts = path_parts[0].split(path.altsep, 1)
    return path_parts[0]


def read_config(config_file):
    log.info('Reading config from {}'.format(config_file))
    config = ConfigParser.RawConfigParser()
    config.read(config_file)

    local_config_file = path.join(path.dirname(config_file), 'local-' + path.basename(config_file))
    config.read(local_config_file)

    return config


def get_config_option(config, section, option, def_value=None):
    return config.get(section, option) if config.has_option(section, option) else def_value


def generate_csv(sql_columns, sql_rows):
    csv_file, csv_file_name = tempfile.mkstemp(prefix=datetime.now().strftime('%Y%m%d_%H%M%S_'), suffix=".csv")
    logging.info("Writing {} rows into csv file {}".format(len(sql_rows), csv_file_name))

    csv_columns = {i: c for i, c in enumerate(sql_columns) if not c.startswith('__')}
    csv_cols_ind = csv_columns.keys()
    csv_cols_names = csv_columns.values()
    have_non_csv_columns = csv_cols_names != sql_columns

    with open(csv_file_name, 'wb') as fp:
        writer = csv.writer(fp, delimiter=';')
        writer.writerow(csv_cols_names)
        for row in sql_rows:
            row = [v for i, v in enumerate(row) if i in csv_cols_ind] if have_non_csv_columns else row
            writer.writerow(row)
    convert_file_encoding(csv_file_name, 'utf-8', 'cp1251')
    return csv_file_name


def convert_file_encoding(filename, from_encoding, to_encoding):
    block_size = 102400
    new_filename = filename + ".new"
    with codecs.open(filename, "r", from_encoding) as sourceFile:
        with codecs.open(new_filename, "w", to_encoding) as targetFile:
            while True:
                contents = sourceFile.read(block_size)
                if not contents:
                    break
                targetFile.write(contents)
    shutil.move(new_filename, filename)


def split_and_strip(s, delim):
    if s is None:
        return ()
    parts = [p.strip() for p in str(s).split(delim)]
    if parts == ['']:
        return ()
    return parts


def comma_list(list):
    return ', '.join(list)


class PreviousRunResourceSaver:
    _log = logging.getLogger(__name__)

    def __init__(self, filename, task, use_dates):
        self._use_dates = use_dates
        self._filename = filename
        self._task = task

    def __enter__(self):
        if not self._use_dates:
            return ()

        timezone = pytz.timezone('Europe/Moscow')
        self._datetime_now = datetime.now(timezone).strftime("%Y-%m-%d %H:%M:%S")

        datetime_resource = PreviousRunResource.find(
            resource_type=PreviousRunResource,
            attrs={'sql_name': self._filename}
        ).first()

        if datetime_resource:
            previous_run_timestamp = datetime_resource.previous_run_timestamp
            self._log.info('Found resource for filename {}, last run {}'
                           .format(self._filename, previous_run_timestamp))
        else:
            previous_run_timestamp = (datetime.now(timezone) - timedelta(minutes=5)).strftime("%Y-%m-%d %H:%M:%S")
        self._log.info('Resource for filename {} not found using {} for last run time'
                       .format(self._filename, previous_run_timestamp))
        return '\'{}\''.format(previous_run_timestamp), '\'{}\''.format(self._datetime_now)

    def __exit__(self, *args):
        if not self._use_dates:
            return

        resource = PreviousRunResource(self._task, 'Last run datetime', 'last_run')
        resource.sql_name = self._filename
        resource.previous_run_timestamp = self._datetime_now

        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)
        data.path.joinpath(self._filename).write_bytes(self._datetime_now)
        data.ready()
