import logging

from sandbox import common, sdk2


class DatabaseFetcher:

    _log = logging.getLogger(__name__)

    def __init__(self, config):

        self._config = config
        self._psycopg2 = __import__('psycopg2')
        self._pymssql = __import__('pymssql')
        self._yql = __import__('yql.dbapi').dbapi

    def __enter__(self):
        self._conn = self.__open_db_connection()
        return self

    def __exit__(self, *args):
        self._log.info('Closing connection')
        self._conn.close()

    def __read_config_options(self):
        host = self._config.get("Database", "host")
        port = self._config.get("Database", "port")
        user = self._config.get("Database", "user")
        vault_name = self._config.get("Database", "vaultName")
        try:
            vault_owner = self._config.get("Database", "vaultOwner")
        except Exception:
            vault_owner = None
        database = self._config.get("Database", "database")
        password = self.__get_vault_secret(vault_owner, vault_name)
        return host, port, user, password, database

    def __open_db_connection(self):
        sql_driver = self._config.get("Database", "driver")
        if sql_driver == 'psycopg':
            return self.__get_postgres_connection()
        elif sql_driver == 'pymssql':
            return self.__get_mysql_connection()
        elif sql_driver == 'yql':
            return self.__get_yql_connection()
        else:
            raise Exception('Wrong config for connecting to database')

    def __get_postgres_connection(self):
        host, port, user, password, database = self.__read_config_options()
        self._log.info('Connecting to postgres database {} at host {}:{} with user {}'
                       .format(database, host, port, user))
        return self._psycopg2.connect(user=user,
                                      password=password,
                                      host=host,
                                      port=port,
                                      database=database,
                                      sslmode='require')

    def __get_mysql_connection(self):
        host, port, user, password, database = self.__read_config_options()
        self._log.info('Connecting to mysql database {} at host {}:{} with user {}'
                       .format(database, host, port, user))
        return self._pymssql.connect(user=user,
                                     password=password,
                                     host=host,
                                     port=port,
                                     database=database)

    def __get_yql_connection(self):
        _, _, _, oauth_token, database = self.__read_config_options()
        syntax_version = self._config.getint("Database", "syntax_version")
        self._log.info('Connecting to yql database {} with syntax_version {}'
                       .format(database, syntax_version))
        return self._yql.connect(database=database,
                                 token=oauth_token,
                                 syntax_version=syntax_version)

    def execute_sql(self, prepared_sql):
        try:
            cur = self._conn.cursor()

            self._log.info('Executing sql\n{}'.format(prepared_sql))

            cur.execute(prepared_sql)

            self._log.info('Fetching results')
            columns = [i[0] for i in cur.description]
            rows = cur.fetchall()

            return columns, rows
        except Exception as e:
            self._log.error('Exception of type {} occurred during fetching db data, message {}'
                            .format(type(e), e.message), e)
            raise e

    @staticmethod
    def __get_vault_secret(secret_owner, secret_name):
        try:
            return sdk2.Vault.data(secret_owner, secret_name)
        except common.errors.VaultError:
            raise common.errors.TaskFailure('Vault secret for name {} and owner {} not found'.format(secret_name, secret_owner))
