import os
import re

from sandbox.projects.market.solomon_pusher.helpers import common
from sandbox.projects.market.solomon_pusher.helpers.database import DatabaseFetcher


def prepare_sql(sql_file, sql_args=()):
    sql = common.read_file(sql_file)

    dic = {str(i + 1): a for i, a in enumerate(sql_args)}
    sql_path = os.path.dirname(sql_file)

    regexp = re.compile(r'\$([/A-Za-z0-9_-]*)(\((.*?(\(.*?\))*?.*?)\))?')
    return regexp.sub(lambda m: __fill_sql_placeholder(m, dic, sql_path), sql)


def __fill_sql_placeholder(match, dic, sql_path):
    groups = match.groups()
    name = groups[0]
    params = groups[2].split(',') if groups[2] else ()
    params = [p.strip() for p in params]

    if name in dic:
        return dic.get(name)

    for i, param in enumerate(params):
        if param.startswith('$'):
            param_name = param[1:len(param)]
            if param_name in dic:
                params[i] = dic.get(param_name)

    is_absolute_name = name.startswith('/')
    sql_files = []
    if is_absolute_name:
        sql_files.append(os.path.join(common.SERVICES_ROOT_PATH, name.lstrip('/') + ".sql"))
    else:
        sql_files.append(os.path.join(sql_path, name + ".sql"))
        util_sql_path = os.path.join(common.SERVICES_ROOT_PATH, common.get_service_from_path(sql_path), 'util')
        sql_files.append(os.path.join(util_sql_path, name + ".sql"))

    for sql_file in sql_files:
        if os.path.exists(sql_file):
            sql = prepare_sql(sql_file, params)
            if is_absolute_name:
                cur_service = common.get_service_from_path(sql_path)
                new_service = common.get_service_from_path(sql_file)
                if cur_service != new_service:
                    new_service_config = common.read_config(common.get_service_config_file(new_service))
                    new_sql = prepare_sql(sql_file, params)
                    with DatabaseFetcher(new_service_config) as fetcher:
                        columns, rows = fetcher.execute_sql(new_sql)
                        sql = write_rows_to_sql(rows, columns)
            return sql

    # Could not resolve placeholder - just leave it as it was
    if groups[2] is None:
        return '$' + name
    else:
        return '$' + name + '(' + str(groups[2]) + ')'


def write_rows_to_sql(rows, columns):
    sql = "("
    for i, row in enumerate(rows):
        if i > 0:
            sql += " union all\n"
        sql += "select "
        for j, value in enumerate(row):
            if j > 0:
                sql += ", "
            if value is None:
                sql += "null"
            elif isinstance(value, int) or isinstance(value, float) or isinstance(value, bool):
                sql += str(value)
            else:
                sql += "'" + str(value) + "'"
            if i == 0:
                sql += " as " + columns[j]
    sql += ")"
    return sql
