# coding=utf-8
import importlib
import logging
import os
from datetime import datetime

from sandbox.projects.market.solomon_pusher.helpers.common import get_config_option, generate_csv, split_and_strip, \
    comma_list, index_of


class ST:
    ST_CONFIG_SECTION = "StarTrek"
    ST_SQL_COLUMN_QUEUE = "__st_queue"
    ST_SQL_COLUMN_SUMMARY = "__st_summary"
    ST_SQL_COLUMN_DESCRIPTION = "__st_description"
    ST_SQL_COLUMN_COMPONENTS = "__st_components"
    ST_SQL_COLUMN_TAGS = "__st_tags"
    ST_SQL_COLUMN_ORDER_NUMBER = "__st_customer_order_number"
    ST_SQL_COLUMN_WEIGHT = "__st_weight"
    ST_SQL_COLUMN_RETURN_REASON = "__st_return_reason"
    ST_SQL_COLUMN_PAYMENT_METHOD = "__st_payment_method"
    ST_SQL_COLUMN_CUSTOMER_EMAIL = "__st_customer_email"
    ST_SQL_COLUMN_CUSTOMER_COST_ADV = "__st_cost_adv"
    ST_TEST_QUEUE_NAME = 'SOLOMONTEST'
    ST_TEST_COMPONENT = 58099

    def __init__(self, config, token, alert_channel_name, is_testing):
        st = importlib.import_module('startrek_client')
        self.not_found_exc = st.exceptions.NotFound
        self.alert_channel_name = alert_channel_name
        self.config = config
        self.is_testing = is_testing
        self.client = self.create_st_client(st, token)

    def create_st_client(self, st, token):
        st_user_agent = get_config_option(self.config, self.ST_CONFIG_SECTION, "userAgent")
        st_url = get_config_option(self.config, self.ST_CONFIG_SECTION, "url")
        client = st.Startrek(useragent=st_user_agent, base_url=st_url, token=token)
        return client

    def get_st_login(self):
        return get_config_option(self.config, self.ST_CONFIG_SECTION, "login")

    def create_st_csv_attachment(self, columns, csv_rows):
        filename = generate_csv(columns, csv_rows) if len(csv_rows) > 1 else None
        if filename:
            att = self.client.attachments.create(file=filename, params={'filename': os.path.basename(filename)})
            return [int(att.id)]
        return []

    @staticmethod
    def format_exploding_comma_list(title, list):
        if not list:
            return ''
        elif len(list) < 6:
            return title + comma_list(list) + '\n'
        else:
            return '<{' + title + '\n' + comma_list(list) + '}>\n'

    def create_or_update_ticket(self, st_params, columns, csv_rows):
        if self.is_testing:
            st_params['queue'] = self.ST_TEST_QUEUE_NAME
            st_params['components'] = self.ST_TEST_COMPONENT
        issues = self.client.issues.find('Queue: ' + st_params['queue'] +
                                         ' Summary: #"' + st_params['summary'] + '"' +
                                         ' "Sort By": Key ASC'
                                         )

        close_tickets = get_config_option(self.config, self.alert_channel_name, 'closeTickets', 'false') == 'true'
        update_tickets = get_config_option(self.config, self.alert_channel_name, 'updateTickets', 'false') == 'true'
        close_duplicate_tickets = get_config_option(self.config, self.alert_channel_name, 'closeDuplicateTickets',
                                                    'false') == 'true'

        if issues:
            main_issue = issues[0]
            if close_tickets or close_duplicate_tickets:
                self.close_duplicates(issues, main_issue, st_params)

            issue = main_issue
            print datetime.now(), 'Found existing ticket', issue.key

            if update_tickets:
                issue.macroTag = datetime.strptime(issue.createdAt[:10], '%Y-%m-%d').strftime('%d.%m.%Y')
                issue = self.reopen_closed(issue)
                self.update_ticket_if_needed(issue, st_params, columns, csv_rows)

            return None

        st_params['attachmentIds'] = self.create_st_csv_attachment(columns, csv_rows)
        st_params['macroTag'] = datetime.now().strftime('%d.%m.%Y')

        issue = self.client.issues.create(**st_params)
        logging.info('Created new ticket: {}'.format(issue.key))

        return issue

    def update_ticket_if_needed(self, issue, st_params, columns, csv_rows):
        if 'customerOrderNumber' in st_params:
            issue_orders = split_and_strip(issue.customerOrderNumber, ',')
            current_orders = split_and_strip(st_params.get('customerOrderNumber'), ',')
            missing_orders = set(issue_orders).difference(current_orders)
            new_orders = set(current_orders).difference(issue_orders)
            if missing_orders or new_orders:
                self.update_ticket(issue, st_params, columns, csv_rows,
                                   self.format_exploding_comma_list(u'Удалены неактуальные заказы '
                                                                    u'(' + str(len(missing_orders)) + u' шт): ',
                                                                    missing_orders) +
                                   self.format_exploding_comma_list(u'Добавлены новые заказы '
                                                                    u'(' + str(len(new_orders)) + u' шт): ',
                                                                    new_orders))
        else:
            self.update_ticket(issue, st_params, columns, csv_rows)

    def reopen_closed(self, issue):
        if issue.status.key == 'closed':
            logging.info('Ticket {} has been closed but the problem remains - reopening the ticket'.format(issue.key))
            self.reopen_ticket(issue, u'Тикет снова актуален')
            issue = self.client.issues[issue.key]
        return issue

    @staticmethod
    def close_duplicates(issues, main_issue, st_params):
        duplicating_issues = [i for i in issues if i.status.key != 'closed' and i != main_issue]
        if duplicating_issues:
            logging.info('Found duplicating tickets matching summary {}, main ticket {}, duplicating tickets: {}'
                         .format(st_params['summary'], main_issue.key, str([i.key for i in duplicating_issues])))
            for issue in duplicating_issues:
                transition = issue.transitions['close']
                transition.execute(comment=u'Тикет дублирует ' + main_issue.key,
                                   resolution='duplicate')

    def reopen_ticket(self, issue, comment):
        for tran_name in ('reopen', 'new'):
            try:
                transition = issue.transitions[tran_name]
            except self.not_found_exc:
                continue
            transition.execute(comment=comment)
            return
        raise Exception('Could not find appropriate transition to reopen ticket ' + issue.key +
                        ' Available transitions for this ticket: ' + str([t.id for t in issue.transitions]))

    def update_ticket(self, issue, st_params, columns, csv_rows, comment_text=''):
        logging.info('Updating ticket {}'.format(issue.key))

        new_descr = st_params.get('description') if 'description' in st_params else ''
        for key in ('queue', 'description', 'summary'):
            st_params.pop(key)
        comment_text = \
            u'Информация в тикете была автоматически изменена.\n\n' + \
            comment_text + \
            '\n' + new_descr.decode('utf-8')
        issue.update(**st_params)
        issue.comments.create(
            text=comment_text,
            attachmentIds=self.create_st_csv_attachment(columns, csv_rows))

    def create_or_update_tickets_from_sql_rows(self, columns, rows):
        logging.info('Creating ST tickets for {} rows'.format(len(rows)))

        queue_idx = columns.index(self.ST_SQL_COLUMN_QUEUE)
        summary_idx = columns.index(self.ST_SQL_COLUMN_SUMMARY)
        descr_idx = index_of(columns, self.ST_SQL_COLUMN_DESCRIPTION, -1)
        components_idx = index_of(columns, self.ST_SQL_COLUMN_COMPONENTS, -1)
        tags_idx = index_of(columns, self.ST_SQL_COLUMN_TAGS, -1)
        customer_order_number_idx = index_of(columns, self.ST_SQL_COLUMN_ORDER_NUMBER, -1)
        weight_idx = index_of(columns, self.ST_SQL_COLUMN_WEIGHT, -1)
        return_reason = index_of(columns, self.ST_SQL_COLUMN_RETURN_REASON, -1)
        payment_method = index_of(columns, self.ST_SQL_COLUMN_PAYMENT_METHOD, -1)
        customer_email = index_of(columns, self.ST_SQL_COLUMN_CUSTOMER_EMAIL, -1)
        cost_adv_idx = index_of(columns, self.ST_SQL_COLUMN_CUSTOMER_COST_ADV, -1)

        prev_st_params = None
        csv_rows = []
        st_queues = set()
        actual_orders = set()
        for row in rows:
            st_params = {'queue': row[queue_idx] if not self.is_testing else self.ST_TEST_QUEUE_NAME,
                         'summary': row[summary_idx]}
            st_queues.add(st_params['queue'])
            if descr_idx >= 0:
                st_params['description'] = row[descr_idx]
            if components_idx >= 0:
                st_params['components'] = [int(c) for c in row[components_idx].split(',')] \
                    if not self.is_testing else self.ST_TEST_COMPONENT
            if tags_idx >= 0:
                st_params['tags'] = [tag.replace(' ', '_') for tag in split_and_strip(row[tags_idx], ',')]
            if customer_order_number_idx >= 0:
                orders_list = row[customer_order_number_idx]
                st_params['customerOrderNumber'] = orders_list
                [actual_orders.add(o) for o in split_and_strip(orders_list, ',')]
            if weight_idx >= 0:
                st_params['weight'] = row[weight_idx]
            if payment_method >= 0:
                st_params['paymentMethod'] = row[payment_method]
            if return_reason >= 0:
                st_params['returnReason'] = row[return_reason]
            if customer_email >= 0:
                st_params['customerEmail'] = row[customer_email]
            if cost_adv_idx >= 0:
                st_params['costAdv'] = float(row[cost_adv_idx])

            # если тикет изменился(изменились колонки __st), переходим к формированию следующего
            if prev_st_params and prev_st_params != st_params:
                self.create_or_update_ticket(prev_st_params, columns, csv_rows)
                prev_st_params = None
                csv_rows = []
            # если это тот же самый тикет, добавляем записи в csv выгрузку тикета
            if not prev_st_params or prev_st_params == st_params:
                csv_rows.append(row)
            prev_st_params = st_params

        if prev_st_params:
            self.create_or_update_ticket(prev_st_params, columns, csv_rows)

        if get_config_option(self.config, self.alert_channel_name, 'closeTickets') == 'true':
            self.close_non_actual_tickets(st_queues, actual_orders)

        logging.info('Finished creating tickets')

    def close_non_actual_tickets(self, st_queues, actual_orders):
        if not st_queues:
            logging.warn('Found no problem orders - cannot figure out ST-queue and clean it - '
                         'you can clean it by yourself - just close all open tickets :)')
            return
        if len(st_queues) > 1:
            logging.warn('Found more than one ST-queue in sql results - cannot clean ST-queue!')
            return
        st_queue = st_queues.pop()
        logging.info('Cleaning ST queue {}'.format(st_queue))

        st_login = self.get_st_login()
        issues = self.client.issues.find('Queue: ' + st_queue + ' Status: !Closed')  # Assignee: empty()
        issues_keys = [i.key for i in issues]

        logging.info('Found {} actual problem orders {}'.format(len(actual_orders), actual_orders))
        logging.info('Found {} open issues in {}'.format(len(issues_keys), st_queue))

        for issue_key in issues_keys:
            logging.info('Checking ticket {}'.format(issue_key))

            issue = self.client.issues[issue_key]
            issue_orders = split_and_strip(issue.customerOrderNumber, ',')
            actual_issue_orders = [o for o in issue_orders if o in actual_orders]

            logging.info('Ticket {} has {} actual problem orders: {}'
                         .format(issue.key, len(actual_issue_orders), comma_list(actual_issue_orders)))

            if not actual_issue_orders:
                if issue.assignee and issue.assignee.login != st_login:
                    if issue_orders:
                        logging.info('Updating assigned ticket {}'.format(issue_key))
                        comment_text = \
                            u'Информация в тикете была автоматически изменена.\n\n' + \
                            u'Удалены неактуальные заказы: ' + comma_list(issue_orders) + '\n\n' + \
                            u'Проблема более не актуальна'
                        issue.update(customerOrderNumber='')
                        issue.comments.create(text=comment_text)
                else:
                    logging.info('Closing unassigned ticket {} '.format(issue.key))
                    issue.update(assignee=st_login)
                    transition = issue.transitions['close']
                    transition.execute(comment='Проблема более не актуальна', resolution="won'tFix")
