# coding=utf-8

import logging
import os.path
import re

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sandboxsdk import environments
from sandbox.sdk2.vcs.svn import Arcadia, SvnError
from sandbox.projects.abc.client import AbcClient


class ArcadiaGroupActualizer(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('startrek_client', custom_parameters=['requests==2.18.4']),
            environments.PipEnvironment("toml")
        ]

    class Parameters(sdk2.Task.Parameters):
        ARCADIA_USER = "zomb-sandbox-rw"
        GROUP_FILES_DIR = "groups"
        DOCS = "Документация: https://nda.ya.ru/t/eDzQ0Ct83cu9j6"

        _ = sdk2.parameters.Info(DOCS)
        with sdk2.parameters.Group("Общие параметры") as base_block:
            yav_secret = sdk2.parameters.YavSecret("Секрет в Yav с OAuth токенами", required=True)
            group_name = sdk2.parameters.String(
                "Имя группы",
                required=True,
                description="Оно же имя файла. Имя не должно содержать точек")
            display_name = sdk2.parameters.String(
                "Отображаемое имя группы",
                default=None,
                description="Необязательное. Должно состоять только из ASCII символов")
            mail = sdk2.parameters.String(
                "E-mail группы",
                default=None,
                description="Необязательное. Должно состоять только из ASCII символов и заканчиваться на @yandex-team.ru")

        with sdk2.parameters.Group("Параметры ABC") as abc_block:
            abc_secret_key = sdk2.parameters.String("Ключ токена ABC", required=True, default="ABC_TOKEN")
            abc_service = sdk2.parameters.String("Ключ ABC сервиса", required=True, description="Например, marketito")
            abc_role_scope = sdk2.parameters.String(
                "Scope роли ABC-сервиса",
                default=None,
                description="Например, administration или development. "
                            "Можно оставить пустым, тогда будут найдены все участники сервиса")
            abc_include_robots = sdk2.parameters.Bool("Включать ли роботов в члены группы?", default=False)

        with sdk2.parameters.Group("Параметры Startrek") as st_block:
            st_secret_key = sdk2.parameters.String("Ключ токена Startrek", required=True, default="STARTREK_TOKEN")
            st_queue = sdk2.parameters.String("Очередь в Startrek", required=True)
            st_tags = sdk2.parameters.List(
                "Теги для поиска тикета", required=True,
                description="Уникальные теги необходимы чтобы можно было найти тикеты")

        with sdk2.parameters.Output:
            output = sdk2.parameters.String("Результат работы", default=None, required=True)

    def on_execute(self):
        self.st = self._get_st_api(self.Parameters.yav_secret.data()[self.Parameters.st_secret_key])
        open_tickets = self._get_open_tickets(self.st, self.Parameters.st_queue, self.Parameters.st_tags)
        if open_tickets:
            limit = 10
            tickets = ", ".join([t.key for i, t in enumerate(open_tickets) if i < limit])
            message = "Found open tickets: {}".format(tickets)
            logging.info(message)
            self.Parameters.output = message
            return
        logging.info("There are no open tickets. Continue.")

        arcadia_dir = self._checkout_arcadia(self.Parameters.GROUP_FILES_DIR)
        filename = os.path.join(arcadia_dir, self.Parameters.group_name)
        old_content = self._get_group_content(filename)
        new_content = {
            "members": self._get_actual_members(
                self.Parameters.yav_secret.data()[self.Parameters.abc_secret_key],
                self.Parameters.abc_service,
                self.Parameters.abc_role_scope,
                self.Parameters.abc_include_robots
            )
        }
        if self.Parameters.display_name:
            new_content["display_name"] = self._validate_display_name(self.Parameters.display_name)
        if self.Parameters.mail:
            new_content["mail"] = self._validate_mail(self.Parameters.mail)
        if old_content != new_content:
            self._write_toml_content(filename, new_content)
            is_new_file = not bool(old_content)
            if is_new_file and "." in self.Parameters.group_name:
                raise TaskFailure("File name should not contain dots.")
            self._commit_changes(arcadia_dir, filename, is_new_file)
        else:
            logging.info("Nothing to commit")
            self.Parameters.output = "Nothing to commit"

    def _get_st_api(self, st_token):
        """Получение объекта Startrek API."""
        from startrek_client import Startrek

        return Startrek(useragent='sandbox-arcadia_group_actualizer', token=st_token)

    def _get_open_tickets(self, st, st_queue, st_tags):
        query = [
            "Queue: {}".format(st_queue),
            "Status: !closed"
        ]
        for tag in st_tags:
            query.append("Tags: {}".format(tag))
        return st.issues.find(query=" ".join(query))

    def _checkout_arcadia(self, arcadia_path):
        arcadia_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, arcadia_path)
        arcadia_dir = Arcadia.checkout(arcadia_url, str(self.path("arcadia")))
        logging.info("Arcadia is checkouted to %s", arcadia_dir)
        return arcadia_dir

    def _get_group_content(self, filename):
        import toml

        if os.path.exists(filename):
            with open(filename) as fd:
                return toml.load(fd)
        return None

    def _get_actual_members(self, abc_token, abc_service, role_scope, include_robots):
        abc_client = AbcClient(abc_token)
        service_info = abc_client.get_service_info_by_slug(abc_service.lower())

        # is_robot: True - only robots, False - exclude robots, None = both
        is_robot_map = {True: None, False: False}

        members = abc_client.get_people_from_service(
            service_id=service_info["id"],
            role_scope=role_scope.lower(),
            is_robot=is_robot_map[include_robots]
        )
        logging.info(str(members))
        if not members:
            raise TaskFailure("Can't find members for {}/{}".format(abc_service.lower(), role_scope.lower()))
        return sorted(set(members))

    def _is_ascii(self, text):
        if isinstance(text, unicode):
            try:
                text.encode('ascii', errors='strict')
            except UnicodeEncodeError:
                return False
        else:
            try:
                text.decode('ascii', errors='strict')
            except UnicodeDecodeError:
                return False
        return bool(text.replace('?', ''))

    def _validate_display_name(self, display_name):
        if display_name and not self._is_ascii(display_name):
            raise TaskFailure("Non-ascii symbols not allowed in display_name")
        return display_name

    def _validate_mail(self, mail):
        if mail:
            if not mail.endswith("@yandex-team.ru"):
                raise TaskFailure("Mail not at yandex-team")
            if not self._is_ascii(mail):
                raise TaskFailure("Non-ascii symbols not allowed in mail")
        return mail

    def _get_formatted_members(self, members):
        # Если научите меня делать это силами toml - буду благодарен
        formatted_members = ["members = ["]
        for member in members:
            formatted_members.append('  "{}",'.format(member))
        formatted_members.append("]\n")
        return "\n".join(formatted_members)

    def _write_toml_content(self, filename, content):
        import toml

        with open(filename, "w") as fd:
            fd.write(self._get_formatted_members(content["members"]))
            for field in ("display_name", "mail"):
                if field in content:
                    fd.write("\n" + toml.dumps({field: content[field]}))

    def _create_issue(self, review_url):
        description = """В ходе актуализации Аркадийной группы {} был создан PR {}.
Необходимо убедиться, что он успешно смержен и закрыть этот тикет.

{}
""".format(self.Parameters.group_name, review_url, self.Parameters.DOCS)
        ticket = self.st.issues.create(
            queue=self.Parameters.st_queue,
            summary="Актуализация аркадийной группы {}".format(self.Parameters.group_name),
            description=description,
            tags=self.Parameters.st_tags,
        )
        return ticket.key

    def _commit_changes(self, arcadia_dir, filename, is_new_file):
        logging.info("Commiting changes")

        if is_new_file:
            Arcadia.add(filename)

        logging.info("Svn status: %s", Arcadia.status(arcadia_dir))
        logging.info("Svn diff: \n%s", Arcadia.diff(arcadia_dir))

        try:
            result = Arcadia.commit(
                path=arcadia_dir,
                message="Actualize group {}".format(self.Parameters.group_name),
                user=self.Parameters.ARCADIA_USER
            )
            logging.info("Commit result: %s", result)
            revision = re.findall(r'Committed revision (\d+)\.', result)
            if revision:
                self.Parameters.output = "https://a.yandex-team.ru/arc/commit/{}".format(revision[0])
        except SvnError as e:
            logging.error(e.message)
            review = re.findall(r'https://a\.yandex-team\.ru/review/(\d+)', e.message)
            if review:
                review_url = "https://a.yandex-team.ru/review/{}".format(review[0])
                ticket = self._create_issue(review_url)
                self.Parameters.output = "Create ticket {} for {}".format(ticket, review_url)

        logging.info("Commit/Review URL: %s\nAll done", self.Parameters.output)
