# -*- coding: utf-8


import logging
import sandbox.common.types.notification as ctn

from datetime import timedelta
from collections import defaultdict
from datetime import datetime as dt

from sandbox import sdk2
from sandbox.sandboxsdk import environments


u'''
История тут: MARKETSRETODO-237.

Таска выгребает по фильтру из стартрека список тикетов,
берет из них автора, и формирует для него запрос с просьбой заполнить форму
фидбека о службе эксплуатации Маркета.

В письме также будет перечень тикетов, которые разработчик создал.

Планируется, что эта таска будет запускаться каждые 30 дней.
'''


_ST_SECRET_ID = u'sec-01cyha56hmq89wv6rc8g6ae9dz'
_ST_SECRET_VERSION = u'ver-01e31t9vsa8ranrqqqe4m8rnxx'
_FEEDBACK_FORM_URL = u'https://forms.yandex.net/surveys/46373/'
_STARTREK_FILTER = u'Queue: CSADMIN ' +\
                   u'AND Resolution: Решен ' +\
                   u'AND Author: !group(value: "Group operation of a Market in Moscow" ) ' +\
                   u'AND Author: !group(value: "Group operating Market in Yekaterinburg" ) ' +\
                   u'AND Author: !"Alexey Simakov" AND Author: !"cs admin" ' +\
                   u'AND Author: !"Давид Бурназян" AND Status: "Closed" ' +\
                   u'AND "Resolved": > today() - 30d'
_MAIL_SUBJ = u'Запрос фидбека о работе службы эксплуатации'
_MAIL_BODY = u'''
Здравствуйте!

Вы получили это письмо, потому что админы, разработчики
и руководители из службы эксплуатации решили один или
несколько ваших тикетов в очереди CSADMIN в стартреке
за последние 30 дней.

Список решенных и закрытых тикетов:
{tickets}

Оцените, пожалуйста, качество выполненных работ, ответив
на несколько простых вопросов в анонимной форме:
{form_url}

Это поможет нам улучшить процессы и обращать внимание на самые
важные места в нашей работе.

По любым вопросам, связанным с данным письмом, не стесняйтесь
приходить к le087.

Большое спасибо!
'''


class AskDevelopersFeedbackTask(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('startrek_client', custom_parameters=['requests==2.18.4']),
        ]

    class Parameters(sdk2.Task.Parameters):

        secret = sdk2.parameters.YavSecret(
            'Secrets csadmin for access to StarTrek API.',
            default=_ST_SECRET_ID
        )

        startrek_filter = sdk2.parameters.String(
            'Filter for startrek tickets',
            default_value=_STARTREK_FILTER,
            required=True
        )
        form_url = sdk2.parameters.Url(
            'Url to feedback form',
            default_value=_FEEDBACK_FORM_URL,
            required=True
        )

    def _send_emails(self, authors):
        mail_subj = _MAIL_SUBJ
        mail_body_tmpl = _MAIL_BODY
        for author, keys in authors.items():
            mail_body = mail_body_tmpl.format(
                tickets='\n'.join(['https://st.yandex-team.ru/%s' % k for k in keys]),
                form_url=self.Parameters.form_url
            )
            logging.info('E-mail for %s', author)
            logging.info(mail_body)
            try:
                self.server.notification(
                    subject=mail_subj,
                    body=mail_body,
                    recipients=[author],
                    transport=ctn.Transport.EMAIL,
                    type=ctn.Type.TEXT,
                    charset=ctn.Charset.UTF,
                    task_id=self.id,
                    view=ctn.View.DEFAULT
                )
            except Exception:
                logging.exception('Exception: Could not send an email to %s' % author)

    def _get_month_ago_date(self):
        u'''Получем дату, что была 30 дней назад'''
        month_ago = dt.now() - timedelta(days=30)
        return str(month_ago.date())

    def _get_st_tickets(self, oauth_token, from_date):
        u'''Возвращает список тикетов из стартрека.
        Нужные тикеты выбираются согласно фильтру из параметров таски.
        '''
        import startrek_client
        st_client = startrek_client.Startrek(
            token=oauth_token,
            useragent='sandbox-task ASK_DEVELOPERS_FEEDBACK'
        )
        issues = st_client.issues.find(self.Parameters.startrek_filter)
        return issues

    def _get_authors(self, tickets):
        u'''Возвращает словарь с авторами и списком тикетов,
        которые для него решили.

        Возвращаемый словарь имеет вид:
        {
           user_one: [CSADMIN-1, CSADMIN-2],
           user_two: [CSADMIN-3, CSADMIN-4],
        }
        '''
        authors = defaultdict(list)
        for t in tickets:
            author = t.createdBy
            if 'robot-' not in author.login:
                authors[author.login].append(t.key)
        return authors

    def on_execute(self):
        logging.info('Hello, World!')
        oauth_token = self.Parameters.secret.data()['STARTREK_TOKEN']
        from_date = self._get_month_ago_date()
        tickets = self._get_st_tickets(oauth_token, from_date)
        logging.info(tickets)
        authors = self._get_authors(tickets)
        logging.info(authors)
        self._send_emails(authors)
        logging.info('Goodbye, America!')
