# coding: utf-8


import errno
import logging
import os
import shutil
import tarfile


import sandbox.projects.common.file_utils as fu
import sandbox.projects.common.utils as utils
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common.nanny import nanny
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.common.arcadia.sdk import mount_arc_path

from sandbox import sdk2

_ETCD_KEY_TEMPLATES = [
    '/{}/yandex/market-datasources/datasources.properties/',
    '/{}/yandex/market-datasources/picrobot.cfg/',
    '/ir/{}/',
    '/checkout/{}/',
    '/mboc/{}/',
    '/mbo/{}',
    '/marketcms/{}',
    '/redmarket/{}',
    '/lgw/{}/',
    '/lms/{}/',
    '/mdb/{}/',
    '/contentlab/{}/',
    '/pers/{}/',
    '/wrapinfor/{}/',
    '/marketpromo/{}/',
    '/deepmind/{}/'
]


_DEPLOY_DATASOURCES_PATH = 'app/datasources'


class MarketDatasourcesTesting(sdk2.Resource):
    """Contains Market Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesStable(sdk2.Resource):
    """Contains Market Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesPrestable(sdk2.Resource):
    """Contains Market Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesTestingDeploy(sdk2.Resource):
    """Contains Market Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesStableDeploy(sdk2.Resource):
    """Contains Market Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesPrestableDeploy(sdk2.Resource):
    """Contains Market Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesCheckoutLoadTestingCheckouter(sdk2.Resource):
    """Contains Market Checkout Load Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesCheckoutLoadTestingPushApi(sdk2.Resource):
    """Contains Market Checkout Load Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesCheckoutLoadTestingShopadminStub(sdk2.Resource):
    """Contains Market Checkout Load Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesCheckoutLoadTestingCarter(sdk2.Resource):
    """Contains Market Checkout Load Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesCheckoutLoadTestingNotifier(sdk2.Resource):
    """Contains Market Checkout Load Datasources
    https://github.yandex-team.ru/cs-admin
    """
    releasable = True
    backup_task = True


class MarketDatasourcesMultitesting(sdk2.Resource):
    """Contains Market Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = False


class MarketDatasourcesCheckoutTestingPlaneshift(sdk2.Resource):
    """Contains Market Checkout Planeshift Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


class MarketDatasourcesCheckoutProductionPlaneshift(sdk2.Resource):
    """Contains Market Checkout Planeshift Datasources
    https://github.yandex-team.ru/cs-admin/datasources-ng
    """
    releasable = True
    backup_task = True


def ensure_directory_exists(directory):
    try:
        os.makedirs(directory)
    except OSError as err:
        if err.errno != errno.EEXIST:
            raise


def ensure_directory_does_not_exist(directory):
    try:
        shutil.rmtree(directory)
    except OSError as err:
        if err.errno != errno.ENOENT:
            raise


def write_confd_config_file(
        confd_config_file_path,
        etcd_keys,
        etcd_prefix,
        file_mode,
        output_file_path,
        template_name
):
    contents = """
            [template]
            keys = {}
            prefix = "{}"
            mode = "{}"
            src = "{}"
            dest = "{}"
            """.format(etcd_keys, etcd_prefix, file_mode, template_name, output_file_path)
    logging.info("Writing confd template resource to %s: %s", confd_config_file_path, contents)
    fu.write_file(confd_config_file_path, contents)


class BuildMarketDatasources(nanny.ReleaseToNannyTask2, sdk2.Task):
    """A task that packs Market Datasources into tarball"""

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.String('Arcadia url', default='arcadia:/arc/trunk/arcadia', required=True)
        vault_key = sdk2.parameters.String(
            "Vault key, containing etcd password",
            default="market_etcd_datasources_read_all",
            required=True
        )
        etcd_username = sdk2.parameters.String("etcd username", default="datasources_read_all", required=True)
        github_ssh_key_vault_name = sdk2.parameters.String(
            'Имя Vault содержащее ssh ключ для доступа к github',
            default="robot-market-dist-ssh",
            required=True
        )
        github_ssh_key_vault_owner = sdk2.parameters.String(
            'Владелец Vault содержащее ssh ключ для доступа к github',
            default="MARKETSRE",
            required=True
        )
        multitesting_etcd_prefix = sdk2.parameters.String(
            'Префикс мультитестинга (только для генерации мультитестовых датасорсов)',
            required=False,
            description='Если указано, то будет сгенерирован только ресурс MARKET_DATASOURCES_MULTITESTING вместо'
                        ' обычных трёх. В etcd будет использоваться указанный префикс.'
                        ' Пример значения: /multitesting/front-desktop--my_awesome_mt/.',
        )
        multitesting_parent_environment_name = sdk2.parameters.String(
            'Окружение, на котором будут основаны мультитестовые датасорсы'
            ' (только для генерации мультитестовых датасорсов)',
            required=False,
            description='В префиксе /multitesting/{ID мультитестинга}/ будет все три маркетных окружения (testing,'
                        ' prestable, production). Здесь нужно указать какое из них нужно использовать для генерации'
                        ' мультитестовых датасорсов.',
        )
        ignore_not_existing_files = sdk2.parameters.Bool(
            'Игнорировать несуществующие файлы (или файлы, которые не получилось сгенерировать)',
            required=True,
            default=False
        )

    def _git_clone_confd_templates(self):
        logging.info('Cloning confd templates from github')
        ensure_directory_does_not_exist('salt-ng')
        git = Git('git@github.yandex-team.ru:cs-admin/salt-ng.git')
        with sdk2.ssh.Key(self, self.Parameters.github_ssh_key_vault_owner,
                                self.Parameters.github_ssh_key_vault_name):
            git.clone('salt-ng', 'master')

        git_directory = os.path.join('salt-ng', '.git')
        logging.info('Removing %s', git_directory)

        shutil.rmtree(git_directory, ignore_errors=True)

        shutil.copytree(os.path.join(os.getcwd(), 'salt-ng/formulaes/confd/configs/templates/'),
                        os.path.join(os.getcwd(), 'confd/templates/'))

    def _arc_clone_confd_templates(self):
        with mount_arc_path(self.Parameters.arcadia_url) as mp:
            shutil.copytree(os.path.join(mp, 'market/sre/conf/salt/data/states/formulaes/confd/configs/templates/'),
                            os.path.join(os.getcwd(), 'confd/templates/'))

    def _write_confd_config(self):
        confd_toml_content = """
                          backend = "{}"
                          basic_auth = {}
                          nodes = ["{}"]
                          client_cakeys = "{}"
                          """.format('etcdv3',
                                     'true',
                                     'etcd.vs.market.yandex.net:3379',
                                     '/usr/share/yandex-internal-root-ca/YandexInternalRootCA.crt')

        logging.info('Writing cond config: {}'.format(confd_toml_content))
        with open('confd/confd.toml', 'w+') as f:
            f.write(confd_toml_content)

    def _get_confd_executable_path(self):
        resource_id = apihelpers.get_last_resource_with_attribute(resource_types.CONFD_BIN).id
        logging.info('CONFD_BIN resource id: %s', resource_id)
        resource_path = utils.sync_resource(resource_id)
        logging.info('CONFD_BIN resource path: %s', resource_path)

        return resource_path

    def _prepare_confd_directory(self, etcd_prefix, template_name, output_file_path, etcd_keys, file_mode):
        confd_directory = self.confd_directory

        template_config_directory = os.path.join(confd_directory, 'conf.d')
        template_config_path = os.path.join(template_config_directory, 'config.toml')

        ensure_directory_exists(template_config_directory)
        write_confd_config_file(
            confd_config_file_path=template_config_path,
            etcd_keys=etcd_keys,
            etcd_prefix=etcd_prefix,
            file_mode=file_mode,
            output_file_path=os.path.join(os.getcwd(), output_file_path),
            template_name=template_name
        )

        return confd_directory

    def _run_confd(self, environment_name):
        run_env = os.environ.copy()
        run_env["ETCD_PASSWORD"] = sdk2.Vault.data(self.Parameters.vault_key)
        run_env["YANDEX_ENVIRONMENT"] = environment_name
        logging.info('YANDEX_ENVIRONMENT=%s', environment_name)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("run_confd")) as pl:
            sp.Popen(
                "{} -log-level info -onetime -config-file confd/confd.toml -confdir confd/"
                " -sync-only -username {} -password $ETCD_PASSWORD".format(
                    self._get_confd_executable_path(), self.Parameters.etcd_username
                ),
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stdout,
                env=run_env
            ).communicate()

    def _create_datasources_file(self, environment_name, etcd_prefix, template_name, output_file_path, etcd_keys=None,
                                 file_mode='0644'):
        etcd_keys = etcd_keys or ['/']

        logging.info('Creating file %s from template %s and etcd prefix %s',
                     output_file_path, template_name, etcd_prefix)

        self._prepare_confd_directory(
            etcd_prefix=etcd_prefix,
            template_name=template_name,
            output_file_path=output_file_path,
            etcd_keys=etcd_keys,
            file_mode=file_mode,
        )

        ensure_directory_exists(os.path.dirname(output_file_path))
        self._run_confd(environment_name=environment_name)

        return output_file_path

    def _create_datasources_resource_from_files(self, resource_type, tarball_file_name, input_file_names):
        data = sdk2.ResourceData(resource_type(self, 'Market Datasources', tarball_file_name))

        logging.info('Creating tarball %s', tarball_file_name)
        with tarfile.open(name=tarball_file_name, mode='w:gz') as tar:
            for input_file_name in input_file_names:
                logging.info('Adding %s to tarball', input_file_name)
                if (self.Parameters.ignore_not_existing_files == False or os.path.exists(input_file_name)):
                    tar.add(input_file_name)
                else:
                    logging.warn('%s does not exists', input_file_name)

        logging.info('Marking resource %s as ready', resource_type.type)
        data.ready()

    def _create_datasources_resource(self, resource_type, environment_name, multitesting_etcd_prefix='/', dir_if_deploy=None):
        """
        :param environment_name: Название окружения, используется в названиях файлов и ключах etcd
        """

        common_etcd_prefix = os.path.join(multitesting_etcd_prefix, 'datasources')

        # для деплоя мы кладем датасорсы в отдельный каталог, обычно это:
        # app/datasources, а также делаем отдельный архив, иначе он конфликтует
        # по имени с архивом датасорсов не для деплоя
        if dir_if_deploy:
            datasources_path = dir_if_deploy
            archive_name = 'datasources-{}-deploy.tar.gz'.format(environment_name)
        else:
            datasources_path = environment_name
            archive_name = 'datasources-{}.tar.gz'.format(environment_name)

        logging.info(
            'Creating resource %s for environment %s from etcd prefix %s',
            resource_type.__name__,
            environment_name,
            common_etcd_prefix
        )

        input_file_names = [
            self._create_datasources_file(
                environment_name=environment_name,
                etcd_prefix=os.path.join(common_etcd_prefix, '{}/oracle/tnsnames.ora'.format(environment_name)),
                template_name='oracle__tnsnames.ora.tmpl',
                output_file_path=os.path.join(datasources_path, 'properties.d/tnsnames.ora'),
            ),
            self._create_datasources_file(
                environment_name=environment_name,
                etcd_prefix=common_etcd_prefix,
                etcd_keys=[t.format(environment_name) for t in _ETCD_KEY_TEMPLATES],
                template_name='datasources.properties.tmpl',
                # В обычных датасорсах проперти попадают в файл properties.d/50-datasources.properties
                # В мультитестовых в properties.d/{multitesting_parent_environment_name}/50-datasources.properties.
                # Причины в https://st.yandex-team.ru/MARKETINFRA-4302.
                output_file_path=os.path.join(
                    datasources_path,
                    'properties.d',
                    '' if multitesting_etcd_prefix == '/' else environment_name,
                    '50-datasources.properties'
                ),
            ),
            self._create_datasources_file(
                environment_name=environment_name,
                etcd_prefix=os.path.join(common_etcd_prefix, 'pricelabs'),
                etcd_keys=["/{}/".format(environment_name)],
                template_name='pricelabs/parameters-yml.tmpl',
                output_file_path=os.path.join(datasources_path, 'properties.d/parameters.yml'),
            )
        ]

        for mi_type in ['stratocaster', 'gibson', 'planeshift.stratocaster', 'planeshift.gibson', 'yellow.stratocaster', 'yellow.gibson', 'blue.stratocaster', 'blue.gibson', 'turbo.gibson', 'turbo.stratocaster', 'fresh.stratocaster', 'fresh.gibson']:
            input_file_names.append(
                self._create_datasources_file(
                    environment_name=environment_name,
                    etcd_prefix=common_etcd_prefix,
                    etcd_keys=['{}/yandex/market-datasources/indexer/{}.conf'.format(
                        environment_name, mi_type)],
                    template_name='subdir-config-file.tmpl',
                    output_file_path=os.path.join(datasources_path, 'properties.d/indexer/{}.conf'.format(mi_type)),
                )
            )
            input_file_names.append(
                self._create_datasources_file(
                    environment_name=environment_name,
                    etcd_prefix=common_etcd_prefix,
                    etcd_keys=['{}/yandex/market-datasources/zookeeper/{}.conf'.format(
                        environment_name, mi_type)],
                    template_name='subdir-config-file.tmpl',
                    output_file_path=os.path.join(datasources_path, 'properties.d/zookeeper/{}.conf'.format(mi_type)),
                )
            )

        self._create_datasources_resource_from_files(
            resource_type=resource_type,
            tarball_file_name=archive_name,
            input_file_names=input_file_names
        )

    def _create_load_datasources_resource(self, resource_type, environment_name, component, file_name='properties.d/51_datasources_load.properties'):
        if environment_name != "testing":
            # Нельзя собирать не в тестинге
            return

        common_etcd_prefix = '/datasources'
        logging.info('Creating resource %s for component %s from etcd_prefix %s', resource_type, component, common_etcd_prefix)
        input_file_names = [
            self._create_datasources_file(
                environment_name=environment_name,
                etcd_prefix='{}/checkoutload/{}'.format(common_etcd_prefix, component),
                template_name='java-properties.tmpl',
                output_file_path=os.path.join(environment_name, file_name)
            )
        ]
        self._create_datasources_resource_from_files(
            resource_type=resource_type,
            tarball_file_name='datasources-{}-load-{}.tar.gz'.format(environment_name, component),
            input_file_names=input_file_names
        )

    def _create_planeshift_datasources_resource(self, resource_type, environment_name, file_name='properties.d/52_datasources_planeshift.properties'):
        common_etcd_prefix = '/datasources'
        planeshift_prefix = 'yandex/market-datasources/planeshift.stratocaster/datasources.properties'
        etcd_prefix = '{}/{}/{}'.format(common_etcd_prefix, environment_name, planeshift_prefix)

        logging.info('Creating resource %s from etcd_prefix %s', resource_type, etcd_prefix)
        input_file_names = [
            self._create_datasources_file(
                environment_name=environment_name,
                etcd_prefix=etcd_prefix,
                template_name='java-properties.tmpl',
                output_file_path=os.path.join(environment_name, file_name)
            )
        ]
        self._create_datasources_resource_from_files(
            resource_type=resource_type,
            tarball_file_name='datasources-{}-planeshift.tar.gz'.format(environment_name),
            input_file_names=input_file_names
        )

    def on_execute(self):
        self._arc_clone_confd_templates()
        self.confd_directory = os.path.join(os.getcwd(), 'confd')
        self._write_confd_config()

        if self.Parameters.multitesting_etcd_prefix and self.Parameters.multitesting_parent_environment_name:
            self._create_datasources_resource(
                resource_type=MarketDatasourcesMultitesting,
                environment_name=self.Parameters.multitesting_parent_environment_name,
                multitesting_etcd_prefix=self.Parameters.multitesting_etcd_prefix,
            )
        else:
            self._create_datasources_resource(
                resource_type=MarketDatasourcesTesting,
                environment_name='testing',
            )

            self._create_datasources_resource(
                resource_type=MarketDatasourcesPrestable,
                environment_name='prestable',
            )

            self._create_datasources_resource(
                resource_type=MarketDatasourcesStable,
                environment_name='production',
            )

            # Собираем аналогичные датасорсы для деплоя
            self._create_datasources_resource(
                resource_type=MarketDatasourcesTestingDeploy,
                environment_name='testing',
                dir_if_deploy=_DEPLOY_DATASOURCES_PATH,
            )

            self._create_datasources_resource(
                resource_type=MarketDatasourcesPrestableDeploy,
                environment_name='prestable',
                dir_if_deploy=_DEPLOY_DATASOURCES_PATH,
            )

            self._create_datasources_resource(
                resource_type=MarketDatasourcesStableDeploy,
                environment_name='production',
                dir_if_deploy=_DEPLOY_DATASOURCES_PATH,
            )

            # собираем остальные датасорсы
            self._create_load_datasources_resource(
                resource_type=MarketDatasourcesCheckoutLoadTestingCheckouter,
                environment_name='testing',
                component='checkouter',
                file_name='properties.d/52_datasources_load.properties'
            )

            self._create_load_datasources_resource(
                resource_type=MarketDatasourcesCheckoutLoadTestingPushApi,
                environment_name='testing',
                component='push_api'
            )

            self._create_load_datasources_resource(
                resource_type=MarketDatasourcesCheckoutLoadTestingShopadminStub,
                environment_name='testing',
                component='shopadmin_stub'
            )

            self._create_load_datasources_resource(
                resource_type=MarketDatasourcesCheckoutLoadTestingCarter,
                environment_name='testing',
                component='carter',
                file_name='properties.d/53_datasources_load.properties'
            )

            self._create_load_datasources_resource(
                resource_type=MarketDatasourcesCheckoutLoadTestingNotifier,
                environment_name='testing',
                component='notifier',
                file_name='properties.d/53_datasources_load.properties'
            )

            self._create_planeshift_datasources_resource(
                resource_type=MarketDatasourcesCheckoutTestingPlaneshift,
                environment_name='testing',
            )

            self._create_planeshift_datasources_resource(
                resource_type=MarketDatasourcesCheckoutProductionPlaneshift,
                environment_name='production',
            )
