# -*- coding: utf-8 -*-

import logging
import os
from glob import glob

import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.task as sdk_task
from sandbox.projects import resource_types
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
from sandbox.projects.common import utils
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.channel import channel
from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.vcs.svn import Arcadia


class GitUrl(sdk_parameters.SandboxGitUrlParameter):
    name = 'git_url'
    description = 'Git-repo with sources for BUILD_DOCKER_IMAGE_V6 resources'
    required = False
    group = 'Resource prepare'


class GitBranch(sdk_parameters.SandboxStringParameter):
    name = 'git_branch'
    description = 'Git branch:'
    default_value = 'master'
    required = False
    group = 'Resource prepare'


class ArcadiaUrl(sdk_parameters.SandboxArcadiaUrlParameter):
    name = 'arcadia_url'
    description = 'Svn url for arcadia'
    required = False
    group = 'Resource prepare'


class DockerPath(sdk_parameters.SandboxStringParameter):
    name = 'docker_path'
    description = 'Path in repo with dirs with Dockerfile and another sources'
    default_value = 'docker'
    required = True
    group = 'Resource prepare'


class DockerImagesList(sdk_parameters.ListRepeater, sdk_parameters.SandboxStringParameter):
    name = 'docker_images'
    description = ('List of dirs in "docker_path" directory in git '
                   '(accept globes, globes will be interpreted as parallel building)')
    default_value = ['*']
    required = True
    group = 'Resource prepare'


class DockerRepoPrefix(sdk_parameters.SandboxStringParameter):
    name = 'docker_repo_prefix'
    description = 'Prefix before image name (without leading and trailing "/")'
    required = True
    group = 'Container build and publish'


class DockerTags(sdk_parameters.ListRepeater, sdk_parameters.SandboxStringParameter):
    name = 'docker_tags'
    description = ('List of tags which will be pushed' +
                   '(we try to update it with append version without release from debian/changelog if exists)')
    default_value = ['latest']
    required = False
    group = 'Container build and publish'


class DockerPublisher(sdk_parameters.SandboxStringParameter):
    name = 'docker_publisher'
    description = 'Yandex login to use with docker login'
    required = True
    group = 'Container build and publish'


class DockerVaultName(sdk_parameters.SandboxStringParameter):
    name = 'docker_vault'
    description = 'Vault item with oauth token for registry.yandex.net (vault item name)'
    required = True
    group = 'Container build and publish'


class DockerVaultOwner(sdk_parameters.SandboxStringParameter):
    name = 'docker_vault_owner'
    description = 'Vault item owner'
    required = True
    group = 'Container build and publish'

class DockerBuildArgs(sdk_parameters.ListRepeater, sdk_parameters.SandboxStringParameter):
    name = 'docker_build_args'
    description = 'Docker --build-arg option'
    required = False
    group = 'Container build and publish'


class BuildMarketDockerImages(sdk_task.SandboxTask):
    type = 'BUILD_MARKET_DOCKER_IMAGES'

    TARGET_RESOURCES = [
        ('', '', resource_types.MARKET_DOCKER_RESOURCE)
    ]

    input_parameters = [
        GitUrl,
        GitBranch,
        ArcadiaUrl,
        DockerPath,
        DockerImagesList,
        DockerRepoPrefix,
        DockerTags,
        DockerPublisher,
        DockerVaultName,
        DockerVaultOwner,
        DockerBuildArgs
    ]

    repo = 'repo'

    def clone_repo(self):
        if 'repo_cloned' not in self.ctx:
            if self.ctx[GitUrl.name]:
                logging.info("Cloning from github")
                git = Git(self.ctx.get(GitUrl.name, ''))
                git.clone(self.repo, self.ctx.get(GitBranch.name, ''))
            else:
                logging.info("Cloning from arcadia")
                arcadia_url = self.ctx.get(ArcadiaUrl.name)
                sources_dir = Arcadia.checkout(arcadia_url, path=str(self.path(self.repo)))

            self.ctx['repo_cloned'] = True
            return self.ctx['repo_cloned']
        else:
            return self.ctx['repo_cloned']


    def build_resources(self, ctx_res):
        _docker_path = utils.get_or_default(self.ctx, DockerPath)
        if ctx_res not in self.ctx:
            self.ctx[ctx_res] = {}
            self.clone_repo()
            try:
                with open(os.path.join(self.repo, 'debian', 'changelog')) as chlog:
                    full_version = chlog.readline().split()[1][1:-1]
                    short_version = full_version.split('-')[0]
                if short_version not in self.ctx[DockerTags.name]:
                    self.ctx[DockerTags.name].append(short_version)
            except:
                logging.warning("Failed to get short version from debian/changelog")
            for img_glob in utils.get_or_default(self.ctx, DockerImagesList):
                # ['name', 'glo*b']
                self.ctx[ctx_res][img_glob] = {}
                images = glob(os.path.join(self.repo, _docker_path, img_glob))
                if len(images) == 0:
                    raise errors.SandboxTaskFailureError(
                        'There are not {} in "{}" dir in {} repo'.format(
                            img_glob,
                            _docker_path,
                            self.ctx.get(GitUrl.name, '')
                        )
                    )
                for path in images:
                    _type = resource_types.MARKET_DOCKER_RESOURCE
                    logging.info("Building resource from {}".format(path))
                    resource = self.create_resource(
                        description="dir '{}', type '{}'".format(path, _type),
                        resource_type=_type,
                        resource_path=path
                    )
                    name = path.split('/')[-1]
                    self.ctx[ctx_res][img_glob][name] = resource.id
                    self.mark_resource_ready(resource.id)

    def build_images(self, ctx_img, ctx_res):
        if not self.ctx.get(ctx_img):
            self.ctx[ctx_img] = {}
        for img_glob in utils.get_or_default(self.ctx, DockerImagesList):
            if img_glob not in self.ctx[ctx_img]:
                self.ctx[ctx_img][img_glob] = {}
                # Parallel subtasks for one img_glob
                for name in self.ctx[ctx_res][img_glob]:
                    if self.ctx[DockerTags.name]:
                        _tags = [
                            "{}:{}".format(os.path.join(self.ctx[DockerRepoPrefix.name], name), tag)
                            for tag in self.ctx[DockerTags.name]
                        ]
                    else:
                        _tags = [os.path.join(self.ctx[DockerRepoPrefix.name])]
                    build_image_task = self.create_subtask(
                        task_type=BuildDockerImageV6.type,
                        description='Launch building of {} image from {} task'.format(name, self.id),
                        inherit_notifications=True,
                        input_parameters={
                            BuildDockerImageV6.PackagedResource.name: self.ctx[ctx_res][img_glob][name],
                            BuildDockerImageV6.RegistryTags.name: _tags,
                            BuildDockerImageV6.RegistryLogin.name: self.ctx[DockerPublisher.name],
                            BuildDockerImageV6.VaultItemName.name: self.ctx[DockerVaultName.name],
                            BuildDockerImageV6.VaultItemOwner.name: self.ctx[DockerVaultOwner.name],
                            BuildDockerImageV6.DockerBuildArgs.name: self.ctx[DockerBuildArgs.name]
                        }
                    )
                    self.ctx[ctx_img][img_glob][name] = build_image_task.id

    def on_execute(self):
        ctx_res = 'resources_builded'
        ctx_img = 'images_builded'
        # 1. Clone repo and making resources
        self.build_resources(ctx_res=ctx_res)
        self.build_images(ctx_img=ctx_img, ctx_res=ctx_res)

        # waiting for all tasks and check for a success
        utils.check_subtasks_fails()


__Task__ = BuildMarketDockerImages
