import logging
import os
import shutil
import tarfile

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.resource_types import marketsre
from sandbox.projects.market.resources import market_front
from sandbox.sandboxsdk.ssh import Key
from sandbox.sdk2.vcs.git import Git


class MarketStaticPages(sdk2.Resource):
    """A resource contains market static pages
    https://github.yandex-team.ru/market/market-static-pages
    """
    releasable = True
    releasers = marketsre + market_front
    description_ = 'Static pages'
    path_ = 'static.tar.gz'


resource_types = {
    MarketStaticPages.__name__: MarketStaticPages,
}


class BuildMarketFrontStatic(nanny.ReleaseToNannyTask2, sdk2.Task):
    """A task that packs static files into tarball"""

    class Parameters(sdk2.Task.Parameters):
        url = sdk2.parameters.String('Git repository url', required=True)
        branch = sdk2.parameters.String('Git repository branch', default='master', required=False)

        with sdk2.parameters.String('Resource type', required=True, multiline=True) as resource_type:
            for k in resource_types.iterkeys():
                resource_type.values[k] = k

        startrek_ticket = sdk2.parameters.List('Startrek ticket', required=True)

        secret_ssh = sdk2.parameters.String('Vault secret with SSH key', default='robot-market-packer-ssh',
                                            required=True)
        secret_owner = sdk2.parameters.String('Vault owner of the secret', default='MARKETSRE', required=True)

    def _clone_repository(self, checkout_path):
        logging.info('Getting ssh public key and cloning repository {}:{}'.format(
            self.Parameters.url, self.Parameters.branch,
        ))
        with Key(self, self.Parameters.secret_owner, self.Parameters.secret_ssh):
            git = Git(self.Parameters.url)
            git.clone(checkout_path, self.Parameters.branch)

        logging.info('Removing .git')
        shutil.rmtree(os.path.join(checkout_path, '.git'), ignore_errors=True)

    def on_execute(self):
        checkout_path = os.path.basename(self.Parameters.url)
        self._clone_repository(checkout_path)

        logging.info('Creating resource')
        resource_type = resource_types[self.Parameters.resource_type]
        data = sdk2.ResourceData(resource_type(self, resource_type.description_, resource_type.path_))

        logging.info('Creating tarball {}'.format(resource_type.path_))
        with tarfile.open(name=resource_type.path_, mode='w:gz') as tar:
            tar.add(checkout_path)

        logging.info('Marking resource as ready')
        data.ready()
