import os
import shutil
import platform
import multiprocessing as mp

import sandbox.common.types.misc as ctm
import sandbox.sandboxsdk.process as sdk_process
# import sandboxsdk.svn

from sandbox.sandboxsdk import ssh
from sandbox.projects import resource_types
from sandbox.projects.common import environments
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter  # , Container
# from sandbox.sdk2.vcs.git import Git


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git tag'
    default_value = 'yandex-stable-1.8'
    required = True


# class ContainerType(Container):
#    name = 'conttype'
#    description = 'Container Type'


class BuildMarketNginxBinary(nanny.ReleaseToNannyTask, SandboxTask):
    """
    Build Nginx statically from sources.
    """
    privileged = True
    dns = ctm.DnsType.DNS64
    type = "MARKET_NGINX_BINARY"

    REPO_NAME = "nginx"
    GIT_URL_TEMPLATE = "https://github.yandex-team.ru/InfraComponents/nginx.git"
    GIT_BRANCH = 'yandex-stable-1.8'
    VAULT_OWNER = 'MARKETSRE'
    SSH_KEY_VAULT_NAME = 'csadmin-ssh'
    CHECKOUT_PATH_TEMPLATE = "nginx_repo"

    PCRE_RESOURCE_ID = 109325093
    LIBOPENSSL_RESOURCE_ID = 202636069

    environment = [environments.SandboxLibOpensslEnvironment()]
    input_parameters = [GitTagParameter]  # , ContainerType]

    def _make_resource(self, path, version, resource_type=resource_types.OTHER_RESOURCE, arch='linux', attributes={},
                       **kwargs):
        """
        Simply create sandbox resource.
        """
        resource_attributes = {'version': version, 'platform': platform.platform()}
        if attributes:
            resource_attributes.update(attributes)
        self.create_resource(resource_path=path,
                             resource_type=resource_type,
                             arch=arch,
                             attributes=resource_attributes,
                             **kwargs)

    def _get_version(self):
        with open(os.path.join(self._checkout_path(), 'src', 'core', 'nginx.h')) as header:
            for s in header.read().split('\n'):
                if len(s.split()) > 1 and s.split()[1] == "NGINX_VERSION":
                    return s.split()[-1].strip('"')

    def _get_sandbox(self, name, resource_id, srcpath):
        with self.current_action("Downloading {0} from sandbox".format(name)):
            resource_path = self.sync_resource(resource_id)
        with self.current_action("Extracting sources to '{0}'".format(srcpath)):
            if os.path.exists(srcpath):
                shutil.rmtree(srcpath)
            os.mkdir(srcpath)
            sdk_process.run_process(
                ["tar", "xf", resource_path, "--strip-components=1", "--directory", srcpath],
                log_prefix="extract_{0}".format(name)
            )

    # def _get_sandbox_without_unzip(self, name, resource_id):
    #     with self.current_action("Downloading {0} from sandbox".format(name)):
    #         self.sync_resource(resource_id)

    def _get_git(self, name, url, srcpath):
        with self.current_action("Cloning git repo for {0}".format(name)):
            if os.path.exists(srcpath):
                shutil.rmtree(srcpath)
            os.mkdir(srcpath)
            sdk_process.run_process(
                ["git", "clone", url, srcpath],
                log_prefix="clone_{0}".format(name)
            )

    # def _get_arcadia(self, name, url, srcpath):
    #     with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_VAULT_NAME):
    #         with self.current_action("Cloning svn repo for {0}".format(name)):
    #             if os.path.exists(srcpath):
    #                 shutil.rmtree(srcpath)
    #             os.mkdir(srcpath)
    #             sdk_process.run_process(
    #                 ["svn", "co", url, srcpath],
    #                 log_prefix="clone_{0}".format(name)
    #             )

    def _checkout_path(self):
        return self.abs_path(self.CHECKOUT_PATH_TEMPLATE)

    def _add_dist_repo(self):
        logfeller_repos = '''
deb http://yandex-trusty.dist.yandex.ru/yandex-trusty unstable/all/
deb http://yandex-trusty.dist.yandex.ru/yandex-trusty unstable/$(ARCH)/
deb http://yandex-trusty.dist.yandex.ru/yandex-trusty testing/all/
deb http://yandex-trusty.dist.yandex.ru/yandex-trusty testing/$(ARCH)/'''
        with open('/etc/apt/sources.list.d/yandex_unstable.list', mode='w') as repo_list:
            repo_list.write(logfeller_repos)
        pin_repos = '''
Package: *
Pin: release a=unstable
Pin-Priority: 999'''
        with open('/etc/apt/preferences.d/9999999-unstable', mode='w') as pin_list:
            pin_list.write(pin_repos)
        with self.current_action("apt-get update"):
            sdk_process.run_process(
                ["sudo", "apt-get", "update"],
                log_prefix='apt_update')
        with self.current_action("apt-cache yandex-libvcdcom-dev"):
            sdk_process.run_process(
                ["sudo", "apt-cache", "policy",
                 "yandex-libvcdcom-dev",
                 "yandex-libvcddec-dev",
                 "yandex-libvcdenc-dev"],
                log_prefix='apt_update')

    def _prepare(self):
        with self.current_action("prepare environment"):
            sdk_process.run_process(
                ["sudo", "apt-get", "update"],
                log_prefix='install_environment')
            sdk_process.run_process(
                ["sudo", "apt-get", "--assume-yes", "--force-yes", "install", "libboost-all-dev",
                 "libpng++-dev", "pkg-config", "libmagic-dev", "libjson-c-dev", "libgd-dev",
                 "libgeoip-dev", "make", "makedev", "cmake", "cmake-data", "gcc", "gcc-4.8",
                 "libticket-parser=0.1.31", "libticket-parser-dev=0.1.31", "yandex-libvcdcom-dev=0.8.4ubuntu1",
                 "yandex-libvcddec-dev=0.8.4ubuntu1", "yandex-libvcdenc-dev=0.8.4ubuntu1"],
                log_prefix='install_environment')

    def _checkout(self):
        """
        Clone git repository and checkout to given branch/tag.
        """
        with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_VAULT_NAME):
            with self.current_action("Cloning git repo for {0} in {1}".format(self.GIT_BRANCH, self._checkout_path())):
                if os.path.exists(self._checkout_path()):
                    shutil.rmtree(self._checkout_path())
                os.mkdir(self._checkout_path())
                sdk_process.run_process(['git',
                                                'clone',
                                                '--branch={}'.format(self.GIT_BRANCH),
                                                self.GIT_URL_TEMPLATE,
                                                self._checkout_path()],
                                               work_dir=self.abs_path(""),
                                               log_prefix='git_nginx')
                sdk_process.run_process(['git',
                                                'submodule',
                                                'update',
                                                '--init'],
                                               work_dir=self._checkout_path(),
                                               log_prefix='git_nginx_submodule')

    def on_execute(self):
        self._add_dist_repo()
        self._prepare()
        self._checkout()

        taskdir_path = self.abs_path("")
        modulesdir_path = os.path.join(self._checkout_path(),
                                       "src",
                                       "http",
                                       "modules")

        # ticketparser_path = self.abs_path(os.path.join(modulesdir_path,
        #                                               "ticket_parser"))
        # self._get_arcadia("ticket_parser", "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/library/ticket_parser",
        #                  ticketparser_path)

        lua_path = self.abs_path("lua")
        self._get_git("lua", "https://git.qe-infra.yandex-team.ru/scm/juggler/lua.git", lua_path)

        pcre_path = self.abs_path("pcre")
        self._get_sandbox("pcre", self.PCRE_RESOURCE_ID, pcre_path)

        openssl_path = self.abs_path("openssl")
        self._get_sandbox("pcre", self.LIBOPENSSL_RESOURCE_ID, openssl_path)

        with self.current_action("Building lua"):
            env = os.environ.copy()
            env["MYCFLAGS"] = "-ULUA_USE_DLOPEN"
            sdk_process.run_process(["make",
                                            "all",
                                            "-j",
                                            str(mp.cpu_count()),
                                            "-C",
                                            os.path.join(lua_path, "src"),
                                            "MYCFLAGS=-ULUA_USE_DLOPEN"],
                                           work_dir=lua_path,
                                           environment=env,
                                           log_prefix="build_lua")

        with self.current_action("Configuring nginx"):
            env = os.environ.copy()
            env["LUA_INC"] = os.path.join(lua_path, 'src')
            env["LUA_LIB"] = os.path.join(lua_path, 'src')
            sdk_process.run_process(
                ["auto/configure",
                 "--prefix=.",
                 "--with-cc-opt=-O2",
                 "--with-cc-opt=-DNGX_LUA_ABORT_AT_PANIC",
                 "--with-cc-opt='-I {0} -I {1}'".format(taskdir_path,
                                                        modulesdir_path),
                 "--with-ld-opt='-L{0} -L{1} -L{2} -Wl,-Bstatic -llua \
                                -lcontrib-newssl-openssl \
                                -Wl,-Bdynamic'".format(os.path.join(lua_path, "src"),
                                                       taskdir_path,
                                                       modulesdir_path),
                 "--with-debug",
                 "--with-http_addition_module",
                 "--with-http_flv_module",
                 "--with-http_dav_module",
                 "--with-http_gzip_static_module",
                 "--with-http_gunzip_module",
                 "--with-http_realip_module",
                 "--with-http_stub_status_module",
                 "--with-http_ssl_module",
                 "--with-http_spdy_module",
                 "--with-http_sub_module",
                 "--with-http_xslt_module",
                 "--with-http_auth_request_module",
                 "--with-ipv6",
                 "--with-yandex-newssl",
                 "--with-yandex-newssl-dir={}".format(openssl_path),
                 "--with-sha1={}".format(os.path.join(openssl_path, "include", "openssl")),
                 "--with-md5={}".format(os.path.join(openssl_path, "include", "openssl")),
                 "--with-mail",
                 "--with-mail_ssl_module",
                 "--with-http_perl_module",
                 "--with-http_geoip_module",
                 "--with-http_image_filter_module",
                 "--add-module=debian/modules/nginx-echo",
                 "--add-module=debian/modules/nginx-headers-more",
                 "--add-module=debian/modules/nginx-lua",
                 "--add-module=debian/modules/nginx-upstream-fair",
                 "--add-module=debian/modules/nginx-flv-filter",
                 "--add-module=debian/modules/nginx-ip-tos-filter",
                 "--add-module=debian/modules/nginx-addtag-exe",
                 "--add-module=debian/modules/nginx-speedtest",
                 "--add-module=debian/modules/nginx-request-id",
                 "--add-module=debian/modules/nginx-auth-sign",
                 "--add-module=debian/modules/nginx-yandex-antirobot-module",
                 "--add-module=debian/modules/nginx-pinba",
                 "--add-module=debian/modules/nginx-rtmp-module",
                 "--add-module=debian/modules/nginx-limit-upstream",
                 "--add-module=debian/modules/nginx-json-log",
                 "--add-module=debian/modules/nginx-eblob",
                 "--add-module=debian/modules/nginx-favicon",
                 "--add-module=debian/modules/nginx-development-kit",
                 "--add-module=debian/modules/nginx-tvm",
                 # "--add-module=debian/modules/nginx-sdch-module",
                 "--with-pcre={}".format(pcre_path), ],
                work_dir=self._checkout_path(), environment=env, log_prefix="configure_nginx"
            )

        sdk_process.run_process(['ls',
                                        '-laR'],
                                       work_dir=taskdir_path,
                                       log_prefix='ls_laR')

        with self.current_action("Building nginx"):
            sdk_process.run_process(
                ["make", "-j", str(mp.cpu_count()), "-C", self._checkout_path()],
                log_prefix="make_nginx", work_dir=self._checkout_path()
            )

        version = self._get_version()
        self._make_resource(
            path=os.path.join(self._checkout_path(), "objs", "nginx"),
            resource_type=resource_types.MARKET_NGINX_BINARY,
            description='nginx binary {0}, {1}'.format(self.ctx[GitTagParameter.name], version),
            version=version)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        services = ()
        self._prepare_nanny_snapshot(
            service_names=services,
            services_to_prepare=(),
            resource_type=resource_types.MARKET_NGINX_BINARY
        )


__Task__ = BuildMarketNginxBinary
