# -*- coding: utf-8 -*-

import os
import logging
import requests

import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.ssh import Key
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.gnupg import GpgKey
from sandbox.projects.common.debpkg import DebRelease


class UrlRepoString(parameters.SandboxStringParameter):
    name = 'urlrepostring'
    description = 'Ссылки на пакеты и репозиторий куда заливать в формате url:repo'
    multiline = True


class DebFullName(parameters.SandboxStringParameter):
    name = 'deb_full_name'
    description = 'DEBFULLNAME имя от кого будет инкриментирован changelog'
    group = 'Main Parameters'
    default_value = 'robot-market-dist'


class GpgUidEmail(parameters.SandboxStringParameter):
    name = 'gpg_uid_email'
    description = 'UID по которому можно найти ключ в брелке(gpg --list-keys)'
    group = 'Main Parameters'
    default_value = 'robot-market-dist@yandex-team.ru'


class VaultSshItemLogin(parameters.SandboxStringParameter):
    name = 'vault_ssh_item_login'
    description = 'SSH логин под которым пакет заливается в dist'
    group = 'Main Parameters'
    default_value = 'robot-market-dist'


class VaultGpgItemOwner(parameters.SandboxStringParameter):
    name = 'vault_gpg_item_owner'
    description = 'Владелец/команда GPG ключа из Sandbox Vault'
    group = 'Vault Parameters'
    default_value = 'MARKETSRE'


class VaultGpgPubItemName(parameters.SandboxStringParameter):
    name = 'vault_gpg_pub_item_name'
    description = 'Название/id открытой части GPG ключа'
    group = 'Vault Parameters'
    default_value = 'robot-market-dist-gpg-public'


class VaultGpgPrivItemName(parameters.SandboxStringParameter):
    name = 'vault_gpg_priv_item_name'
    description = 'Название/id закрытой части GPG ключа'
    group = 'Vault Parameters'
    default_value = 'robot-market-dist-gpg-private'


class VaultSshItemOwner(parameters.SandboxStringParameter):
    name = 'vault_ssh_item_owner'
    description = 'Владелец/команда SSH ключа из Sandbox Vault для заливки в dist'
    group = 'Vault Parameters'
    default_value = 'MARKETSRE'


class VaultSshItemName(parameters.SandboxStringParameter):
    name = 'vault_ssh_item_name'
    description = 'Название/id SSH ключа из Sandbox Vault для заливки в dist'
    group = 'Vault Parameters'
    default_value = 'robot-market-dist-ssh'


class VersionPostfix(parameters.SandboxStringParameter):
    name = "version_postfix"
    default_value = 'yandex0'
    description = 'Постфикс добавляется в конец версии пакета. \
                    Если версия 1.0, то вместо с постфиксом 1.0+' + default_value
    required = True


class EnableAutorestart(parameters.SandboxBoolParameter):
    name = 'autorestart'
    description = 'Убрать из pre/post скриптов остановку/запуск сервиса'
    default_value = False


class BuildMarketPackageRebuild(task.SandboxTask):
    """ Таск, пересобирающий и переподписывающий внешние пакеты и заливающий их на dist. """

    type = "BUILD_MARKET_PACKAGE_REBUILD"
    privileged = True
    dns = ctm.DnsType.DNS64

    input_parameters = [
        UrlRepoString,
        VersionPostfix,
        EnableAutorestart,
        DebFullName,
        GpgUidEmail,
        VaultSshItemLogin,
        VaultGpgItemOwner,
        VaultGpgPubItemName,
        VaultGpgPrivItemName,
        VaultSshItemOwner,
        VaultSshItemName
    ]

    def _get_packages(self):
        # создаём .dupload.conf для всех нужных нам реп
        # выкачиваем все нужные нам пакеты
        os.environ['DEBFULLNAME'] = self.ctx.get(DebFullName.name)
        os.environ['EMAIL'] = self.ctx.get(GpgUidEmail.name)
        login = self.ctx.get(VaultSshItemLogin.name)
        DUPLOAD_CONF = {}

        for item in self.ctx[UrlRepoString.name].strip().split():
            url, repo = item.rsplit(':', 1)
            tries = 3
            DUPLOAD_CONF[repo] = {
                'fqdn': repo + '.dupload.dist.yandex.ru',
                'method': 'scpb',
                'incoming': '/repo/' + repo + '/mini-dinstall/incoming/',
                'dinstall_runs': 0,
                'login': login,
                'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
            }
            filename = url.split('/')[-1]
            while tries > 0:
                try:
                    r = requests.get(url)
                    if r.status_code == 200:
                        with open(filename, "wb") as deb:
                            deb.write(r.content)
                        self.set_info("url %s successful download" % (url))
                        break
                    else:
                        eh.check_failed("failed to get url %s: request code %s" % (url, r.status_code))
                except requests.exceptions.HTTPError as err:
                    self.set_info("failed to get url %s: %s" % (url, err))
                    tries -= 1
        self.ctx['DUPLOAD_CONF'] = DUPLOAD_CONF

    def on_execute(self):
        self._get_packages()
        vp = self.ctx.get(VersionPostfix.name)
        ea = int(self.ctx.get(EnableAutorestart.name))
        with DebRelease(self.ctx['DUPLOAD_CONF']):
            with GpgKey(self, self.ctx.get(VaultGpgItemOwner.name),
                        self.ctx.get(VaultGpgPrivItemName.name),
                        self.ctx.get(VaultGpgPubItemName.name)):
                with Key(self, self.ctx.get(VaultSshItemOwner.name), self.ctx.get(VaultSshItemName.name)):
                    for item in self.ctx[UrlRepoString.name].strip().split():
                        url, repo = item.rsplit(':', 1)
                        filename = url.split('/')[-1]
                        cmd = "rebuild-deb.sh -f %s -p %s -a %s -r %s" % (filename, vp, ea, repo)
                        try:
                            run_process([os.path.dirname(__file__) + "/" + cmd],
                                        shell=True, wait=True, log_prefix="rebuild-deb.sh")
                        except Exception as e:
                            logging.exception(e)
                            self.set_info("something bad in %s" % cmd)
                            eh.check_failed("something bad in %s" % cmd)
